/* kemo-M104.c --
 Driver for the kemo M104 output card
 Copyright 2002
 Lionel, trollhunter Bouchpan-Lerust-Juery <trollhunter@linuxfr.org>
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License as
 published by the Free Software Foundation; either version 2 of the
 License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.
 
 You should have received a copy of the GNU General Public License 
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 Boston, MA 02111-1307, USA.

*/

#include <unistd.h>
#include <sys/io.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include "kemo-M104.h"
#include "my_errors.h"

static int
port2int (const char *str)
{
  int result;

  result = -1;
  if (!str)
    return (result);
  if (!strcmp ("lp0", str))
    result = 0x3bc;
  if (!strcmp ("lp1", str))
    result = 0x378;
  if (!strcmp ("lp2", str))
    result = 0x278;
  return (result);
}



int
M104_check (const char *port)
{
  int result, tmp, num_port;

  result = -1;
  if ((num_port = port2int (port)) < 0)
    {
      my_error ("M104_check: invalid port");
      return (result);
    }
  if (ioperm (num_port, 3, 1))
    {
      my_error ("M104_check: can't access port");
      return (result);
    }
  tmp = inb (num_port);
  usleep (DELAY);
  if (ioperm (num_port, 3, 0))
    {
      my_error ("M104_check: can't access port");
      return (result);
    }
  result = tmp;
  return (result);
}

int
M104_close (const char *port, ...)
{
  int result, num_port, num_relay, current_state, tmp;
  va_list data;

  result = -1;
  tmp = 0;
  if ((num_port = port2int (port)) < 0)
    {
      my_error ("M104_close: invalid port");
      return (result);
    }
  va_start (data, port);
  while (1)
    {
      num_relay = va_arg (data, int);
      if (num_relay == M104_ALL)
	{
	  tmp |= num_relay;
	  break;
	}
      if (num_relay < 1 || num_relay > 4)
	break;
      if (num_relay == 4)
	tmp |= 8;
      else if (num_relay == 3)
	tmp |= 4;
      else
	tmp |= num_relay;
    }
  va_end (data);
  current_state = M104_check (port);
  if (ioperm (num_port, 3, 1))
    {
      perror ("M104_close");
      my_error ("M104_close: can't access port");
      return (result);
    }
  tmp = ~tmp & current_state;
  outb (tmp, num_port);
  usleep (DELAY);
  if (ioperm (num_port, 3, 0))
    {
      perror ("M104_close");
      my_error ("M104_close: can't access port");
      return (result);
    }
  result = 0;
  return (result);
}


int
M104_open (const char *port, ...)
{
  int result, num_port, num_relay, current_state, tmp;
  va_list data;

  result = -1;
  tmp = 0;
  if ((num_port = port2int (port)) < 0)
    {
      my_error ("M104_open: invalid port");
      return (result);
    }
  va_start (data, port);
  while (1)
    {
      num_relay = va_arg (data, int);
      if (num_relay == M104_ALL)
	{
	  tmp |= num_relay;
	  break;
	}
      if (num_relay < 1 || num_relay > 4)
	break;
      if (num_relay == 4)
	tmp |= 8;
      else if (num_relay == 3)
	tmp |= 4;
      else
	tmp |= num_relay;
    }
  va_end (data);
  current_state = M104_check (port);
  if (ioperm (num_port, 3, 1))
    {
      perror ("M104_open");
      my_error ("M104_open: can't access port");
      return (result);
    }
  tmp = tmp | current_state;
  outb (tmp, num_port);
  usleep (DELAY);
  if (ioperm (num_port, 3, 0))
    {
      perror ("M104_open");
      my_error ("M104_open: can't access port");
      return (result);
    }
  result = 0;
  return (result);
}


int
M104_reset (const char *port)
{
  int result, num_port;

  result = -1;
  if ((num_port = port2int (port)) < 0)
    {
      my_error ("M104_reset: invalid port");
      return (result);
    }
  if (ioperm (num_port, 3, 1))
    {
      perror ("M104_reset");
      my_error ("M104_reset: can't access port");
      return (result);
    }
  outb (M104_RESET, num_port);
  usleep (DELAY);
  if (ioperm (num_port, 3, 0))
    {
      perror ("M104_reset");
      my_error ("M104_reset: can't access port");
      return (result);
    }
  result = 0;
  return (result);
}
