#!/bin/bash

# make_base.sh, created by Yann Guidon <whygee@f-cpu.org> april, 10th 2002
# this script somewhat automates the operations that install a LFS-like
# linux system. This particular script corresponds to the chapters
# up to chapter 5 of the Linux From Scratch book (hi to Gerard Beekmans)

# version : initial (0.0)
#   based on LFS3.3 (more or less), plus personal additions/mods
# version : Fri Apr 12 07:37:43 CEST 2002
#   starting to go beyond bash and binutils ;-)
# version : Mon Apr 15 00:57:38 CEST 2002
#   the Linux source tree creates some new problems,
#   gotta make a more sophisticated version for detecting files
#   and directory, and also moving some things away from tmp
#   to make it completely erasable. At this level, maybe the use
#   of PERL would be better ?

# colors and message functions :
source message_functions.sh || exit -1

# check if custom settings are done :
if [ "$YGLFS" == "" ]
then
  export YGLFS=/LFS
# please update this line if it changes (could also become
# a command line parameter)
#   Mon Apr 15 00:57:38 CEST 2002 => should become /usr/yglfs !
  message_warning "setting \$YGLFS to $YGLFS"
else
  message_warning "\$YGLFS is already set to $YGLFS"
  if [[ $YGLFS != /* ]]
  then
    message_ko "\$YGLFS is not an absolute path"
  fi
fi

# then check that our directories exist :
for MISSING_DIR in \
   $YGLFS/src/pkgs \
   $YGLFS/src/uname-trick \
   $YGLFS/src/lfs-commands/chapter4 \
   $YGLFS/src/lfs-commands/chapter5
do
  if [[ ! -d $MISSING_DIR ]]
  then
    message_ko "SORRY : $MISSING_DIR is missing"
  fi
done

message_warning "setting our working directory to $YGLFS"

# the log directory :
export YGLFS_LOGDIR=$YGLFS/log
mkdir -v -p $YGLFS_LOGDIR

# now that the log directory is setup,
# we can send the log messages there.
export YGLFS_LOG=$YGLFS_LOGDIR/.yglfs_log.1


export LFS=$YGLFS/root_base
# where we will install the new root directory
#if [! -d $LFS ]
#then
  mkdir -v -p $LFS
#else
#  message_warning ""
#fi

if [ -e $LFS/usr/yglfs/.step1 ]
then
  message_ok "STEP 1 already performed in $LFS : nothing to do !"
  exit
fi

export LFS_PKGS=$YGLFS/src/pkgs
# the original compressed source packages are there.

export LFS_CMD=$YGLFS/src/lfs-commands
# where the LFS command scripts are located

# create a temp directory for decompacting the sources (if not already done)
mkdir -v -p $YGLFS/tmp

# creates a time stamp
message_ok "# YGLFS : make_base.sh started " $(date)

####### Some environmental checks ###########

if [ ! -e /usr/lib/libcurses.a ]
then
  message_ko "/usr/lib/libcurses.a is not installed,\
please read the LFS-BOOK (chapter 5 about bash)"
fi

# detection of the host configuration : detect the glibc version
if [ ! -e /usr/bin/glibcbug ]
then
  message_ko "can't detect installed glibc version through /usr/bin/glibcbug"
else
  GLIBC_VERSION=$(grep "^VERSION=" /usr/bin/glibcbug | \
      head -n1 | sed -e 's/"//' | sed -e 's/"//' )
  if [ "$GLIBC_VERSION" != "" ]
  then
    export $GLIBC_VERSION &&
    export GLIBC_MAJOR=$( echo $VERSION | cut -f 1 -d "." ) &&
    export GLIBC_MINOR=$( echo $VERSION | cut -f 2 -d "." ) &&
    export GLIBC_MICRO=$( echo $VERSION | cut -f 3 -d "." ) &&
    echo "GLIBC version detected = $GLIBC_MAJOR.$GLIBC_MINOR.$GLIBC_MICRO" ||
        message_ko "big trouble with glibc version :-/"
  else
    message_ko "can't detect installed glibc version through /usr/bin/glibcbug"
  fi
fi

# uname stuff : it's used to cross-compile to a different target
# by inserting a kernel module that mimicks a different setting.
if [ -x $YGLFS/src/uname-trick/commands.auto ]
then
  cd $YGLFS/src/uname-trick/
  source commands.auto install
  cd $YGLFS
fi


####### And now, let's go. #########

# As written in the LFS-BOOK
umask 022
export LC_ALL=POSIX

if [ ! -e $YGLFS_LOGDIR/chapter4.ok ]
  then
    source $LFS_CMD/chapter4/creating-dirs ||
      message_ko "can\'t execute $LFS_CMD/chapter4/creating-dirs"

  # oh, don't forget to create the yglfs subtree :-)
  # it's how the system can replicate itself !
  (
    cd $LFS/usr &&
    mkdir -p yglfs &&
    cd yglfs &&
    cp $YGLFS/*.sh . &&    # just files, omitting directories
    mkdir -p root_base root_chroot &&
    if [ "$YGLFS_TEST" != "" ]
    then
      message_warning " \$YGLFS_TEST is set : skipping copy, linking instead"
      if [ -e $LFS/usr/yglfs/src ]
      then
        message_warning "$LFS/usr/yglfs/src already exists : skipping link"
      else
        ln -i -v -s $YGLFS/src .
        # this speeds up the tests, particularly
        #  it avoids the long copy of the source packages.
      fi
    else
      message_warning "Copying the source packages (please wait 30s)"
      echo "sorry, no time to create a nice progress bar"
      echo "so open a VT and do this for me, thanks :-P"
      cp -r $YGLFS/src .   # all the files, that's it ...
    fi
  ) || message_ko "argh, can\'t implant the YGLFS tree in $LFS/usr"
  message_ok " ============ chapter 4 completed ============"
  touch $YGLFS_LOGDIR/chapter4.ok
else
  message_warning "skipping chapter4 (already done)"
fi

# This is now the row of extract, compile, install...

export LFS_CMD=$YGLFS/src/lfs-commands/chapter5
# now moving to the new commands directory

BASENAME_LIST="bash binutils bzip2 diffutils fileutils \
 gawk gcc grep gzip linux make patch sed sh-utils tar \
 texinfo textutils"

source $YGLFS/install_functions.sh || message_ko ' install_functions.sh failed '

if [ -x $YGLFS/src/uname-trick/commands.auto ]
then
  cd $YGLFS/src/uname-trick/
  source commands.auto remove
  cd $YGLFS
fi

if [ ! -e $LFS/etc/passwd ]
then
  message_warning "creating $LFS/etc/passwd"
  echo "root::0:0:root:/root:/bin/bash
q::0:0:root:/root:/sbin/halt" > $LFS/etc/passwd
fi

if [ ! -e $LFS/etc/group ]
then
  message_warning "creating $LFS/etc/group"
  cat > $LFS/etc/group << "EOF"
root:x:0:
bin:x:1:
sys:x:2:
kmem:x:3:
tty:x:4:
tape:x:5:
daemon:x:6:
floppy:x:7:
disk:x:8:
lp:x:9:
dialout:x:10:
audio:x:11:
EOF
fi

if [[ "GLIBC_MAJOR" == "2" && "GLIBC_MINOR" == "0" ]]
then
  message_warning " ======== !!! copying /lib/libc.so.6 !!! ======== "
  cp -av /lib/libnss* $LFS/lib
fi

touch $LFS/usr/yglfs/.step1

message_ok "Congratulations ! Step 1 is finished."

exit
