-- prtree.vhdl -- ReduceTree w/ Optional Input Register
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: prtree.vhdl,v 1.1 2000/12/07 04:08:04 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity Piped_ReduceTree is
	generic (
		WIDTH : natural := 128;
		ILINES : natural := 4;
		OLINES : natural := 3;
		PIPELINED : boolean := false
	);
	port (
		-- input
		A : in std_ulogic_vector(WIDTH*ILINES-1 downto 0);
		-- clock/reset inputs
		Clk : in std_ulogic := '0';
		Rst : in std_ulogic := '0';
	--
		-- output
		Y : out std_ulogic_vector(WIDTH*OLINES-1 downto 0)
	);
begin
	-- only some combinations are valid:
	assert (4 * OLINES = 3 * ILINES)	-- 4:3 reducer
		or (3 * OLINES = 2 * ILINES)	-- 3:2 reducer
		or (4 * OLINES = 2 * ILINES)	-- 4:2 reducer
		report "can only reduce 4->3, 3->2 or 4->2"
		severity failure;
end Piped_ReduceTree;

architecture Struct_1 of Piped_ReduceTree is
	component ReduceTree
		generic (
			WIDTH : natural := 128;
			ILINES : natural := 4;
			OLINES : natural := 3
		);
		port (
			A : in std_ulogic_vector(WIDTH*ILINES-1 downto 0);
			Y : out std_ulogic_vector(WIDTH*OLINES-1 downto 0)
		);
	end component;
	component PipeReg
		generic (
			WIDTH : natural := 64;
			DOREG : boolean := false
		);
		port (
			Clk, Rst : in std_ulogic;
			D : in std_ulogic_vector(WIDTH-1 downto 0);
			Q : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	constant w : natural := WIDTH * ILINES;

	signal r_A : std_ulogic_vector(w-1 downto 0);
begin
	reg_1 : PipeReg
		generic map (
			WIDTH => w, DOREG => PIPELINED
		)
		port map (
			D => A, Q => r_A, Clk => Clk, Rst => Rst
		);

	tree : ReduceTree
		generic map (
			WIDTH => WIDTH, ILINES => ILINES, OLINES => OLINES
		)
		port map (
			A => r_A, Y => Y
		);
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
