-- pciainc.vhdl -- CIA_Inc w/ Optional Input Register
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: pciainc.vhdl,v 1.1 2000/12/07 03:50:16 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity Piped_CIA_Inc is
	generic (
		WIDTH : natural := 64;
		STEP : natural := 4;
		PIPELINED : boolean := false
	);
	port (
		-- sum/carry inputs
		Yi : in std_ulogic_vector(WIDTH-1 downto 0);
		Ci : in std_ulogic_vector(WIDTH-1 downto 0);
		-- higher-level increment inputs
		Ii : in std_ulogic_vector((WIDTH-1)/STEP downto 0);
		-- carry select inputs
		Cs : in std_ulogic_vector((WIDTH-1)/STEP downto 0);
		-- clock/reset inputs
		Clk : in std_ulogic := '0';
		Rst : in std_ulogic := '0';
	--
		-- sum/carry outputs
		Yo : out std_ulogic_vector(WIDTH-1 downto 0);
		Co : out std_ulogic_vector(WIDTH-1 downto 0)
	);
end Piped_CIA_Inc;

architecture Struct_1 of Piped_CIA_Inc is
	component CIA_Inc
		generic (WIDTH : natural := 64; STEP : natural := 4);
		port (
			Yi : in std_ulogic_vector(WIDTH-1 downto 0);
			Ci : in std_ulogic_vector(WIDTH-1 downto 0);
			Ii : in std_ulogic_vector((WIDTH-1)/STEP downto 0);
			Cs : in std_ulogic_vector((WIDTH-1)/STEP downto 0);
		--
			Yo : out std_ulogic_vector(WIDTH-1 downto 0);
			Co : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;
	component PipeReg
		generic (WIDTH : natural := 64; DOREG : boolean := false);
		port (
			Clk, Rst : in std_ulogic;
			D : in std_ulogic_vector(WIDTH-1 downto 0);
			Q : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	constant w : natural := (WIDTH-1)/STEP+1;

	signal r_Yi, r_Ci : std_ulogic_vector(WIDTH-1 downto 0);
	signal r_Ii, r_Cs : std_ulogic_vector(w-1 downto 0);
begin
	reg_1 : PipeReg
		generic map (WIDTH => WIDTH, DOREG => PIPELINED)
		port map (D => Yi, Q => r_Yi, Clk => Clk, Rst => Rst);
	reg_2 : PipeReg
		generic map (WIDTH => WIDTH, DOREG => PIPELINED)
		port map (D => Ci, Q => r_Ci, Clk => Clk, Rst => Rst);
	reg_3 : PipeReg
		generic map (WIDTH => w, DOREG => PIPELINED)
		port map (D => Ii, Q => r_Ii, Clk => Clk, Rst => Rst);
	reg_4 : PipeReg
		generic map (WIDTH => w, DOREG => PIPELINED)
		port map (D => Cs, Q => r_Cs, Clk => Clk, Rst => Rst);

	core : CIA_Inc
		generic map (WIDTH => WIDTH, STEP => STEP)
		port map (
			Yi => r_Yi, Ci => r_Ci, Ii => r_Ii, Cs => r_Cs,
			Yo => Yo, Co => Co
		);
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
