-- im64test3.vhdl - F-CPU Integer Multiplication Unit Testbench #3
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: im64test3.vhdl,v 1.1 2000/12/12 15:21:06 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

entity IMul64_test3 is
end IMul64_test3;

architecture Arch_1 of IMul64_test3 is
	component IMul64
		generic (
			PIPE_AFTER : natural := 0;	-- gates per stage; 0 means no pipelining
			PIPE_DELAY : natural := 0	-- additional delay before 1st stage
		);
		port (
			A : in std_ulogic_vector(63 downto 0);
			B : in std_ulogic_vector(63 downto 0);
			X : in std_ulogic_vector(63 downto 0) := (others => '0');
			SignedMode : in std_ulogic := '0';
			MacLo : in std_ulogic := '0';
			MacHi : in std_ulogic := '0';
			MacAlt : in std_ulogic := '0';
			U08, U16, U32 : in std_ulogic := '1';
			Clk : in std_ulogic := '0';
			Rst : in std_ulogic := '0';
		--
			Y08l : out std_ulogic_vector(63 downto 0);
			Y08h : out std_ulogic_vector(63 downto 0);
			Y16l : out std_ulogic_vector(63 downto 0);
			Y16h : out std_ulogic_vector(63 downto 0);
			Y32l : out std_ulogic_vector(63 downto 0);
			Y32h : out std_ulogic_vector(63 downto 0);
			Y64l : out std_ulogic_vector(63 downto 0);
			Y64h : out std_ulogic_vector(63 downto 0)
		);
	end component;

	constant WIDTH : natural := 64;

	signal A, B : std_ulogic_vector(WIDTH-1 downto 0);
	signal X : std_ulogic_vector(WIDTH-1 downto 0);
	signal M : std_ulogic_vector(6 downto 0);
	signal Y08l, Y08h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y16l, Y16h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y32l, Y32h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y64l, Y64h : std_ulogic_vector(WIDTH-1 downto 0);
	signal L, H : std_ulogic_vector(WIDTH-1 downto 0);
	signal Clk, Rst : std_ulogic;

	signal zero : std_ulogic := '0';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 x_1, x_2 : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("X := ")); write(lout, X); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("H := ")); write(lout, H); writeline(output, lout);
		write(lout, string'("L := ")); write(lout, L); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x_1);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, x_2);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;

	procedure pulse (signal x : out std_ulogic; t : time := 1 ns) is
	begin
		wait for t; x <= '1';
		wait for t; x <= '0';
	end pulse;
begin
	mut: IMul64
		generic map (PIPE_AFTER => 2, PIPE_DELAY => 0)
		port map (
			A => A, B => B, X => X,
			SignedMode => M(0),
			MacLo => M(1), MacHi => M(2), MacAlt => M(3),
			U08 => M(4), U16 => M(5), U32 => M(6),
			Clk => Clk, Rst => Rst,
			Y08l => Y08l, Y08h => Y08h,
			Y16l => Y16l, Y16h => Y16h,
			Y32l => Y32l, Y32h => Y32h,
			Y64l => Y64l, Y64h => Y64h
		);

	-- output selector
	L <= Y64l when M(6) = '1'
	else Y32l when M(5) = '1'
	else Y16l when M(4) = '1'
	else Y08l;
	H <= Y64h when M(6) = '1'
	else Y32h when M(5) = '1'
	else Y16h when M(4) = '1'
	else Y08h;

	test : process
		variable pv, rv : std_ulogic_vector(2*WIDTH-1 downto 0);
		variable gran : natural;
		variable lout : line;
	begin
		for level in 0 to 3 loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit pipelining ***"));
			writeline(output, lout);

			-- select mode
			M <= "0000001";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;

			-- reset registers
			-- output data should be '0'
			Clk <= '0';
			pv := (others => '0');
			pulse(Rst); rv := H & L; check_logic("Y", rv, pv);

			A <= (others => 'X');
			B <= (others => 'X');
			X <= (others => 'X');
			-- clock 'X' through the pipe
			-- output should stay at '0' until 'X' arrives at the end
			for i in 2 to 8+3*level loop
				pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			end loop;
			-- data reaches final stage
			pv := (others => 'X');
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			-- clock 0*0 and (-1)*(-1) through the pipe
			-- output should stay at 'X' until '0' arrives at the end
			M(0) <= '1';
			A <= (others => '0');
			B <= (others => '0');
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			A <= (others => 'X');
			B <= (others => 'X');
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			A <= (others => '1');
			B <= (others => '1');
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			A <= (others => 'X');
			B <= (others => 'X');
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			for i in 6 to 8+3*level loop
				pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			end loop;
			-- data reaches final stage
			pv := (others => '0');
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			pv := (others => 'X');
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			pv := (others => '0');
			for j in 0 to 64/gran-1 loop
				pv(gran*j) := '1';
			end loop;
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
			pv := (others => 'X');
			pulse(Clk); rv := H & L; check_logic("Y", rv, pv);
		end loop;

		writestr("*** Simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
