-- ciadds.vhdl -- Small (1..16-Bit) Carry-Increment Adder
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: ciadds.vhdl,v 1.3 2000/12/08 14:23:27 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity CIAddSmall is
	generic (
		WIDTH : natural range 1 to 16 := 8
	);
	port (
		-- summands
		A : in std_ulogic_vector(WIDTH-1 downto 0);
		B : in std_ulogic_vector(WIDTH-1 downto 0);
		-- output
		Y : out std_ulogic_vector(WIDTH-1 downto 0)
	);
end CIAddSmall;

architecture Struct_1 of CIAddSmall is
	component AND2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component AND3
		port (A, B, C : in std_ulogic; Y : out std_ulogic);
	end component;
	component AND4
		port (A, B, C, D : in std_ulogic; Y : out std_ulogic);
	end component;
	component OR2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component OR3
		port (A, B, C : in std_ulogic; Y : out std_ulogic);
	end component;
	component XOR2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component CIA_Row
		generic (WIDTH : natural := 64);
		port (
			Gi : in std_ulogic_vector(WIDTH-1 downto 0);
			Pi : in std_ulogic_vector(WIDTH-1 downto 0);
			Co : out std_ulogic_vector(WIDTH-1 downto 0);
			Io : out std_ulogic_vector(WIDTH-1 downto 0);
			Go : out std_ulogic_vector((WIDTH-1)/4 downto 0);
			Po : out std_ulogic_vector((WIDTH-1)/4 downto 0)
		);
	end component;

	signal G0, P0, C1, I1 : std_ulogic_vector(WIDTH-1 downto 0);
	signal G1, P1 : std_ulogic_vector((WIDTH-1)/4 downto 0);
	signal Y1 : std_ulogic_vector(WIDTH-1 downto 0);
begin
	-- a row of half adders
	-- d=1
	input : for i in 0 to WIDTH-1 generate
		sum   : XOR2 port map (A(i), B(i), P0(i));
		carry : AND2 port map (A(i), B(i), G0(i));
	end generate;

	-- carry-increment tree
	-- d=2 (I1/P1)
	-- d=3 (C1/G1)
	tree : CIA_Row
		generic map (WIDTH => WIDTH)
		port map (
			Gi => G0, Pi => P0, Co => C1, Io => I1, Go => G1, Po => P1
		);

	-- intermediate 4-bit results
	-- d=4
	res_4 : for i in 0 to WIDTH-1 generate
		y_4 : XOR2 port map (P0(i), C1(i), Y1(i));
	end generate;

	-- result
	-- d=6
	output : for i in 0 to WIDTH-1 generate
		-- d=4
		res_0 : if i/4 = 0 generate
			y_l : Y(i) <= Y1(i);
		end generate;
		-- d=5
		res_1 : if i/4 = 1 generate
			bl : block
				signal t1 : std_ulogic;
			begin
				-- d=4(3,2)
				tmp1 : AND2 port map (G1(0), I1(i), t1);
				-- d=5(4,4)
				y_h  : XOR2 port map (Y1(i), t1, Y(i));
			end block;
		end generate;
		-- d=6
		res_2 : if i/4 = 2 generate
			bl : block
				signal t1, t2, t3 : std_ulogic;
			begin
				-- d=4(3,2)
				tmp1 : AND2 port map (G1(1), I1(i), t1);
				-- d=4(2,3,2)
				tmp2 : AND3 port map (P1(1), G1(0), I1(i), t2);
				-- d=5(4,4)
				tmp3 : OR2  port map (t1, t2, t3);
				-- d=6(4,5)
				y_h  : XOR2 port map (Y1(i), t3, Y(i));
			end block;
		end generate;
		-- d=6
		res_3 : if i/4 = 3 generate
			bl : block
				signal t1, t2, t3, t4 : std_ulogic;
			begin
				-- d=4(3,2)
				tmp1 : AND2 port map (G1(2), I1(i), t1);
				-- d=4(2,3,2)
				tmp2 : AND3 port map (P1(2), G1(1), I1(i), t2);
				-- d=4(2,2,3,2)
				tmp3 : AND4 port map (P1(2), P1(1), G1(0), I1(i), t3);
				-- d=5(4,4,4)
				tmp4 : OR3  port map (t1, t2, t3, t4);
				-- d=6(4,5)
				y_h  : XOR2 port map (Y1(i), t4, Y(i));
			end block;
		end generate;
	end generate;
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
