-- ciacore.vhdl -- Carry-Increment Adder Core Element
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: ciacore.vhdl,v 1.2 2000/11/27 18:36:30 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity CIA_Core is
	generic (
		WIDTH : natural range 1 to 4 := 4
	);
	port (
		-- generate in
		Gi : in std_ulogic_vector(WIDTH-1 downto 0);
		-- propagate in
		Pi : in std_ulogic_vector(WIDTH-1 downto 0);
		-- carry out
		Co : out std_ulogic_vector(WIDTH-1 downto 0);
		-- increment out
		Io : out std_ulogic_vector(WIDTH-1 downto 0);
		-- generate out
		Go : out std_ulogic;
		-- propagate out
		Po : out std_ulogic
	);
end CIA_Core;

architecture Behave_1 of CIA_Core is
begin
	w_4 : if WIDTH = 4 generate
		process (Gi, Pi)
		begin
			Co(0) <= '0';
			Co(1) <= Gi(0);
			Co(2) <= Gi(1)
				  or (Pi(1) and Gi(0));
			Co(3) <= Gi(2)
				  or (Pi(2) and Gi(1))
				  or (Pi(2) and Pi(1) and Gi(0));
			Io(0) <= '1';
			Io(1) <= Pi(0);
			Io(2) <= Pi(1) and Pi(0);
			Io(3) <= Pi(2) and Pi(1) and Pi(0);
			Go <= Gi(3)
			   or (Pi(3) and Gi(2))
			   or (Pi(3) and Pi(2) and Gi(1))
			   or (Pi(3) and Pi(2) and Pi(1) and Gi(0));
			Po <= Pi(3) and Pi(2) and Pi(1) and Pi(0);
		end process;
	end generate;
	w_3 : if WIDTH = 3 generate
		process (Gi, Pi)
		begin
			Co(0) <= '0';
			Co(1) <= Gi(0);
			Co(2) <= Gi(1)
				  or (Pi(1) and Gi(0));
			Io(0) <= '1';
			Io(1) <= Pi(0);
			Io(2) <= Pi(1) and Pi(0);
			Go <= Gi(2)
			   or (Pi(2) and Gi(1))
			   or (Pi(2) and Pi(1) and Gi(0));
			Po <= Pi(2) and Pi(1) and Pi(0);
		end process;
	end generate;
	w_2 : if WIDTH = 2 generate
		process (Gi, Pi)
		begin
			Co(0) <= '0';
			Co(1) <= Gi(0);
			Io(0) <= '1';
			Io(1) <= Pi(0);
			Go <= Gi(1)
			   or (Pi(1) and Gi(0));
			Po <= Pi(1) and Pi(0);
		end process;
	end generate;
	w_1 : if WIDTH = 1 generate
		process (Gi, Pi)
		begin
			Co(0) <= '0';
			Io(0) <= '1';
			Go <= Gi(0);
			Po <= Pi(0);
		end process;
	end generate;
end Behave_1;

architecture Struct_1 of CIA_Core is
	component AND2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component AND3
		port (A, B, C : in std_ulogic; Y : out std_ulogic);
	end component;
	component AND4
		port (A, B, C, D : in std_ulogic; Y : out std_ulogic);
	end component;
	component OR2
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;
	component OR3
		port (A, B, C : in std_ulogic; Y : out std_ulogic);
	end component;
	component OR4
		port (A, B, C, D : in std_ulogic; Y : out std_ulogic);
	end component;
begin
	w_4 : if WIDTH = 4 generate
		bl : block
			signal t1, t2, t3, t4, t5, t6 : std_ulogic;
		begin
			co_0 : Co(0) <= '0';
			co_1 : Co(1) <= Gi(0);
			tmp1 : AND2 port map (Pi(1), Gi(0), t1);
			co_2 : OR2  port map (Gi(1), t1, Co(2));
			tmp2 : AND3 port map (Pi(2), Pi(1), Gi(0), t2);
			tmp3 : AND2 port map (Pi(2), Gi(1), t3);
			co_3 : OR3  port map (Gi(2), t3, t2, Co(3));
			io_0 : Io(0) <= '1';
			io_1 : Io(1) <= Pi(0);
			io_2 : AND2 port map (Pi(1), Pi(0), Io(2));
			io_3 : AND3 port map (Pi(2), Pi(1), Pi(0), Io(3));
			tmp4 : AND4 port map (Pi(3), Pi(2), Pi(1), Gi(0), t4);
			tmp5 : AND3 port map (Pi(3), Pi(2), Gi(1), t5);
			tmp6 : AND2 port map (Pi(3), Gi(2), t6);
			g_o  : OR4  port map (Gi(3), t6, t5, t4, Go);
			p_o  : AND4 port map (Pi(3), Pi(2), Pi(1), Pi(0), Po);
		end block;
	end generate;
	w_3 : if WIDTH = 3 generate
		bl : block
			signal t1, t2, t3 : std_ulogic;
		begin
			co_0 : Co(0) <= '0';
			co_1 : Co(1) <= Gi(0);
			tmp1 : AND2 port map (Pi(1), Gi(0), t1);
			co_2 : OR2  port map (Gi(1), t1, Co(2));
			io_0 : Io(0) <= '1';
			io_1 : Io(1) <= Pi(0);
			io_2 : AND2 port map (Pi(1), Pi(0), Io(2));
			tmp2 : AND3 port map (Pi(2), Pi(1), Gi(0), t2);
			tmp3 : AND2 port map (Pi(2), Gi(1), t3);
			g_o  : OR3  port map (Gi(2), t3, t2, Go);
			p_o  : AND3 port map (Pi(2), Pi(1), Pi(0), Po);
		end block;
	end generate;
	w_2 : if WIDTH = 2 generate
		bl : block
			signal t1 : std_ulogic;
		begin
			co_0 : Co(0) <= '0';
			co_1 : Co(1) <= Gi(0);
			io_0 : Io(0) <= '1';
			io_1 : Io(1) <= Pi(0);
			tmp1 : AND2 port map (Pi(1), Gi(0), t1);
			g_o  : OR2  port map (Gi(1), t1, Go);
			p_o  : AND2 port map (Pi(1), Pi(0), Po);
		end block;
	end generate;
	w_1 : if WIDTH = 1 generate
		co_0 : Co(0) <= '0';
		io_0 : Io(0) <= '1';
		g_o  : Go <= Gi(0);
		p_o  : Po <= Pi(0);
	end generate;
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
