-- iadd.vhdl -- F-CPU 64-bit Add/Subtract Unit
-- Copyright (C) 2000 Michael Riepe <michael@stud.uni-hannover.de>
--
-- rev. 0.2: 11/26/2000: YG added assertion on WIDTH
--
-- current revision : 11/26/2000
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id: iadd.vhdl,v 1.15 2000/11/10 03:20:20 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

entity IAdd is
	generic (
		WIDTH : natural := 64	-- do not change!
	);
	port (
		-- operand inputs
		A : in std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
		B : in std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
		-- subtract mode enable
		Sub : in std_ulogic := '0';
		-- saturate/floor mode enable
		Sat : in std_ulogic := '0';
		-- SIMD mode switches
		U08, U16, U32 : in std_ulogic := '0';
		-- 8-bit tap outputs
		Y8l : out std_ulogic_vector(WIDTH-1 downto 0);
		Y8h : out std_ulogic_vector(WIDTH-1 downto 0);
		-- regular outputs
		Yl : out std_ulogic_vector(WIDTH-1 downto 0);
		Yh : out std_ulogic_vector(WIDTH-1 downto 0)
	);
end IAdd;

-- Known limitations:
--
--	1: Not fully tested.
--
--	2: 8-bit SIMD mode works but adds/subf is slower (2 cycles)
--	than add/addc/sub/subb (1 cycle).  8-bit adds/subf seems to
--	be impossible with 6 gates, unless we use lookup tables.
--
--	3: There was no space (or rather, delay time) to include
--	the avg and diff instructions.	They will probably need an
--	additional output port.  Different avg/diff rounding modes
--	won't work either, but `truncate' should be sufficient anyway.
--
--	4: subb mode differs from F-CPU manual.  IMHO the manual
--	should be changed :)  See the rationale in the code below.
--
--	5: The pipeline register is missing.  I planned to tear off
--	the second stage after testing, make it a separate unit and
--	then glue the stages and the register together.

-- Operating Modes:
--
--	Sub = '0', Sat = '0': add operation
--	Sub = '0', Sat = '1': add operation with unsigned saturation (ceiling)
--	Sub = '1', Sat = '0': sub operation
--	Sub = '1', Sat = '1': sub operation with unsigned saturation (floor)
--
--	carry/borrow is always available on the second output port (Yh);
--	that means all operating modes from the manual are supported:
--	add, addc, adds, sub, subb and subf.  8-bit add/addc/sub/subb
--	has its own output ports in the first pipeline stage; 8-bit
--	adds/subf uses the output port of the second stage.

-- SIMD Modes:
--
--	U08 = '0', U16 = '0', U32 = '0': 8-bit mode
--	U08 = '1', U16 = '0', U32 = '0': 16-bit mode
--	U08 = '1', U16 = '1', U32 = '0': 32-bit mode
--	U08 = '1', U16 = '1', U32 = '1': 64-bit mode
--	(others combinations are invalid)
--
--	Note: I intend to use this encoding scheme everywhere; it seems
--	to be the most appropriate one.

-- Modus Operandi:
--
--	The IAdd unit is a multi-level carry look-ahead/increment adder
--	with SIMD capabilities.  Its first level calculates 4-bit
--	slices using carry look-ahead; the second and third level
--	are SIMD-enabled incrementers that provide wider results.
--	There is also a `tap' output that provides faster 8-bit
--	results for some operations.
--
--	Subtraction is implemented as `not ((not A) + B)' rather
--	than the usual `A + (not B) + 1' because that makes the
--	saturation modes easier -- and we don't need a carry input
--	either, which simplifies the SIMD stuff in the final stages.
--	Expressed as a simple equation, this unit calculates:
--
--		Yl := (((A xor Sub) + B) or (Sat and Carry)) xor Sub
--		Yh := 1 when Carry is set, 0 otherwise
--
--	where `Carry' is the appropriate carry output from the adder;
--	any other signals can be found in the entity declaration.

-- Implementation:
--
--	The whole unit consist of ordinary and/or gates with up
--	to 4 inputs, 2-input xor gates and inverters.  In timing
--	calculations, all gates are assumed to have a delay of 1.
--	Some basic elements may be optimized further if the target
--	supports arbitrary functions of 3 or 4 inputs, e.g. the half
--	adders can be combined with the input inverters.  With the
--	conservative assumptions above, the unit has a delay of
--	12, and it can be split in the middle (at d=6) to form two
--	pipeline stages.  *schwitz* :)

architecture Struct_1 of IAdd is
	component AND2 is -- assume d=1
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;

	component AND3 is -- assume d=1
		port (A, B, C : in std_ulogic; Y : out std_ulogic);
	end component;

	component AND4 is -- assume d=1
		port (A, B, C, D : in std_ulogic; Y : out std_ulogic);
	end component;

	component XOR2 is -- assume d=1
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;

	component OR2 is -- assume d=1
		port (A, B : in std_ulogic; Y : out std_ulogic);
	end component;

	component OR3 is -- assume d=1
		port (A, B, C : in std_ulogic; Y : out std_ulogic);
	end component;

	component OR4 is -- assume d=1
		port (A, B, C, D : in std_ulogic; Y : out std_ulogic);
	end component;

	component NOT1 is -- assume d=1
		port (A : in std_ulogic; Y : out std_ulogic);
	end component;

	-- signals used by both stages
	signal C2 : std_ulogic_vector(WIDTH/4-1 downto 0);
	signal G2 : std_ulogic_vector(WIDTH/16-1 downto 0);
	signal P2 : std_ulogic_vector(WIDTH/16-1 downto 0);
	signal Y04 : std_ulogic_vector(WIDTH-1 downto 0);
	signal C08 : std_ulogic_vector(WIDTH/8-1 downto 0);
	signal Inc04 : std_ulogic_vector(WIDTH-1 downto 0);
	signal Inc16 : std_ulogic_vector(WIDTH/4-1 downto 0);
begin
	assert WIDTH = 64
		report "width of IAdd must be 64"
		severity failure;

	stage_1 : block
		-- signals used by stage 1 exclusively
		signal G0 : std_ulogic_vector(WIDTH-1 downto 0);
		signal P0 : std_ulogic_vector(WIDTH-1 downto 0);
		signal C1 : std_ulogic_vector(WIDTH-1 downto 0);
		signal G1 : std_ulogic_vector(WIDTH/4-1 downto 0);
		signal P1 : std_ulogic_vector(WIDTH/4-1 downto 0);
		signal un08 : std_ulogic;
	begin
		-- mode switch inverted
		-- d=1
		u08n : NOT1 port map (U08, un08);

		-- input stage
		-- (half adders with A input inverted by Sub)
		-- d=2
		input : for i in 0 to WIDTH-1 generate
			-- Behaviour:
			--	P0(i) <= (A(i) xor Sub) xor B(i);
			--	G0(i) <= (A(i) xor Sub) and B(i);
			bl : block
				signal t : std_ulogic;
			begin
				inv_a : XOR2 port map (A(i), Sub, t);
				sum   : XOR2 port map (t, B(i), P0(i));
				carry : AND2 port map (t, B(i), G0(i));
			end block;
		end generate;

		-- 4-bit increment vector
		-- d=3
		inc_04 : for i in 0 to WIDTH/4-1 generate
			-- Behaviour:
			--	Inc04(4*i+3) <= P0(4*i) and P0(4*i+1) and P0(4*i+2);
			--	Inc04(4*i+2) <= P0(4*i) and P0(4*i+1);
			--	Inc04(4*i+1) <= P0(4*i);
			--	Inc04(4*i+0) <= '1';
			i_3 : AND3 port map (P0(4*i), P0(4*i+1), P0(4*i+2), Inc04(4*i+3));
			i_2 : AND2 port map (P0(4*i), P0(4*i+1), Inc04(4*i+2));
			i_1 : Inc04(4*i+1) <= P0(4*i);
			i_0 : Inc04(4*i) <= '1';
		end generate;

		-- first-level carry look-ahead
		-- (like regular CLA but without carry input)
		-- d=4
		cla1 : for i in 0 to WIDTH/4-1 generate
			bl : block
				alias Gi : std_ulogic_vector(3 downto 0) is G0(4*i+3 downto 4*i);
				alias Pi : std_ulogic_vector(3 downto 0) is P0(4*i+3 downto 4*i);
				alias Co : std_ulogic_vector(3 downto 0) is C1(4*i+3 downto 4*i);
				alias Go : std_ulogic is G1(i);
				alias Po : std_ulogic is P1(i);
				signal t : std_ulogic_vector(5 downto 0);
			begin
				-- Behaviour:
				--	Po <= Pi(0) and Pi(1) and Pi(2) and Pi(3);
				--	Go <= Gi(3)
				--	   or (Pi(3) and Gi(2))
				--	   or (Pi(3) and Pi(2) and Gi(1))
				--	   or (Pi(3) and Pi(2) and Pi(1) and Gi(0));
				--	Co <= (
				--		3 => Gi(2)
				--		  or (Pi(2) and Gi(1))
				--		  or (Pi(2) and Pi(1) and Gi(0)),
				--		2 => Gi(1)
				--		  or (Pi(1) and Gi(0)),
				--		1 => Gi(0),
				--		0 => '0'
				--	);
				prop : AND4 port map (Pi(3), Pi(2), Pi(1), Pi(0), Po);
				tmp0 : AND2 port map (Pi(3), Gi(2), t(0));
				tmp1 : AND3 port map (Pi(3), Pi(2), Gi(1), t(1));
				tmp2 : AND4 port map (Pi(3), Pi(2), Pi(1), Gi(0), t(2));
				gen  : OR4  port map (Gi(3), t(0), t(1), t(2), Go);
				co0  : Co(0) <= '0';
				co1  : Co(1) <= Gi(0);
				tmp3 : AND2 port map (Pi(1), Gi(0), t(3));
				co2  : OR2  port map (Gi(1), t(3), Co(2));
				tmp4 : AND2 port map (Pi(2), Gi(1), t(4));
				tmp5 : AND3 port map (Pi(2), Pi(1), Gi(0), t(5));
				co3  : OR3  port map (Gi(2), t(4), t(5), Co(3));
			end block;
		end generate;

		-- 4-bit partial results
		-- d=5
		res_04 : for i in 0 to WIDTH-1 generate
			-- Behaviour:
			--	Y04(i) <= P0(i) xor C1(i);
			y_4 : XOR2 port map (P0(i), C1(i), Y04(i));
		end generate;

		-- 16-bit increment vector
		-- d=5
		inc_16 : for i in 0 to WIDTH/16-1 generate
			-- Behaviour:
			--	Inc16(4*i+3) <= (not U08 and P1(4*i+2))
			--				 or (P1(4*i) and P1(4*i+1) and P1(4*i+2));
			--	Inc16(4*i+2) <= (not U08)
			--				 or (P1(4*i) and P1(4*i+1));
			--	Inc16(4*i+1) <= P1(4*i);
			--	Inc16(4*i+0) <= '1';
			bl : block
				signal t1, t2, t3 : std_ulogic;
			begin
				tmp1 : AND3 port map (P1(4*i), P1(4*i+1), P1(4*i+2), t1);
				tmp2 : AND2 port map (un08, P1(4*i+2), t2);
				i_3  : OR2  port map (t1, t2, Inc16(4*i+3));
				tmp3 : AND2 port map (P1(4*i), P1(4*i+1), t3);
				i_2  : OR2  port map (un08, t3, Inc16(4*i+2));
				i_1  : Inc16(4*i+1) <= P1(4*i);
				i_0  : Inc16(4*i) <= '1';
			end block;
		end generate;

		-- second-level carry look-ahead
		-- (regular CLA - carry input + SIMD split after 2 bits)
		-- d=6
		cla2 : for i in 0 to WIDTH/16-1 generate
			bl : block
				alias Gi : std_ulogic_vector(3 downto 0) is G1(4*i+3 downto 4*i);
				alias Pi : std_ulogic_vector(3 downto 0) is P1(4*i+3 downto 4*i);
				alias Co : std_ulogic_vector(3 downto 0) is C2(4*i+3 downto 4*i);
				alias Go : std_ulogic is G2(i);
				alias Po : std_ulogic is P2(i);
				signal t : std_ulogic_vector(6 downto 0);
			begin
				-- Behaviour:
				-- Note: for Po and Go, U08 = '1' is assumed (16-bit mode)
				--	Po <= Pi(0) and Pi(1) and Pi(2) and Pi(3);
				--	Go <= Gi(3)
				--	   or (Pi(3) and Gi(2))
				--	   or (Pi(3) and Pi(2) and Gi(1))
				--	   or (Pi(3) and Pi(2) and Pi(1) and Gi(0));
				--	Co <= (
				--		3 => Gi(2)
				--		  or (U08 and Pi(2) and Gi(1))
				--		  or (U08 and Pi(2) and Pi(1) and Gi(0)),
				--		2 => (U08 and Gi(1))
				--		  or (U08 and Pi(1) and Gi(0)),
				--		1 => Gi(0),
				--		0 => '0'
				--	);
				prop : AND4 port map (Pi(3), Pi(2), Pi(1), Pi(0), Po);
				tmp0 : AND2 port map (Pi(3), Gi(2), t(0));
				tmp1 : AND3 port map (Pi(3), Pi(2), Gi(1), t(1));
				tmp2 : AND4 port map (Pi(3), Pi(2), Pi(1), Gi(0), t(2));
				gen  : OR4  port map (Gi(3), t(0), t(1), t(2), Go);
				co0  : Co(0) <= '0';
				co1  : Co(1) <= Gi(0);
				tmp3 : AND3 port map (U08, Pi(1), Gi(0), t(3));
				tmp4 : AND2 port map (U08, Gi(1), t(4));
				co2  : OR2  port map (t(3), t(4), Co(2));
				tmp5 : AND3 port map (U08, Pi(2), Gi(1), t(5));
				tmp6 : AND4 port map (U08, Pi(2), Pi(1), Gi(0), t(6));
				co3  : OR3  port map (Gi(2), t(5), t(6), Co(3));
			end block;
		end generate;

		-- 8-bit carry out (used in tap and second stage)
		-- d=6
		carry8 : for i in 0 to WIDTH/8-1 generate
			bl : block
				signal t : std_ulogic;
			begin
				-- Behaviour:
				--	C08(8*i) <= G1(2*i+1) or (P1(2*i+1) and G1(2*i));
				tmp  : AND2 port map (P1(2*i+1), G1(2*i), t);
				c_08 : OR2  port map (G1(2*i+1), t, C08(i));
			end block;
		end generate;

		-- 8-bit SIMD add/sub tap
		-- d=6
		simd8 : for i in 0 to WIDTH/8-1 generate
			bl : block
				constant ii : natural := 8 * i;
				signal t1, t2, t3 : std_ulogic_vector(7 downto 0);
				signal Inc : std_ulogic_vector(7 downto 0);
			begin
				-- Behaviour:
				--	Inc <= (7 downto 4 => G1(2*i), others => '0');
				--	Y8l(8*i+j) <= Sub
				--			  xor P0(8*i+j)
				--			  xor C1(8*i+j)
				--			  xor (Inc04(8*i+j) and Inc(j));
				--	Y8h(8*i) <= G1(2*i+1) or (P1(2*i+1) and G1(2*i));
				--	Y8h(8*i+7 downto 8*i+1) <= (others => '0');

				-- local increment vector
				-- d=4
				Inc <= (
					7 downto 4 => G1(2*i),
					others => '0'
				);

				-- sum output
				-- d=6
				out8 : for j in 0 to 7 generate
					-- Note the increasing delay!
					tmp1 : XOR2 port map (Sub, P0(ii+j), t1(j));
					tmp2 : XOR2 port map (t1(j), C1(ii+j), t2(j));
					tmp3 : AND2 port map (Inc(j), Inc04(ii+j), t3(j));
					y_8  : XOR2 port map (t2(j), t3(j), Y8l(ii+j));
				end generate;

				-- carry output
				-- d=6
				cy8_n : Y8h(ii+7 downto ii+1) <= (others => '0');
				cy8_0 : Y8h(ii) <= C08(i);
			end block;
		end generate;
	end block stage_1;

	-- TODO: ADD PIPELINE REGISTER HERE !!!

	stage_2 : block
		-- signals used by stage 2 exclusively
		signal C3 : std_ulogic_vector(WIDTH/16-1 downto 0);
		signal Y16 : std_ulogic_vector(WIDTH-1 downto 0);
		signal Y64 : std_ulogic_vector(WIDTH-1 downto 0);
		signal S : std_ulogic_vector(WIDTH/8-1 downto 0) := (others => '0');
	begin
		-- 16-bit partial results
		-- d=8
		res_16 : for i in 0 to WIDTH-1 generate
			-- Behaviour:
			--	Y16(i) <= Y04(i) xor (Inc04(i) and C2(i/4));
			bl : block
				signal t : std_ulogic;
			begin
				tmp  : AND2 port map (Inc04(i), C2(i/4), t);
				y_16 : XOR2 port map (Y04(i), t, Y16(i));
			end block;
		end generate;

		-- third-level carry look-ahead
		-- (regular CLA - gen/prop outputs - carry input + SIMD splits after every bit)
		-- d=8
		cla3 : for i in 0 to WIDTH/64-1 generate
			bl : block
				alias Gi : std_ulogic_vector(3 downto 0) is G2(4*i+3 downto 4*i);
				alias Pi : std_ulogic_vector(3 downto 0) is P2(4*i+3 downto 4*i);
				alias Co : std_ulogic_vector(3 downto 0) is C3(4*i+3 downto 4*i);
				signal t : std_ulogic_vector(4 downto 0);
			begin
				-- Behaviour:
				--	Co <= (
				--		3 => (U16 and Gi(2))
				--		  or (U32 and Pi(2) and Gi(1))
				--		  or (U32 and Pi(2) and Pi(1) and Gi(0)),
				--		2 => (U32 and Gi(1))
				--		  or (U32 and Pi(1) and Gi(0)),
				--		1 => (U16 and Gi(0)),
				--		0 => '0'
				--	);
				co0  : Co(0) <= '0';
				co1  : AND2 port map (U16, Gi(0), Co(1));
				tmp0 : AND3 port map (U32, Pi(1), Gi(0), t(0));
				tmp1 : AND2 port map (U32, Gi(1), t(1));
				co2  : OR2  port map (t(0), t(1), Co(2));
				tmp2 : AND2 port map (U16, Gi(2), t(2));
				tmp3 : AND3 port map (U32, Pi(2), Gi(1), t(3));
				tmp4 : AND4 port map (U32, Pi(2), Pi(1), Gi(0), t(4));
				co3  : OR3  port map (t(2), t(3), t(4), Co(3));
			end block;
		end generate;

		-- 64-bit result
		-- d=10
		res_64 : for i in 0 to WIDTH-1 generate
			-- Behaviour:
			--	Y64(i) <= Y16(i) xor (Inc04(i) and Inc16(i/4) and C3(i/16));
			bl : block
				signal t : std_ulogic;
			begin
				tmp  : AND3 port map (Inc04(i), Inc16(i/4), C3(i/16), t);
				y_64 : XOR2 port map (Y16(i), t, Y64(i));
			end block;
		end generate;

		-- saturate/carry logic
		-- d=10
		sat_and_carry : block
			-- Note!!! some signals are not driven explicitly!
			signal C16, C32, C64 : std_ulogic_vector(WIDTH/8-1 downto 0)
				:= (others => '0');
			signal un08, un16, un32 : std_ulogic;
		begin
			-- mode switches inverted
			-- d=7 (due to pipelining)
			u08n : NOT1 port map (U08, un08);
			u16n : NOT1 port map (U16, un16);
			u32n : NOT1 port map (U32, un32);

			-- carry output vectors for each width
			-- d=8
			carry16 : for i in WIDTH/16-1 downto 0 generate
				-- 16-bit carry out
				C16(2*i) <= G2(i);
			end generate;

			carry32 : for i in WIDTH/32-1 downto 0 generate
				-- 32-bit carry out
				--	C32(4*i) <= G2(2*i+1) or (P2(2*i+1) and G2(2*i));
				bl : block
					signal t : std_ulogic;
				begin
					tmp : AND2 port map (P2(2*i+1), G2(2*i), t);
					co  : OR2  port map (G2(2*i+1), t, C32(4*i));
				end block;
			end generate;

			carry64 : for i in WIDTH/64-1 downto 0 generate
				-- 64-bit carry out
				--	C64(8*i) <= (G2(4*i+3))
				--			 or (P2(4*i+3) and G2(4*i+2))
				--			 or (P2(4*i+3) and P2(4*i+2) and G2(4*i+1))
				--			 or (P2(4*i+3) and P2(4*i+2) and P2(4*i+1) and G2(4*i+0));
				bl : block
					signal t1, t2, t3 : std_ulogic;
				begin
					tmp1 : AND2 port map (P2(4*i+3), G2(4*i+2), t1);
					tmp2 : AND3 port map (P2(4*i+3), P2(4*i+2), G2(4*i+1), t2);
					tmp3 : AND4 port map (P2(4*i+3), P2(4*i+2), P2(4*i+1), G2(4*i+0), t3);
					co   : OR4  port map (G2(4*i+3), t1, t2, t3, C64(8*i));
				end block;
			end generate;

			-- saturate vector (taken from carry outputs)
			-- d=10
			saturate : for i in WIDTH/8-1 downto 0 generate
				bl : block
					alias S08 : std_ulogic is C08(i);
					alias S16 : std_ulogic is C16(2*(i/2));
					alias S32 : std_ulogic is C32(4*(i/4));
					alias S64 : std_ulogic is C64(8*(i/8));

					signal t1, t2, t3, t4 : std_ulogic;
				begin
					-- Behaviour:
					--	S(i) <= (Sat and S08         and not U08)
					--		 or (Sat and S16 and U08 and not U16)
					--		 or (Sat and S32 and U16 and not U32)
					--		 or (Sat and S64 and U32);
					tmp1  : AND3 port map (Sat, S08, un08, t1);
					tmp2  : AND4 port map (Sat, S16, U08, un16, t2);
					tmp3  : AND4 port map (Sat, S32, U16, un32, t3);
					tmp4  : AND3 port map (Sat, S64, U32, t4);
					sat   : OR4  port map (t1, t2, t3, t4, S(i));
				end block;
			end generate;

			-- high output vector
			-- d=10
			high_out : for i in WIDTH/8-1 downto 0 generate
				bl : block
					signal t1, t2, t3, t4, t5 : std_ulogic;
				begin
					-- Behaviour:
					--	Yh(8*i) <= (C08(i)         and not U08)
					--			or (C16(i) and U08 and not U16)
					--			or (C32(i) and U16 and not U32)
					--			or (C64(i) and U32);
					tmp1 : AND2 port map (C08(i), un08, t1);
					tmp2 : AND3 port map (C16(i), U08, un16, t2);
					tmp3 : AND3 port map (C32(i), U16, un32, t3);
					tmp4 : AND2 port map (C64(i), U32, t4);
					tmp5 : OR4  port map (t1, t2, t3, t4, t5);
					--
					-- Note that this differs from the F-CPU
					-- manual, Rev.0.2.  In the manual, the
					-- `subb' borrow output is set to all 1's
					-- (numeric value -1) while this unit
					-- sets it to the numeric value 1.
					-- This is much easier to do in the
					-- presence of SIMD, and it's also
					-- more logical: `borrow -1' actually
					-- means `add 1', which is wrong.
					--
					Yh(8*i+7 downto 8*i+1) <= (others => '0');
					Yh(8*i) <= t5;
				end block;
			end generate;
		end block;

		-- output stage
		-- (saturate and invert)
		-- d=12
		output : for i in 0 to WIDTH-1 generate
			-- Behaviour:
			--	Yl(i) <= (Y64(i) or S(i/8)) xor Sub;
			bl : block
				signal t : std_ulogic;
			begin
				tmp : OR2  port map (Y64(i), S(i/8), t);
				y_l : XOR2 port map (t, Sub, Yl(i));
			end block;
		end generate;
	end block stage_2;
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please

