-- File F-CPU_config.vhdl
-- (c) Yann GUIDON, oct. 21, 2000
-- whygee@f-cpu.org / f-cpu@egroups.com
-- 
-- v0.2 : Michael Riepe changed F_RANGE
-- v0.3 : YG specified the user-modifiable constants + GPL
-- v0.4 : MR proposed LOGMAXSIZE, YG added the ROP2 constants.
--
--   current release : oct. 26, 2000
-- 
-- This package defines the "characteristic widths" of
-- the internal units. Please respect the restrictions.
--
--  * LOGMAXSIZE : Log2 of the Size of the registers in bytes.
--  Can be any integer above or equal to 2. 2 corresponds to
--  a 32-bit implementation, 3 correspods to a 64-bit version.
--  This is the most important parameter, the first with
--  which one can play. Be careful anyway.
--
--  * L1LogLines : Log2 of the NumBer of cache Lines (MUST be EVEN)
--  This parameter determines how many L1 cache lines are implemented.
--  It must be >=4 and _even_ because of the particular LRU mechanism
--  used for this prototype. Allowed values are 4, 6 or 8 (that is :
--  16, 64 or 256 lines, or 512 bytes, 2KB or 8KB). More would correspond
--  to a L2 cache... but are possible if you have enough ressources.
--
--  * L1ABwidth :Address Bus width, in 32-byte chuncks (32+5=128GB)
--  This determines the width of the address comparator of every L1
--  cache line. Be careful, too many bits might require a LOT of ressources.
--  A reasonable value for a small design would be 16 (2MB of adressable
--  physical memory), adapt as required. Warning : this parameter
--  also determines how many address bits are physically implemented.
--
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--

LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;

package FCPU_config is

------------------------------------------------------
-- Most important F-CPU constants :
------------------------------------------------------

  constant LOGMAXSIZE : natural := 3;                -- Number >=2, 3 corresponds to 64-bit registers
  constant MAXSIZE : natural := 2**LOGMAXSIZE;       -- Size of the registers in bytes
  constant UMAX : natural := MAXSIZE * 8;            -- Size of the registers in bits.
  subtype F_RANGE is natural range UMAX-1 downto 0 ; -- Range of a register width declaration.
  subtype F_VECTOR is std_ulogic_vector(F_RANGE) ;   -- shortcut for a very common declaration.

------------------------------------------------------
-- Some architectural constants, bound to FC0 only :
------------------------------------------------------

-- L1 Caches (split instructions and data)
  constant L1DBwidth  : natural := 256 ; -- Data Bus width, or width of each cache line (32 bytes)
  constant L1ABwidth  : natural := 32  ; -- Address Bus width, in 32-byte chuncks (32+5=128GB)
  constant L1LogLines : natural := 6   ; -- Log2 of the NumBer of cache Lines (MUST be EVEN) (small number for the first attempts)
  constant L1NBlines  : natural := 2**L1LogLines ; -- NumBer of cache Lines (2**L1LogLines)

------------------------------------------------------
-- The Special Registers : (copied from SR.h)
--
-- What the user should modify when implementing the core :
-- * SR_NUMBERS_val  should be updated when the
--     number of implemented SR changes.
-- * SR_FAMILY_val   specifies the type of core (FC0, FC1 etc).
--     This is meant to be used for selecting particular code
--     sections that are optimized for certain cores.
-- * SR_STEPPING_val specifies the mask revision, for example.
--
-- DO NOT MODIFY the other constants unless the specifications change.
-- A set of "signature" SRs will appear soon. Stay tuned.
------------------------------------------------------

  constant SR_NUMBERS      : natural := 0;
  constant SR_NUMBERS_val  : natural := 10;      -- 10 SRs are implemented in this model

  constant SR_FAMILY       : natural := 1;
  constant SR_FAMILY_val   : natural := 4032;     -- F-CPU core number. remark : 0xFC0 = 4032 :-)

  constant SR_STEPPING     : natural := 2;
  constant SR_STEPPING_val : natural := 1;       -- revision/implementation

  constant SR_MAX_SIZE     : natural := 3;
  constant SR_MAX_SIZE_val : natural := MAXSIZE; -- in bytes, a power of two >= 4

  constant SR_SIZE_1       : natural := 4;
  constant SR_SIZE_1_val   : natural := 1;       -- Size attribute 1, hardwired if SR_MAX_SIZE <= 8

  constant SR_SIZE_2       : natural := 5;
  constant SR_SIZE_2_val   : natural := 2;       -- Size attribute 2, hardwired if SR_MAX_SIZE <= 8

  constant SR_SIZE_3       : natural := 6;
  constant SR_SIZE_3_val   : natural := 4;       -- Size attribute 3, hardwired if SR_MAX_SIZE <= 8

  constant SR_SIZE_4       : natural := 7;
  constant SR_SIZE_4_val   : natural := 8;       -- Size attribute 4, hardwired if SR_MAX_SIZE <= 8

  constant SR_CYCLE        : natural := 8;       -- R/W, Value is dynamic, incremented every cycle.

  constant SR_PAGING       : natural := 9;       -- Protected, R/W, Controls the paged memory.


-------------------------------------------------------
-- The ROP2 unit : these constants specify the
-- correspondance between the binary code and the actual
-- operation. These data are copied here for convenience
-- only, for example if you want to make an assembler in
-- VHDL. Check the file ROP2.vhdl for more informations.
--------------------------------------------------------

  constant ROP2_AND        : natural := 0;
  constant ROP2_ANDN       : natural := 1;
  constant ROP2_XOR        : natural := 2;
  constant ROP2_OR         : natural := 3;
  constant ROP2_NOR        : natural := 4;
  constant ROP2_XNOR       : natural := 5;
  constant ROP2_ORN        : natural := 6;
  constant ROP2_NAND       : natural := 7;


end FCPU_config;


package body FCPU_config is

------------------------------------------------------
-- Empty. Only the constants matter until today.
-- Some useful wrappers or functions could be included
-- here if they are necessary for rest of the project.
------------------------------------------------------

end FCPU_config;


