--------------------------------------------------------------------------
-- f-cpu/vhdl/registers/R7.vhdl - R7  = Register Set for the F-CPU
-- Copyright (C) 2002 Etienne Labarre (etienne.labarre@gadz.org)
-- Fri Mar  8 22:35:19 CET 2002
--
--------------------------BEGIN-VHDL-LICENCE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENCE------------------------------
--
-- R7 Unit
-- Register Set : REG_NUMBER registers, with register width = UMAX
-- 
-- R7 organisation :
--
-- REG_NUMBER-1 normal registers, register 0 is NULL by hardware
-- Each register is an association of ROW_NUMBER blocks of bits
--
-- For UMAX = 64 bits, organisation is
-- ROW_NUMBER = 5
-- bit 00 to 07 = block 0 = 8 bits 
-- bit 08 to 15 = block 1 = 8 bits
-- bit 16 to 31 = block 2 = 16 bits 
-- bit 32 to 47 = block 3 = 16 bits 
-- bit 48 to 63 = block 4 = 16 bits 
--
-- authorized combinaisons (size information) for UMAX = 64
-- 00001 first  8 bits  access
-- 00010 second 8 bits  access
-- 00011 first  16 bits access
-- 00100 second 16 bits access
-- 01000 third  16 bits access
-- 10000 fourth 16 bits access
-- 00111 first  32 bits access
-- 11000 second 32 bits access
-- 11111        64 bits access
--
-- Write for each block controled by write_mask (ROW_NUMBER bits)
--   1 : write permission, 0 : no change.
-- At read operation, conditions signals are computed
--   LSB  signal : set when LSB of register is set (compliant with size info)
--   MSB  signal : set when MSB of register is set (compliant with size info)
--   NULL signal : reset when register is NULL (compliant with size info)
--
-- See row.vhdl for details
-- 
--------------------------------------------------------------------------

-- include standard libraries
LIBRARY ieee;
USE ieee.std_logic_1164.ALL;
USE ieee.numeric_std.all;

-- include global definition for F-CPU
LIBRARY work;
USE work.FCPU_config.ALL;

-- include R7_row entity (one block)
USE work.R7_row;

-- dfinition de l'interface du circuit :
ENTITY R7 is
generic(
   -- number of block
   ROW_NUMBER   : natural := 5;
   -- number of registers (log) : 6 => 64 registers
   REGISTER_LOG : natural := 6
   );
port (
   -- 3 read adress
   R7_read_adress_0,
   R7_read_adress_1,
   R7_read_adress_2 	: in std_ulogic_vector(REGISTER_LOG-1 downto 0);

   -- 3 read ports
   R7_read_port_0,
   R7_read_port_1,
   R7_read_port_2 	: inout F_VECTOR;
	
   -- 2 write adress
   R7_write_adress_0,
   R7_write_adress_1 	: in std_ulogic_vector(REGISTER_LOG-1 downto 0);

   -- 2 write ports
   R7_write_port_0,
   R7_write_port_1 	: in F_VECTOR;

   -- 2 masks for write permission. One bit per block
   R7_write_mask_0,
   R7_write_mask_1	: in std_ulogic_vector(ROW_NUMBER-1 downto 0);

   -- 2 masks for read operation.
   -- NULL signal, LSB signal, and MSB signal are computed 
   -- with coherent size information
   R7_read_mask_0,
   R7_read_mask_1,
   R7_read_mask_2	: in std_ulogic_vector(ROW_NUMBER-1 downto 0);

   -- condition signals
   R7_lsb_0  : out std_ulogic;
   R7_lsb_1  : out std_ulogic;
   R7_lsb_2  : out std_ulogic;
   R7_msb_0  : out std_ulogic;
   R7_msb_1  : out std_ulogic;
   R7_msb_2  : out std_ulogic;
   R7_null_0  : out std_ulogic;
   R7_null_1  : out std_ulogic;
   R7_null_2  : out std_ulogic;

   -- control signals
   R7_control_read_0,
   R7_control_read_1,
   R7_control_read_2 	: in std_ulogic;
   R7_control_write_0,
   R7_control_write_1 	: in std_ulogic;
   R7_clock 		: in std_ulogic
   );
end R7;

Architecture simu_R7 of R7 is
   -- definition of blocks configuration
   -- ******* valid only if UMAX = 64 ***********
   -- 
   type t_array is array(natural range <>) of natural;
   constant ROW_BITS : t_array(0 to ROW_NUMBER) := 
      (0,8,16,32,48,64);
   constant ROW_SIZE : t_array(0 to ROW_NUMBER-1) := (
      ROW_BITS(1)-ROW_BITS(0),
      ROW_BITS(2)-ROW_BITS(1),
      ROW_BITS(3)-ROW_BITS(2),
      ROW_BITS(4)-ROW_BITS(3),
      ROW_BITS(5)-ROW_BITS(4) 
      );
   -- definition of valid operations
   -- ***** Width of signal and constant is static for correct with/select operation ****
   -- ***** It's not the best method
   constant BYTE_0_OP  : std_ulogic_vector(4 downto 0) := "00001";
   constant BYTE_1_OP  : std_ulogic_vector(4 downto 0) := "00010";
   constant WORD_0_OP  : std_ulogic_vector(4 downto 0) := "00011";
   constant WORD_1_OP  : std_ulogic_vector(4 downto 0) := "00100";
   constant WORD_2_OP  : std_ulogic_vector(4 downto 0) := "01000";
   constant WORD_3_OP  : std_ulogic_vector(4 downto 0) := "10000";
   constant DWORD_0_OP : std_ulogic_vector(4 downto 0) := "00111";
   constant DWORD_1_OP : std_ulogic_vector(4 downto 0) := "11000";
   constant QWORD_OP   : std_ulogic_vector(4 downto 0) := "11111";
   signal sv0 : std_ulogic_vector(4 downto 0);
   signal sv1 : std_ulogic_vector(4 downto 0);
   signal sv2 : std_ulogic_vector(4 downto 0);

   -- internal signals
   signal i_control_write_0 : std_ulogic_vector(ROW_NUMBER-1 downto 0);
   signal i_control_write_1 : std_ulogic_vector(ROW_NUMBER-1 downto 0);

   signal i_null_0 : std_ulogic_vector(ROW_NUMBER-1 downto 0);
   signal i_null_1 : std_ulogic_vector(ROW_NUMBER-1 downto 0);
   signal i_null_2 : std_ulogic_vector(ROW_NUMBER-1 downto 0);
  

begin
   -- condition signals for read port 0
   sv0 <= R7_read_mask_0;
   sv1 <= R7_read_mask_1;
   sv2 <= R7_read_mask_2;

   with sv0 select
      R7_lsb_0 <=
         R7_read_port_0(ROW_BITS(0)) when BYTE_0_OP,
         R7_read_port_0(ROW_BITS(1)) when BYTE_1_OP,
         R7_read_port_0(ROW_BITS(0)) when WORD_0_OP,
         R7_read_port_0(ROW_BITS(2)) when WORD_1_OP,
         R7_read_port_0(ROW_BITS(3)) when WORD_2_OP,
         R7_read_port_0(ROW_BITS(4)) when WORD_3_OP,
         R7_read_port_0(ROW_BITS(0)) when DWORD_0_OP,
         R7_read_port_0(ROW_BITS(3)) when DWORD_1_OP,
         R7_read_port_0(ROW_BITS(0)) when QWORD_OP,
	                         '0' when others;

   with sv1 select
      R7_lsb_1 <=
         R7_read_port_1(ROW_BITS(0)) when BYTE_0_OP,
         R7_read_port_1(ROW_BITS(1)) when BYTE_1_OP,
         R7_read_port_1(ROW_BITS(0)) when WORD_0_OP,
         R7_read_port_1(ROW_BITS(2)) when WORD_1_OP,
         R7_read_port_1(ROW_BITS(3)) when WORD_2_OP,
         R7_read_port_1(ROW_BITS(4)) when WORD_3_OP,
         R7_read_port_1(ROW_BITS(0)) when DWORD_0_OP,
         R7_read_port_1(ROW_BITS(3)) when DWORD_1_OP,
         R7_read_port_1(ROW_BITS(0)) when QWORD_OP,
	                         '0' when others;

   with sv2 select
      R7_lsb_2 <=
         R7_read_port_2(ROW_BITS(0)) when BYTE_0_OP,
         R7_read_port_2(ROW_BITS(1)) when BYTE_1_OP,
         R7_read_port_2(ROW_BITS(0)) when WORD_0_OP,
         R7_read_port_2(ROW_BITS(2)) when WORD_1_OP,
         R7_read_port_2(ROW_BITS(3)) when WORD_2_OP,
         R7_read_port_2(ROW_BITS(4)) when WORD_3_OP,
         R7_read_port_2(ROW_BITS(0)) when DWORD_0_OP,
         R7_read_port_2(ROW_BITS(3)) when DWORD_1_OP,
         R7_read_port_2(ROW_BITS(0)) when QWORD_OP,
	                         '0' when others;

   with sv0 select
      R7_msb_0 <=
         R7_read_port_0(ROW_BITS(1)-1) when BYTE_0_OP,
         R7_read_port_0(ROW_BITS(2)-1) when BYTE_1_OP,
         R7_read_port_0(ROW_BITS(2)-1) when WORD_0_OP,
         R7_read_port_0(ROW_BITS(3)-1) when WORD_1_OP,
         R7_read_port_0(ROW_BITS(4)-1) when WORD_2_OP,
         R7_read_port_0(ROW_BITS(5)-1) when WORD_3_OP,
         R7_read_port_0(ROW_BITS(3)-1) when DWORD_0_OP,
         R7_read_port_0(ROW_BITS(5)-1) when DWORD_1_OP,
         R7_read_port_0(ROW_BITS(5)-1) when QWORD_OP,
	                           '0' when others;

   with sv1 select
      R7_msb_1 <=
         R7_read_port_1(ROW_BITS(1)-1) when BYTE_0_OP,
         R7_read_port_1(ROW_BITS(2)-1) when BYTE_1_OP,
         R7_read_port_1(ROW_BITS(2)-1) when WORD_0_OP,
         R7_read_port_1(ROW_BITS(3)-1) when WORD_1_OP,
         R7_read_port_1(ROW_BITS(4)-1) when WORD_2_OP,
         R7_read_port_1(ROW_BITS(5)-1) when WORD_3_OP,
         R7_read_port_1(ROW_BITS(3)-1) when DWORD_0_OP,
         R7_read_port_1(ROW_BITS(5)-1) when DWORD_1_OP,
         R7_read_port_1(ROW_BITS(5)-1) when QWORD_OP,
	                           '0' when others;

   with sv2 select
      R7_msb_2 <=
         R7_read_port_2(ROW_BITS(1)-1) when BYTE_0_OP,
         R7_read_port_2(ROW_BITS(2)-1) when BYTE_1_OP,
         R7_read_port_2(ROW_BITS(2)-1) when WORD_0_OP,
         R7_read_port_2(ROW_BITS(3)-1) when WORD_1_OP,
         R7_read_port_2(ROW_BITS(4)-1) when WORD_2_OP,
         R7_read_port_2(ROW_BITS(5)-1) when WORD_3_OP,
         R7_read_port_2(ROW_BITS(3)-1) when DWORD_0_OP,
         R7_read_port_2(ROW_BITS(5)-1) when DWORD_1_OP,
         R7_read_port_2(ROW_BITS(5)-1) when QWORD_OP,
	                           '0' when others;

   with sv0 select
      R7_null_0 <=
         i_null_0(0) when BYTE_0_OP,
         i_null_0(1) when BYTE_1_OP,
         i_null_0(1) or 
	 i_null_0(1) when WORD_0_OP,
         i_null_0(2) when WORD_1_OP,
         i_null_0(3) when WORD_2_OP,
         i_null_0(4) when WORD_3_OP,
         i_null_0(0) or 
	 i_null_0(1) or 
	 i_null_0(2) when DWORD_0_OP,
         i_null_0(3) or 
	 i_null_0(4) when DWORD_1_OP,
         i_null_0(0) or 
	 i_null_0(1) or 
	 i_null_0(2) or 
	 i_null_0(3) or 
	 i_null_0(4) when QWORD_OP,
	         '0' when others;

   with sv1 select
      R7_null_1 <=
         i_null_1(0) when BYTE_0_OP,
         i_null_1(1) when BYTE_1_OP,
         i_null_1(1) or 
	 i_null_1(1) when WORD_0_OP,
         i_null_1(2) when WORD_1_OP,
         i_null_1(3) when WORD_2_OP,
         i_null_1(4) when WORD_3_OP,
         i_null_1(0) or 
	 i_null_1(1) or 
	 i_null_1(2) when DWORD_0_OP,
         i_null_1(3) or 
	 i_null_1(4) when DWORD_1_OP,
         i_null_1(0) or 
	 i_null_1(1) or 
	 i_null_1(2) or 
	 i_null_1(3) or 
	 i_null_1(4) when QWORD_OP,
	         '0' when others;

   with sv2 select
      R7_null_2 <=
         i_null_2(0) when BYTE_0_OP,
         i_null_2(1) when BYTE_1_OP,
         i_null_2(1) or 
	 i_null_2(1) when WORD_0_OP,
         i_null_2(2) when WORD_1_OP,
         i_null_2(3) when WORD_2_OP,
         i_null_2(4) when WORD_3_OP,
         i_null_2(0) or 
	 i_null_2(1) or 
	 i_null_2(2) when DWORD_0_OP,
         i_null_2(3) or 
	 i_null_2(4) when DWORD_1_OP,
         i_null_2(0) or 
	 i_null_2(1) or 
	 i_null_2(2) or 
	 i_null_2(3) or 
	 i_null_2(4) when QWORD_OP,
	         '0' when others;

   -- Make full register set with ROW_NUMBER R7_row components.
   row_loop : for i in ROW_NUMBER-1 downto 0 generate

      i_control_write_0(i) <= R7_control_write_0 or not R7_write_mask_0(i);
      i_control_write_1(i) <= R7_control_write_1 or not R7_write_mask_1(i);

      row_n : entity R7_row
         generic map(
            ROW_WIDTH => ROW_SIZE(i),
	    REG_LOG => REGISTER_LOG
	    )
         port map(
	    row_read_port_0 => R7_read_port_0(ROW_BITS(i+1)-1 downto ROW_BITS(i)),
	    row_read_port_1 => R7_read_port_1(ROW_BITS(i+1)-1 downto ROW_BITS(i)),
	    row_read_port_2 => R7_read_port_2(ROW_BITS(i+1)-1 downto ROW_BITS(i)),

	    row_write_port_0 => R7_write_port_0(ROW_BITS(i+1)-1 downto ROW_BITS(i)),
	    row_write_port_1 => R7_write_port_1(ROW_BITS(i+1)-1 downto ROW_BITS(i)),
	    
	    row_read_adress_0 => R7_read_adress_0,
	    row_read_adress_1 => R7_read_adress_1,
	    row_read_adress_2 => R7_read_adress_2,

	    row_write_adress_0 => R7_write_adress_0,
	    row_write_adress_1 => R7_write_adress_1,
 
            row_control_read_0 => R7_control_read_0, 
            row_control_read_1 => R7_control_read_1, 
            row_control_read_2 => R7_control_read_2, 
            
	    row_control_write_0 => i_control_write_0(i),
	    row_control_write_1 => i_control_write_1(i),

            row_null_0 => i_null_0(i),
            row_null_1 => i_null_1(i),
            row_null_2 => i_null_2(i),

	    row_clock => R7_clock
	    );
   end generate;

end;

