-------------------------------------------------------------------------------
-- register_set.vhdl : the FC0 register set
-- created sun feb 10 11:46:32 GMT 2002 by whygee@f-cpu.org
-- version sun feb 17 06:43:23 GMT 2002 : added synchronous version
-- version Mon Jun 24 05:07:47 CEST 2002 : making it work, new version without
--                                         partial writes etc.
--
--------------------------BEGIN-VHDL-LICENSE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENSE------------------------------
--
-- This file implements the FC0 register set by connecting several
-- building blocks and the appropriate signals.
--
-- Though implementing the sign and LSB flags is trivial, the zero and the NaN
-- flags are a bit more sophisticated.
--
-- Clocking :
-- * the read side is purely "combinational" : remember that the input
-- addresses are latched by the instruction fetcher so there is no clock
-- input and it is not latched locally.
-- * the write side is controlled by the scheduler queue for the address,
-- the data comes from the Xbar (and latches it).
--
-- The output to the decoder is simple and unlatched, too.
--
-- If a multi-set bank must be built from simple blocks, it must be
-- done at the level of sram3r2w, where a new architecture must be created.
--
--------------------------------------------------------------------------

LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.numeric_std.all;
LIBRARY work;
    USE work.FCPU_config.ALL;   
    USE work.sram1r2w;
    USE work.sram3r2w;
    
entity register_set is
  port (
    R7_clk : in std_ulogic;

-- read ports :

    R7_read_address_0,
    R7_read_address_1,
    R7_read_address_2 : in t_reg;

    R7_read_port_0,
    R7_read_port_1,
    R7_read_port_2 : inout F_VECTOR; -- it's inout so we can plug
           -- R7_MSB and R7_LSB to the corresponding bits easily.
    R7_MSB,
    R7_LSB,
    R7_zero,
    R7_NaN : out std_ulogic; -- NaN is implemented only when FP exists.

    R7_align : out std_ulogic_vector(LOGMAXSIZE-1 downto 0);
         -- goes to the decoder to check whether the pointer is aligned
         -- against the size specified in the instruction.

-- write ports :

    R7_write_address_0,
    R7_write_address_1 :  in t_reg;

    R7_write_port_0,
    R7_write_port_1 : in F_VECTOR
  );

end register_set;

architecture simple of register_set is
  signal OR_out1, OR_out2,      -- result of the ORs
         SRAM_out,              -- contents of the flag cache
         bypass_0, bypass_1     -- 1 if a bypass must occur
           : std_ulogic;

begin  -- simple

-- instanciates the main block :

  main_block : entity sram3r2w
    generic map (
      width => UMAX )
    port map (
      clk        => R7_clk,
      AdrRead1   => R7_read_address_0,
      AdrRead2   => R7_read_address_1,
      AdrRead3   => R7_read_address_2,
      AdrWrite1  => R7_write_address_0,
      AdrWrite2  => R7_write_address_1,
      DataWrite1 => R7_write_port_0,
      DataWrite2 => R7_write_port_1,
      DataRead1  => R7_read_port_0,
      DataRead2  => R7_read_port_1,
      DataRead3  => R7_read_port_2);

  -- compute the big ORs :
  OR_out1 <= '0' when (R7_write_port_0 = F_VECTOR'(others=>'0')) else '1';
  OR_out2 <= '0' when (R7_write_port_1 = F_VECTOR'(others=>'0')) else '1';

  -- the zero flags :
  zero_block : entity sram1r2w
    port map (
      clk        => R7_clk,
      AdrRead    => R7_read_address_0,
      AdrWrite1  => R7_write_address_0,
      AdrWrite2  => R7_write_address_1,
      DataWrite1 => OR_out1,
      DataWrite2 => OR_out2,
      DataRead   => SRAM_out);

  -- detect the bypass conditions :
  bypass_0 <= '1' when (R7_read_address_0 = R7_write_address_0) else '0';
  bypass_1 <= '1' when (R7_read_address_0 = R7_write_address_1) else '0';

  -- and finally, the flags :
  R7_zero  <= OR_out1 when (bypass_0 = '1') else
              OR_out2 when (bypass_1 = '1') else
              SRAM_out ;
-- an "assert" would be welcome but i don't know how to
-- prevent the problems when the bank is not initialised...
--  assert ???
--    report "Register output != zero flag"
--    severity NOTICE;

  R7_LSB   <= R7_read_port_0(0);
  R7_MSB   <= R7_read_port_0(MAX_CHUNK_SIZE-1);   -- should that really be that ???...
  R7_align <= R7_read_port_0(LOGMAXSIZE-1 downto 0);
  
end simple;
