--------------------------------------------------------------------------
-- f-cpu/vhdl/eu_inc/test_eu_inc.vhdl - Testbench for the EU_INC
-- Copyright (C) 2002 Etienne LABARRE (etienne.labarre@gadz.org)
-- 
-- Sun Jun 30 13:09:23 CEST 2002
--
--------------------------BEGIN-VHDL-LICENCE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENCE------------------------------
--
-- 
--------------------------------------------------------------------------

LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.numeric_std.all;
-- text I/O
    use IEEE.std_logic_textio.all;
    use std.textio.all;
-- local functions
LIBRARY work;
    USE work.random.all;
    USE work.eu_inc;

Entity test_eu_inc256 is
  -- empty
end test_eu_inc256;

Architecture testbench_eu_inc of test_eu_inc256 is

  constant n_test : natural := 100;

  constant WIDTH : natural := 8;
  constant UMX   : natural := 2**WIDTH;

  signal eu_inc_in1_vector : std_ulogic_vector(UMX-1 downto 0);
  signal eu_inc_in2_vector : std_ulogic_vector(UMX-1 downto 0);
  signal eu_inc_out_vector : std_ulogic_vector(UMX-1 downto 0);
  signal eu_inc_ope        : std_ulogic_vector(1 downto 0);
  signal eu_inc_simd_mode  : std_ulogic_vector(WIDTH-1 downto 0);

begin

   -- Component under test :
   eu_inc_component : entity eu_inc
      generic map (eu_inc_width => WIDTH)
      port map(
	 eu_inc_A    => eu_inc_in1_vector,
	 eu_inc_B    => eu_inc_in2_vector,
	 eu_inc_Y    => eu_inc_out_vector,
	 eu_inc_mode => eu_inc_ope,
	 eu_inc_simd => eu_inc_simd_mode
	 );

   -- test bench :
   testbench : process is
      variable lout  : line;
      variable input_vector    : std_ulogic_vector(UMX-1 downto 0);
      variable output_vector   : std_ulogic_vector(UMX-1 downto 0);
      variable expected_vector : std_ulogic_vector(UMX-1 downto 0);

      variable index      : integer;
      variable nb_chunk   : integer;
      variable nb_chunk1  : integer;
      variable base_chunk : integer;

      -- procedure for debug
      procedure debug_result is
         variable lout  : line;
      begin
         write(lout, string'("input  : "));
         write(lout, input_vector);
	 writeline(output, lout);
         write(lout, string'("output : "));
	 write(lout, output_vector);
	 writeline(output, lout);
         write(lout, string'("expected "));
	 write(lout, expected_vector);
	 writeline(output, lout);
      end procedure;

      -- function for increment value
      function inc_val (val : in std_ulogic_vector)
         return std_ulogic_vector is

         variable tmp   : std_ulogic_vector(val'range);
         variable carry : std_ulogic;
         begin
            tmp := val;
            carry  := '1';
            -- increment tmp
	    for j in val'low to val'high loop
	       if carry = '1' then
	          if tmp(j) = '0' then
	             tmp(j) := '1';
	             carry  := '0';
	          else
	             tmp(j) := '0';
	             carry  := '1';
	          end if;
	       end if;
	    end loop;
	    return tmp;
      end function;

      -- function for decrement value
      function dec_val (val : in std_ulogic_vector)
         return std_ulogic_vector is

         variable tmp   : std_ulogic_vector(val'range);
         variable carry : std_ulogic;
         begin
            tmp := val;
            carry  := '1';
            -- decrement tmp
	    for j in val'low to val'high loop
	       if carry = '1' then
	          if tmp(j) = '0' then
	             tmp(j) := '1';
	             carry  := '1';
	          else
	             tmp(j) := '0';
	             carry  := '0';
	          end if;
	       end if;
	    end loop;
	    return tmp;
      end function;

      -- function for invert value
      function inv_val (val : in std_ulogic_vector)
         return std_ulogic_vector is

         variable tmp   : std_ulogic_vector(val'range);
         variable carry : std_ulogic;
         begin
	    tmp := not val;
	    tmp := inc_val(tmp);
	    return tmp;
      end function;

      procedure test_INC ( value : integer ) is

      begin
         for i in 0 to value loop
	    -- input = random number
            input_vector := rand(input_vector);
	    -- apply to tested unit
	    eu_inc_in1_vector <= input_vector;
	    -- search simd level
	    index := WIDTH-1;
	    nb_chunk := 2**WIDTH;
            while eu_inc_simd_mode(index) = '0' loop
	       nb_chunk := nb_chunk/2;
	       index := index-1;
	       exit when index < 0;
	    end loop;
	    -- perform expected value
            for i in 1 to nb_chunk loop
	       expected_vector(i*UMX/nb_chunk-1 downto (i-1)*UMX/nb_chunk) := 
	          inc_val(input_vector(i*UMX/nb_chunk-1 downto (i-1)*UMX/nb_chunk));
            end loop;
	    -- wait for 1 cycle
	    wait for 1 ns;
	    -- Compare result
	    output_vector := eu_inc_out_vector;
	    assert expected_vector = output_vector
	       report "bad increment perform"
	       severity warning;
            if output_vector /= expected_vector then
               debug_result;
	    end if;
	 end loop;
      end procedure;
   
      procedure test_DEC ( value : integer ) is

      begin
         for i in 0 to value loop
	    -- input = random number
            input_vector := rand(input_vector);
	    -- apply to tested unit
	    eu_inc_in1_vector <= input_vector;
	    -- search simd level
	    index := WIDTH-1;
	    nb_chunk := 2**WIDTH;
            while eu_inc_simd_mode(index) = '0' loop
	       nb_chunk := nb_chunk/2;
	       index := index-1;
	       exit when index < 0;
	    end loop;
	    -- perform expected value
            for i in 1 to nb_chunk loop
	       expected_vector(i*UMX/nb_chunk-1 downto (i-1)*UMX/nb_chunk) := 
	          dec_val(input_vector(i*UMX/nb_chunk-1 downto (i-1)*UMX/nb_chunk));
            end loop;
	    -- wait for 1 cycle
	    wait for 1 ns;
	    -- Compare result
	    output_vector := eu_inc_out_vector;
	    assert expected_vector = output_vector
	       report "bad decrement perform"
	       severity warning;
            if output_vector /= expected_vector then
               debug_result;
	    end if;
	 end loop;
      end procedure;
         
      procedure test_NEG ( value : integer ) is

      begin
         for i in 0 to value loop
	    -- input = random number
            input_vector := rand(input_vector);
	    -- apply to tested unit
	    eu_inc_in1_vector <= input_vector;
	    -- search simd level
	    index := WIDTH-1;
	    nb_chunk := 2**WIDTH;
            while eu_inc_simd_mode(index) = '0' loop
	       nb_chunk := nb_chunk/2;
	       index := index-1;
	       exit when index < 0;
	    end loop;
	    -- perform expected value
            for i in 1 to nb_chunk loop
	       expected_vector(i*UMX/nb_chunk-1 downto (i-1)*UMX/nb_chunk) := 
	          inv_val(input_vector(i*UMX/nb_chunk-1 downto (i-1)*UMX/nb_chunk));
            end loop;
	    -- wait for 1 cycle
	    wait for 1 ns;
	    -- Compare result
	    output_vector := eu_inc_out_vector;
	    assert expected_vector = output_vector
	       report "bad invert perform"
	       severity warning;
            if output_vector /= expected_vector then
               debug_result;
	    end if;
	 end loop;
      end procedure;
         
  -- process test
  begin    
     
     write(lout, string'("******************* START OF TEST *******************"));
     writeline(output, lout);
    
     eu_inc_simd_mode <= "00000000";
     wait for 1 ns;
     write(lout, string'("SIMD = "));
     write(lout, eu_inc_simd_mode);
     writeline(output, lout);
     wait for 1 ns;

     write(lout, string'("Test of INC"));
     writeline(output, lout);
     eu_inc_ope <= "11";
     test_INC(n_test);

     write(lout, string'("Test of DEC"));
     writeline(output, lout);
     eu_inc_ope <= "10";
     test_DEC(n_test);

     write(lout, string'("Test of NEG"));
     writeline(output, lout);
     eu_inc_ope <= "00";
     test_NEG(n_test);

     eu_inc_simd_mode <= "00000001";
     wait for 1 ns;
     write(lout, string'("SIMD = "));
     write(lout, eu_inc_simd_mode);
     writeline(output, lout);
     wait for 1 ns;

     write(lout, string'("Test of INC"));
     writeline(output, lout);
     eu_inc_ope <= "11";
     test_INC(n_test);

     write(lout, string'("Test of DEC"));
     writeline(output, lout);
     eu_inc_ope <= "10";
     test_DEC(n_test);

     write(lout, string'("Test of NEG"));
     writeline(output, lout);
     eu_inc_ope <= "00";
     test_NEG(n_test);

     eu_inc_simd_mode <= "00000011";
     wait for 1 ns;
     write(lout, string'("SIMD = "));
     write(lout, eu_inc_simd_mode);
     writeline(output, lout);
     wait for 1 ns;

     write(lout, string'("Test of INC"));
     writeline(output, lout);
     eu_inc_ope <= "11";
     test_INC(n_test);

     write(lout, string'("Test of DEC"));
     writeline(output, lout);
     eu_inc_ope <= "10";
     test_DEC(n_test);

     write(lout, string'("Test of NEG"));
     writeline(output, lout);
     eu_inc_ope <= "00";
     test_NEG(n_test);

     eu_inc_simd_mode <= "00000111";
     wait for 1 ns;
     write(lout, string'("SIMD = "));
     write(lout, eu_inc_simd_mode);
     writeline(output, lout);
     wait for 1 ns;

     write(lout, string'("Test of INC"));
     writeline(output, lout);
     eu_inc_ope <= "11";
     test_INC(n_test);

     write(lout, string'("Test of DEC"));
     writeline(output, lout);
     eu_inc_ope <= "10";
     test_DEC(n_test);

     write(lout, string'("Test of NEG"));
     writeline(output, lout);
     eu_inc_ope <= "00";
     test_NEG(n_test);

     eu_inc_simd_mode <= "00001111";
     wait for 1 ns;
     write(lout, string'("SIMD = "));
     write(lout, eu_inc_simd_mode);
     writeline(output, lout);
     wait for 1 ns;

     write(lout, string'("Test of INC"));
     writeline(output, lout);
     eu_inc_ope <= "11";
     test_INC(n_test);

     write(lout, string'("Test of DEC"));
     writeline(output, lout);
     eu_inc_ope <= "10";
     test_DEC(n_test);

     write(lout, string'("Test of NEG"));
     writeline(output, lout);
     eu_inc_ope <= "00";
     test_NEG(n_test);

     eu_inc_simd_mode <= "00011111";
     wait for 1 ns;
     write(lout, string'("SIMD = "));
     write(lout, eu_inc_simd_mode);
     writeline(output, lout);
     wait for 1 ns;

     write(lout, string'("Test of INC"));
     writeline(output, lout);
     eu_inc_ope <= "11";
     test_INC(n_test);

     write(lout, string'("Test of DEC"));
     writeline(output, lout);
     eu_inc_ope <= "10";
     test_DEC(n_test);

     write(lout, string'("Test of NEG"));
     writeline(output, lout);
     eu_inc_ope <= "00";
     test_NEG(n_test);

     eu_inc_simd_mode <= "00111111";
     wait for 1 ns;
     write(lout, string'("SIMD = "));
     write(lout, eu_inc_simd_mode);
     writeline(output, lout);
     wait for 1 ns;

     write(lout, string'("Test of INC"));
     writeline(output, lout);
     eu_inc_ope <= "11";
     test_INC(n_test);

     write(lout, string'("Test of DEC"));
     writeline(output, lout);
     eu_inc_ope <= "10";
     test_DEC(n_test);

     write(lout, string'("Test of NEG"));
     writeline(output, lout);
     eu_inc_ope <= "00";
     test_NEG(n_test);

     write(lout, string'("*******************  END OF TEST ********************"));
     writeline(output, lout);
     wait;
  end process;

end; 
