--------------------------BEGIN-VHDL-LICENCE-----------------------------
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENCE------------------------------
--
-- f-cpu/vhdl/clock/clock.vhdl, clock generator for the FC0
-- created Wed Sep  5 04:35:14 2001 by Yann GUIDON <whygee@f-cpu.org>
-- version Fri Sep  7 10:19:44 2001
--
-- This file mimics a programable PLL. It is only a first approach that will be
-- refined later. It can already be used for simulation. Synthesised designs
-- could work but i am not sure. Remark : Vanilla doesn't accept it (yet).
-- A separate, stripped-down version is provided in
-- f-cpu/vhdl/clock/clock_vanilla.vhdl
--
-- Mode 0 (prescaled internal clock), Mode 1 (direct external),
-- Mode 2 (prescaled external) should be OK. Mode 3 (PLL) is not working yet :
-- the phase accumulation and frequency correction are not yet implemented.
-- Mode 4 and 5 are variants of mode 0 wich use the PLL's HW to
-- scale the frequency up or down smoothly, in order to avoid power spikes.
-- They are not implemented either, yet.
--
-- Remark : If you want to implement this circuit, you will have to be
-- extremely careful. perform extensive simulations and detect the apropriate
-- ranges for the critical values such as the minimum NCO loop size. SPICE
-- simulations will be necessary for any ASIC target. If you can, use existing
-- analog PLLs available as macros, because of this one is very
-- bad and i believe that it does not behave correctly. At least the other
-- clocking modes will work :-)
--
--
-- WARNING : THIS IS VERY VERY VERY BAD AND UGLY !!!!!

LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.std_logic_textio.all;
    USE ieee.numeric_std.all;
    USE ieee.std_logic_arith.ALL;
LIBRARY std;
    USE std.textio.ALL;
LIBRARY work;
    USE work.clock_config.ALL;

Entity clock is
  port(
-- external (off-core) inputs :
    reset_in : in Std_ulogic;    -- circuit can operate if 1.
    clock_in : in Std_ulogic;    -- external clock input
    snore : in Std_ulogic;       -- ask the CPU to reduce its clock frequency
      -- Note : it has priority over the internal signals but it only works if the
      -- current mode is PLL. If you want to slow-down in HW, you'll have to divide
      -- the external clock yourself. This part should be better managed in the future.
-- configuration (read from the SR) :
    clock_mode : in clock_mode_type;
    prediv,                      -- division factor for the external clock
    premul : in std_ulogic_vector(CLOCK_COUNTER_SIZE-1 downto 0);
       -- division factor for the internal clock
    NCO : in std_ulogic_vector(CLOCK_NCO_SIZE-1 downto 0);  -- internal clock frequency

-- outputs :
    clock_ok : out Std_ulogic;   -- it is 1 if the output frequency is OK.
    clock_out : out Std_ulogic  -- clock output for the FC0
-- warning : this is only a preliminary interface. It will be changed later.
  );
end clock;

-------------------------------------------------------------------------------
-- this one can be used to make quick tests.
-- it won't synthesise but it's helpful for debugging.
-------------------------------------------------------------------------------
Architecture dumb of clock is
  signal clk : std_ulogic := '0';
begin
   -- dumb 100MHz clock (with a nice on-off button)
  clk <= (reset_in and not clk) after 5 ns;
  clock_out <= clk;
   -- allow the core to boot
  clock_ok <= reset_in;
end;


-------------------------------------------------------------------------------
-- this one can be used to make "real" tests.
-- it is a very limited version which requires an external clock and nothing is
-- done at all. Use this architecture if you want to synthesize for the first time.
-------------------------------------------------------------------------------
Architecture simple of clock is
begin
   -- copy the output ("mode 1 only")
  clock_out <= clock_in;
   -- allow the core to boot
  clock_ok <= reset_in;
end;

-------------------------------------------------------------------------------
-- This one is the "big bad ugly" code. Danger ! keep out !
-------------------------------------------------------------------------------

Architecture ugly of clock is
  -- the NCO inverter loop :
  signal NCO_min : std_ulogic_vector(CLOCK_NCO_MIN-1 downto 0) := "101";
  signal NCO_neg : std_ulogic_vector(2**CLOCK_NCO_SIZE downto 0) := (others =>'0');
  signal NCO_pos : std_ulogic_vector(2**CLOCK_NCO_SIZE downto 0) := (others =>'1');
  -- beware of this value : the simulations will become crazy if you put a '0' !
  signal NCO_mux : std_ulogic_vector(2**CLOCK_NCO_SIZE downto 0) := (others =>'0');
  -- controls the internal clock frequency (the result of some multiplexing) :
  signal NCO_control : std_ulogic_vector(CLOCK_NCO_SIZE-1 downto 0)  := (others =>'0');
  -- this one is a "latched signal", the output of a count up/down counter.
  signal NCO_PLL : std_ulogic_vector(CLOCK_NCO_SIZE-1 downto 0)  := (others =>'1');
  -- phase comparator :
  signal phase_int, phase_ext :  std_ulogic_vector(CLOCK_PHASE_ACC-1 downto 0)  := (others =>'0');
  signal phase_min :  std_ulogic_vector(CLOCK_PHASE_ACC-1 downto 0)  := (others =>'0');
  signal phase_max :  std_ulogic_vector(CLOCK_PHASE_ACC-1 downto 0)  := (others =>'1');
  signal phase_overflow : Std_ulogic;   -- set if all the bits in phase_ext are '1'
                                        --> increase the internal frequency
  signal phase_reset : Std_ulogic;      -- set if all the bits in phase_int are '0'
--  signal phase_handshake : Std_ulogic;  -- set/reset to negociate the restart of the externak counter
  signal divide_counter,
         divide_step,
         multiply_counter,
         multiply_step : std_ulogic_vector(CLOCK_COUNTER_SIZE downto 0) := (others =>'0');
begin

-------------------------------------------------------------------------------
--  Some "static" assignations :
-------------------------------------------------------------------------------

  clock_ok <= reset_in;   -- not much test yet. The phase accumulator's
                          -- activity will be reported here one day.

-- take the addition out of the "critical loop" (see below)
--  divide_step   <= '0' & to_StdUlogicVector(unsigned(prediv) + 1);
--  multiply_step <= '0' & to_StdUlogicVector(unsigned(premul) + 1);
-- if you can have a very fast 4-bit adder with carry in, forget these two lines.

-- (mux assumed here)
      -- mode 0 : use the user-defined frequency.
      -- internal mode : delay set to 32 inverter gates => 32/2-3= 13
      -- but the divider reduces this at least by two, so we better take 6.
  NCO_control <=  NCO when (clock_mode=CLOCK_MODE_INTERNAL)
      -- PLL mode
             else NCO_PLL when (clock_mode=CLOCK_MODE_LOCKED_LOOP)
      -- shutdown when the external source is used
             else (others =>'1');

-------------------------------------------------------------------------------
-- YG's braindead design of a NCO (Numerically Controlled Oscillator) :
-- I can't use analog components so i'll use whatever i can.
-------------------------------------------------------------------------------
  -- block the clock when reset active.
  NCO_min(0) <= NCO_mux(0) when (reset_in = '1') else '0' ;

  -- the minimal loop :
  NCO_min(NCO_min'high downto 1) <= not NCO_min(NCO_min'high-1 downto 0) after CLOCK_INVERTER_DELAY;
  NCO_neg(0) <= not NCO_min(NCO_min'high) after CLOCK_INVERTER_DELAY;

  -- the rest of the loop :
  NCO_loop: for i in 1 to 2**CLOCK_NCO_SIZE generate

  begin
    -- inverters (again)
    NCO_pos(i) <= not NCO_neg(i-1) after CLOCK_INVERTER_DELAY;
    NCO_neg(i) <= not NCO_pos(i) after CLOCK_INVERTER_DELAY;
    -- selection
    NCO_mux(i-1) <= NCO_neg(i-1)
                    when (i-1) = unsigned(NCO_control)
                    else NCO_mux(i);
  end generate NCO_loop;
  -- last bit of the loop :
  NCO_mux(2**CLOCK_NCO_SIZE) <= NCO_neg(2**CLOCK_NCO_SIZE);

-- old code : (just FYI, because it is not valid VHDL)
--  NCO_osc_neg(0) <= NCO_osc_pos(NCO_control)
--                       when (NCO_control /= "0000")
--                       else '0' ;
-- oh, btw, i dropped the "else '0'" : the inverter loop is "stalled"
-- at another place now. see the line with the reset.
--
-- remember :
--   1- the minimum "frequency" is 2n+1 inverters plus the MUX delay.
--      n is 1 currently but this may vary
--   2- The total number of inverters in the loop must be odd !
--      otherwise it will not oscillate.


-------------------------------------------------------------------------------
-- NCO output divider :
-------------------------------------------------------------------------------

  -- imply a register here :
  clk_multiply : process (NCO_min(0)) is
  begin
    if rising_edge(NCO_min(0)) then
      if ((clock_mode = CLOCK_MODE_INTERNAL)
       or (clock_mode = CLOCK_MODE_LOCKED_LOOP)) then  -- data gating
        multiply_counter <= to_StdUlogicVector(unsigned(multiply_counter) + unsigned(prediv) + 1);
          -- 5 bits adder, the output is the MSB
      end if;
    end if;
  end process;

-- note : fastest "speed" is when the 'multiply' increment amount is maximum ("1111"). In
-- this case, the output is the input/2. A "0000" input will increase the counter
-- by one. I hope that the synthesiser is smart enough to use the carry in of the adder !
-- this would spare one addition with a constant ...


-- external clock divider :
-- copy + paste the above lines + rename.

  clk_divide : process (clock_in) is
  begin
    if rising_edge(clock_in) then
      if (clock_mode /= CLOCK_MODE_INTERNAL) then   -- disable if not used
        divide_counter <= to_StdUlogicVector(unsigned(divide_counter) + unsigned(prediv) + 1);
      end if;
    end if;
  end process;

-------------------------------------------------------------------------------
-- dummy PLL feedback design :
-------------------------------------------------------------------------------

  -- counter for the internal oscillator
  PLL_int: process (multiply_counter(multiply_counter'high),phase_overflow) is
    variable lout : line ;
  begin
    if phase_overflow='1' then
          WRITE(lout,string'("   overflow phase_ext, phase_int="));
          WRITE(lout,phase_int);
          WRITE(lout,string'(" - time : "));
          WRITE(lout,now);
          WRITELINE(OUTPUT, lout);
      phase_int <= (others => '0');
    elsif rising_edge(multiply_counter(multiply_counter'high)) then
      phase_int <= to_StdUlogicVector(unsigned(phase_int) + 1) ;
    end if;
  end process;
  
  -- counter for the external oscillator
  PLL_ext: process (divide_counter(divide_counter'high),phase_reset) is
    variable lout : line ;
  begin
    if phase_reset='1' then
          WRITE(lout,string'("   overflow phase_int, phase_ext="));
          WRITE(lout,phase_ext);
          WRITE(lout,string'(" - time : "));
          WRITE(lout,now);
          WRITELINE(OUTPUT, lout);
      phase_ext <= (others => '0');
    elsif rising_edge(divide_counter(divide_counter'high)) then
      phase_ext <= to_StdUlogicVector(unsigned(phase_ext) + 1) ;
    end if;
  end process;

  phase_overflow <= '1' when phase_ext = phase_max else '0';  -- big AND
  phase_reset <= '1' when phase_int = phase_max else '0';  -- big AND too

  
  -- INC/DEC
  incdec : process (phase_overflow, phase_reset) is
      variable lout : line ;
    begin
      if rising_edge(phase_reset) or snore='1' then
        -- decrement
        if NCO_PLL /= phase_min then  -- avoid the saturation
          NCO_PLL <= to_StdUlogicVector(unsigned(NCO_PLL) - 1);
          WRITE(lout,string'("(-)  now : "));
          WRITE(lout,NCO_PLL);
          WRITE(lout,string'(" - time : "));
          WRITE(lout,now);
          WRITELINE(OUTPUT, lout);
        end if;
      elsif rising_edge(phase_overflow) then
        -- increment
        if NCO_PLL /= phase_max then  -- avoid the saturation
          NCO_PLL <= to_StdUlogicVector(unsigned(NCO_PLL) + 1);
          WRITE(lout,string'("(+)  now : "));
          WRITE(lout,NCO_PLL);
          WRITE(lout,string'(" - time : "));
          WRITE(lout,now);
          WRITELINE(OUTPUT, lout);
        end if;
      end if;
    end process;

-------------------------------------------------------------------------------
-- select the output :
-------------------------------------------------------------------------------

  with clock_mode select
    clock_out <=
      multiply_counter(multiply_counter'high)  -- take the MSB
        when CLOCK_MODE_INTERNAL,
          -- 'prescaled internal clock'
      clock_in
        when CLOCK_MODE_EXTERNAL,
          -- 'direct external clock'
      divide_counter(divide_counter'high)  -- take the MSB
        when CLOCK_MODE_EXT_PRESCALED,
          -- 'prescaled external clock' 
      NCO_min(0)
        when others;
          -- 'failsafe', PLL mode, slowdown, speedup

end;
