-- im64test.vhdl - F-CPU Integer Multiplication Unit Testbench
-- Copyright (C) 2000, 2001 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- $Id: im64test.vhdl,v 1.10 2001/08/22 13:49:35 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

entity IMul64_test is
end IMul64_test;

architecture Arch_1 of IMul64_test is
	component IMul64
		generic (
			PIPE_AFTER : natural := 0;	-- gates per stage; 0 means no pipelining
			PIPE_DELAY : natural := 0	-- additional delay before 1st stage
		);
		port (
			A : in std_ulogic_vector(63 downto 0);
			B : in std_ulogic_vector(63 downto 0);
			X : in std_ulogic_vector(63 downto 0);
			SignedMode : in std_ulogic;
			Mac : in std_ulogic_vector(2 downto 0);
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y08l : out std_ulogic_vector(63 downto 0);	-- d=18 (3 stages)
			Y08h : out std_ulogic_vector(63 downto 0);	-- d=20 (4 stages)
			Y16l : out std_ulogic_vector(63 downto 0);	-- d=24 (4 stages)
			Y16h : out std_ulogic_vector(63 downto 0);	-- d=26 (5 stages)
			Y32l : out std_ulogic_vector(63 downto 0);	-- d=28 (5 stages)
			Y32h : out std_ulogic_vector(63 downto 0);	-- d=28 (5 stages)
			Y64l : out std_ulogic_vector(63 downto 0);	-- d=32 (6 stages)
			Y64h : out std_ulogic_vector(63 downto 0)	-- d=34 (6 stages)
		);
	end component;

	constant WIDTH : natural := 64;

	signal A, B : std_ulogic_vector(WIDTH-1 downto 0);
	signal X : std_ulogic_vector(WIDTH-1 downto 0);
	signal M : std_ulogic_vector(6 downto 0);
	signal Y08l, Y08h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y16l, Y16h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y32l, Y32h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y64l, Y64h : std_ulogic_vector(WIDTH-1 downto 0);

	signal Clk, Rst, En : std_ulogic;
begin
	Clk <= '0'; Rst <= '0'; En <= '1';

	mut: IMul64
		generic map (PIPE_AFTER => 0, PIPE_DELAY => 0)
		port map (
			A => A, B => B, X => X, SignedMode => M(0),
			Mac => M(3 downto 1), U => M(6 downto 4),
			Clk => Clk, Rst => Rst, En => En,
			Y08l => Y08l, Y08h => Y08h,
			Y16l => Y16l, Y16h => Y16h,
			Y32l => Y32l, Y32h => Y32h,
			Y64l => Y64l, Y64h => Y64h
		);

	test : process
		constant zero : std_ulogic_vector(2*WIDTH-1 downto 0) := (others => '0');
		variable prod, res : integer;
		variable slice : natural;
		variable lout : line;
		variable av, bv : std_ulogic_vector(WIDTH-1 downto 0);
		variable pv8, rv8 : std_ulogic_vector(15 downto 0);
		variable pv16, rv16 : std_ulogic_vector(31 downto 0);
		variable pv32, rv32 : std_ulogic_vector(63 downto 0);
		variable pv64, rv64 : std_ulogic_vector(127 downto 0);
	begin
if true then
		write(lout, string'("*** Testing 8-bit signed multiplication"));
		writeline(output, lout);
		M <= "0000001";
		X <= (others => '0');
		av := (others => '0');
		bv := (others => '0');
		slice := 0;
		o8 : for i in -128 to 127 loop
			for j in -128 to 127 loop
				av(slice+7 downto slice) :=
					std_ulogic_vector(to_signed(i, 8));
				bv(slice+7 downto slice) :=
					std_ulogic_vector(to_signed(j, 8));
				A <= av;
				B <= bv;
				wait for 1 ns;
				rv8 := Y08h(slice+7 downto slice)
				     & Y08l(slice+7 downto slice);
				prod := i * j;
				pv8 := std_ulogic_vector(to_signed(prod, 16));
				if (pv8 /= rv8) then
					write(lout, string'("WHOA THERE!!! "));
					write(lout, i);
					write(lout, string'(" * "));
					write(lout, j);
					write(lout, string'(" => "));
					write(lout, rv8);
					write(lout, string'(" /= "));
					write(lout, pv8);
					write(lout, string'(" slice = "));
					write(lout, slice);
					writeline(output, lout);
					next o8;
				end if;
				slice := (slice + 8) mod 64;
			end loop;
		end loop;
end if;

if true then
		write(lout, string'("*** Testing 16-bit signed multiplication"));
		writeline(output, lout);
		M <= "0010001";
		X <= (others => '0');
		av := (others => '0');
		bv := (others => '0');
		slice := 0;
		o16 : for i in -128 to 127 loop
			for j in -128 to 127 loop
				av(slice+15 downto slice) :=
					std_ulogic_vector(to_signed(i, 16));
				bv(slice+15 downto slice) :=
					std_ulogic_vector(to_signed(j, 16));
				A <= av;
				B <= bv;
				wait for 1 ns;
				rv16 := Y16h(slice+15 downto slice)
				      & Y16l(slice+15 downto slice);
				prod := i * j;
				pv16 := std_ulogic_vector(to_signed(prod, 32));
				if (pv16 /= rv16) then
					write(lout, string'("WHOA THERE!!! "));
					write(lout, i);
					write(lout, string'(" * "));
					write(lout, j);
					write(lout, string'(" => "));
					write(lout, rv16);
					write(lout, string'(" /= "));
					write(lout, pv16);
					write(lout, string'(" slice = "));
					write(lout, slice);
					writeline(output, lout);
					next o16;
				end if;
				slice := (slice + 16) mod 64;
			end loop;
		end loop;
end if;

if true then
		write(lout, string'("*** Testing 32-bit signed multiplication"));
		writeline(output, lout);
		M <= "0110001";
		X <= (others => '0');
		av := (others => '0');
		bv := (others => '0');
		slice := 0;
		o32 : for i in -128 to 127 loop
			for j in -128 to 127 loop
				av(slice+31 downto slice) :=
					std_ulogic_vector(to_signed(i, 32));
				bv(slice+31 downto slice) :=
					std_ulogic_vector(to_signed(j, 32));
				A <= av;
				B <= bv;
				wait for 1 ns;
				rv32 := Y32h(slice+31 downto slice)
				      & Y32l(slice+31 downto slice);
				prod := i * j;
				pv32 := std_ulogic_vector(to_signed(prod, 64));
				if (pv32 /= rv32) then
					write(lout, string'("WHOA THERE!!! "));
					write(lout, i);
					write(lout, string'(" * "));
					write(lout, j);
					write(lout, string'(" => "));
					write(lout, rv32);
					write(lout, string'(" /= "));
					write(lout, pv32);
					write(lout, string'(" slice = "));
					write(lout, slice);
					writeline(output, lout);
					next o32;
				end if;
				slice := (slice + 32) mod 64;
			end loop;
		end loop;
end if;

if true then
		write(lout, string'("*** Testing 64-bit signed multiplication"));
		writeline(output, lout);
		M <= "1110001";
		X <= (others => '0');
		av := (others => '0');
		bv := (others => '0');
		o64 : for i in -128 to 127 loop
			for j in -128 to 127 loop
				av := std_ulogic_vector(to_signed(i, 64));
				bv := std_ulogic_vector(to_signed(j, 64));
				A <= av;
				B <= bv;
				wait for 1 ns;
				rv64 := Y64h & Y64l;
				prod := i * j;
				pv64 := std_ulogic_vector(to_signed(prod, 128));
				if (pv64 /= rv64) then
					write(lout, string'("WHOA THERE!!! "));
					write(lout, i);
					write(lout, string'(" * "));
					write(lout, j);
					write(lout, string'(" => "));
					write(lout, rv64);
					write(lout, string'(" /= "));
					write(lout, pv64);
					writeline(output, lout);
					next o64;
				end if;
			end loop;
		end loop;
end if;

		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
