/*
 * fcotest.c -- Test Program for Flex-based Scanner/Encoder.
 * Copyright (C) 2001 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

static const char rcsid[] = "@(#) $Id: fcotest.c,v 1.1 2001/09/07 22:29:43 michael Exp $";

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>

#include <fcpu_opcodes.h>

void
print_msg(FILE *fp, const char *label, const char *fmt, va_list ap) {
	if (*label) {
		fputs(label, fp);
	}
	vfprintf(fp, fmt, ap);
	fputs("\n", fp);
}

void
warn(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "warning: ", fmt, ap);
	va_end(ap);
}

void
error(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "", fmt, ap);
	va_end(ap);
}

void
fatal(const char *fmt, ...) {
	va_list ap;

	va_start(ap, fmt);
	print_msg(stderr, "fatal error: ", fmt, ap);
	va_end(ap);
	/* no way to recover */
	exit(1);
}

static const char *opcode_table[256] = {
	[OP_ABS] = "abs",
	[OP_ADD] = "add",
	[OP_ADDI] = "addi",
	[OP_ADDSUB] = "addsub",
	[OP_AND] = "and",
	[OP_ANDI] = "andi",
	[OP_ANDN] = "andn",
	[OP_ANDNI] = "andni",
	[OP_BCHG] = "bchg",
	[OP_BCHGI] = "bchgi",
	[OP_BCLR] = "bclr",
	[OP_BCLRI] = "bclri",
	[OP_BITREV] = "bitrev",
	[OP_BITREVI] = "bitrevi",
	[OP_BSET] = "bset",
	[OP_BSETI] = "bseti",
	[OP_BTST] = "btst",
	[OP_BTSTI] = "btsti",
	[OP_BYTEREV] = "byterev",
	[OP_CACHEMM] = "cachemm",
	[OP_CMPL] = "cmpl",
	[OP_CMPLE] = "cmple",
	[OP_CMPLEI] = "cmplei",
	[OP_CMPLI] = "cmpli",
	[OP_DEC] = "dec",
	[OP_DIV] = "div",
	[OP_DIVI] = "divi",
	[OP_EXPAND] = "expand",
	[OP_F2INT] = "f2int",
	[OP_FADD] = "fadd",
	[OP_FADDSUB] = "faddsub",
	[OP_FDIV] = "fdiv",
	[OP_FEXP] = "fexp",
	[OP_FIAPRX] = "fiaprx",
	[OP_FLOG] = "flog",
	[OP_FMAC] = "fmac",
	[OP_FMUL] = "fmul",
	[OP_FSQRT] = "fsqrt",
	[OP_FSQRTIAPRX] = "fsqrtiaprx",
	[OP_FSUB] = "fsub",
	[OP_GET] = "get",
	[OP_GETI] = "geti",
	[OP_HALT] = "halt",
	[OP_INC] = "inc",
	[OP_INT2F] = "int2f",
	[OP_INT2L] = "int2l",
	[OP_JMP] = "jmp",
	[OP_L2INT] = "l2int",
	[OP_LADD] = "ladd",
	[OP_LOAD] = "load",
	[OP_LOADADDR] = "loadaddr",
	[OP_LOADADDRI] = "loadaddri",
	[OP_LOADCONS+0] = "loadcons",
	[OP_LOADCONS+1] = "loadcons",
	[OP_LOADCONS+2] = "loadcons",
	[OP_LOADCONS+3] = "loadcons",
	[OP_LOADCONSX+0] = "loadconsx",
	[OP_LOADCONSX+1] = "loadconsx",
	[OP_LOADCONSX+2] = "loadconsx",
	[OP_LOADCONSX+3] = "loadconsx",
	[OP_LOADF] = "loadf",
	[OP_LOADI] = "loadi",
	[OP_LOADIF] = "loadif",
	[OP_LOADM] = "loadm",
	[OP_LOOP] = "loop",
	[OP_LSUB] = "lsub",
	[OP_MAC] = "mac",
	[OP_MAX] = "max",
	[OP_MAXI] = "maxi",
	[OP_MIN] = "min",
	[OP_MINI] = "mini",
	[OP_MIX] = "mix",
	[OP_MOD] = "mod",
	[OP_MODI] = "modi",
	[OP_MOVE] = "move",
	[OP_MUL] = "mul",
	[OP_MULI] = "muli",
	[OP_NAND] = "nand",
	[OP_NANDI] = "nandi",
	[OP_NEG] = "neg",
	[OP_NOP] = "nop",
	[OP_NOR] = "nor",
	[OP_NORI] = "nori",
	[OP_OR] = "or",
	[OP_ORI] = "ori",
	[OP_ORN] = "orn",
	[OP_ORNI] = "orni",
	[OP_POPC] = "popcount",
	[OP_POPCI] = "popcounti",
	[OP_PUT] = "put",
	[OP_PUTI] = "puti",
	[OP_RFE] = "rfe",
	[OP_ROTL] = "rotl",
	[OP_ROTLI] = "rotli",
	[OP_ROTR] = "rotr",
	[OP_ROTRI] = "rotri",
	[OP_SCAN] = "scan",
	[OP_SDUP] = "sdup",
	[OP_SERIALIZE] = "serialize",
	[OP_SHIFTL] = "shiftl",
	[OP_SHIFTLI] = "shiftli",
	[OP_SHIFTR] = "shiftr",
	[OP_SHIFTRA] = "shiftra",
	[OP_SHIFTRAI] = "shiftrai",
	[OP_SHIFTRI] = "shiftri",
	[OP_SORT] = "sort",
	[OP_SRB_RESTORE] = "srb_restore",
	[OP_SRB_SAVE] = "srb_save",
	[OP_STORE] = "store",
	[OP_STOREF] = "storef",
	[OP_STOREI] = "storei",
	[OP_STOREIF] = "storeif",
	[OP_STOREM] = "storem",
	[OP_SUB] = "sub",
	[OP_SUBI] = "subi",
	[OP_SYSCALL] = "syscall",
	[OP_WIDEN] = "widen",
	[OP_XNOR] = "xnor",
	[OP_XNORI] = "xnori",
	[OP_XOR] = "xor",
	[OP_XORI] = "xori",
};

int
main(int argc, char **argv) {
	char buf[BUFSIZ];
	const char *s;
	unsigned x;

	while (fgets(buf, sizeof(buf), stdin)) {
		s = strtok(buf, " \t\r\f\v\n");
		if (!*s) continue;
		x = fcpu_encode_instruction(s);
		if ((int)x == -2) continue;	/* unsupported */
		if ((int)x == -1) {
			error("unrecognized opcode \"%s\"", s);
			continue;
		}
		printf("%-24s", s);
		s = opcode_table[x >> 24];
		if (!s) {
			error("unknown opcode %02x\n", x >> 24);
			printf("%08x  # unknown\n", x & ~0x3f);
			continue;
		}
		printf("%08x  %-12s", x & ~0x3f, s);
		switch (x & 0x3f) {
			case ARG_NONE:	/* (halt, rfe, srb_save, srb_restore, serialize, nop) */
				break;
			case ARG_R:		/* reg (loopentry) */
				fputs("reg1", stdout);
				break;
			case ARG_RO:		/* reg[,reg] (jmpa) */
				fputs("reg2[,reg1]", stdout);
				break;
			case ARG_RR:		/* reg,reg */
				fputs("reg2,reg1", stdout);
				break;
			case ARG_U16R:	/* uimm16,reg (loadcons, geti, puti, syscall, trap) */
				fputs("uimm16,reg1", stdout);
				break;
			case ARG_S16R:	/* simm16,reg (loadconsx) */
				fputs("simm16,reg1", stdout);
				break;
			case ARG_S17R:	/* simm17,reg (loadaddri) */
				fputs("simm17,reg1", stdout);
				break;
			case ARG_U64R:	/* uimm64,reg (generic loadcons) */
				fputs("uimm64,reg1", stdout);
				break;
			case ARG_S64R:	/* simm64,reg (generic loadconsx) */
				fputs("simm64,reg1", stdout);
				break;
			case ARG_ORR:	/* [reg,]reg,reg (popcount, load[f], store[f]) */
				fputs("[reg3,]reg2,reg1", stdout);
				break;
			case ARG_RRO:	/* reg,reg[,reg] (cjmp) */
				fputs("reg3,reg2[,reg1]", stdout);
				break;
			case ARG_RRR:	/* reg,reg,reg */
				fputs("reg3,reg2,reg1", stdout);
				break;
			case ARG_U8RR:	/* uimm8,reg,reg */
				fputs("uimm8,reg2,reg1", stdout);
				break;
			case ARG_S8RR:	/* simm8,reg,reg (muli, divi) */
				fputs("simm8,reg2,reg1", stdout);
				break;
			case ARG_S9RR:	/* simm9,reg,reg (loadi[f], storei[f]) */
				fputs("simm9,reg2,reg1", stdout);
				break;
			default:
				fputs("# invalid args", stdout);
				error("invalid encoding");
		}
		putchar('\n');
	}
	exit(0);
}
