/*
 * delete.c - copy (some) sections from one ELF file to another.
 * Copyright (C) 1995 - 2002 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <elftools.h>
#include <assert.h>

#ifndef lint
static const char rcsid[] = "@(#) $Id: delete.c,v 1.2 2002/12/30 16:04:51 michael Exp $";
#endif /* lint */

static Elf*
_et_copy(Elf *elf, Elf *nelf, Elf_Scn **delscns, size_t ndel) {
	Elf_Data strtab;
	Elf_Data *data, *ndata;
	Elf_Scn *scn, *nscn;
	GElf_Ehdr eh;
	GElf_Phdr ph;
	GElf_Shdr sh;
	GElf_Addr shstrndx;
	char *name;
	size_t found;
	size_t i;
	size_t index;
	size_t off;

	/*
	 * copy ehdr
	 */
	elf_errno();
	if (!gelf_getehdr(elf, &eh)
	 || !gelf_newehdr(nelf, eh.e_ident[EI_CLASS])) {
		return NULL;
	}
	shstrndx = et_get_shstrndx(elf);
	/*
	 * copy phdr table
	 */
	if (!gelf_newphdr(nelf, eh.e_phnum) && eh.e_phnum) {
		return NULL;
	}
	for (i = 0; i < eh.e_phnum; i++) {
		/*
		 * copy single phdr
		 */
		if (!gelf_getphdr(elf, i, &ph)
		 || !gelf_update_phdr(nelf, i, &ph)) {
			return NULL;
		}
	}
	/*
	 * copy sections
	 */
	et_build_string_table(&strtab, NULL);
	et_build_string_table(&strtab, "");
	assert(strtab.d_size == 1);
	scn = NULL;
	elf_errno();
	found = 0;
	off = 0;
	while ((scn = elf_nextscn(elf, scn))) {
		if (!gelf_getshdr(scn, &sh)) {
			return NULL;
		}
		index = elf_ndxscn(scn);
		assert(index != SHN_UNDEF);
		for (i = 0; i < ndel; i++) {
			if (scn == delscns[i]) {
				break;
			}
		}
		if (i < ndel) {
			/*
			 * delete (i.e., do not copy) section
			 */
			found++;
			continue;
		}
		if (index == shstrndx) {
			/*
			 * do not copy string table
			 */
			continue;
		}
		/*
		 * create new section
		 */
		nscn = elf_newscn(nelf);
		if (!nscn) {
			return NULL;
		}
		/*
		 * update string table
		 */
		name = elf_strptr(elf, shstrndx, sh.sh_name);
		if (!name) {
			return NULL;
		}
		sh.sh_name = et_build_string_table(&strtab, name);
		/*
		 * copy data
		 */
		if (sh.sh_type != SHT_NULL) {
			data = NULL;
			while ((data = elf_getdata(scn, data))) {
				if (!(ndata = elf_newdata(nscn))) {
					return NULL;
				}
				*ndata = *data;
			}
			/*
			 * update offset
			 */
			if (sh.sh_type == SHT_NOBITS) {
				if (off < sh.sh_offset) {
					off = sh.sh_offset;
				}
			}
			else {
				if (off < sh.sh_offset + sh.sh_size) {
					off = sh.sh_offset + sh.sh_size;
				}
			}
		}
		/*
		 * copy section header
		 */
		if (!gelf_update_shdr(nscn, &sh)) {
			return NULL;
		}
	}
	if (elf_errmsg(0)) {
		return NULL;
	}
	if (found != ndel) {
		return NULL;
	}
	/*
	 * rebuild string table
	 */
	nscn = elf_newscn(nelf);
	if (!nscn) {
		return NULL;
	}
	sh.sh_name = et_build_string_table(&strtab, ".shstrtab");
	assert(sh.sh_name + 10 <= strtab.d_size);
	sh.sh_type = SHT_STRTAB;
	sh.sh_flags = 0;
	sh.sh_addr = 0;
	sh.sh_offset = off;
	sh.sh_size = strtab.d_size;
	sh.sh_link = 0;
	sh.sh_info = 0;
	sh.sh_addralign = 1;
	sh.sh_entsize = 0;
	data = elf_newdata(nscn);
	if (!data) {
		return NULL;
	}
	*data = strtab;
	if (!gelf_update_shdr(nscn, &sh)) {
		return NULL;
	}
	if (!gelf_update_ehdr(nelf, &eh)) {
		return NULL;
	}
	if (et_set_shstrndx(nelf, elf_ndxscn(nscn))) {
		return NULL;
	}
	return nelf;
}

Elf*
et_delete_sections(Elf *elf, Elf *nelf, Elf_Scn **delscns, size_t ndel) {
	if (nelf) {
		return _et_copy(elf, nelf, delscns, ndel);
	}
	if (!(nelf = elf_begin(-1, ELF_C_WRITE, NULL))) {
		return NULL;
	}
	if (_et_copy(elf, nelf, delscns, ndel)) {
		return nelf;
	}
	elf_end(nelf);
	return NULL;
}

Elf*
et_delete_section(Elf *elf, Elf *nelf, Elf_Scn *delscn) {
	return et_delete_sections(elf, nelf, &delscn, 1);
}

Elf*
et_copy(Elf *elf, Elf *nelf) {
	return et_delete_sections(elf, nelf, NULL, 0);
}
