/*
 * ar.c - ar(1) for ELF.
 * Copyright (C) 1995 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

#include <common.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <elftools.h>
#include <ar.h>
#include <assert.h>

#if HAVE_UTIME_H
# include <utime.h>
#endif

static const char *name_in_table(int argc, char **argv, const char *name);
static const char *name_from_table(int *argc, char **argv, const char *name);

static int verbose = 0;
static char *arch = NULL;

static int do_delete(int, char**);
static int do_move(int, char**);
static int do_replace(int, char**, int quick);

typedef int (*ro_func)(Elf *elf, Elf_Arhdr *ah, const char *name);

static int ro_print(Elf *elf, Elf_Arhdr *ah, const char *name);
static int ro_table(Elf *elf, Elf_Arhdr *ah, const char *name);
static int ro_extract(Elf *elf, Elf_Arhdr *ah, const char *name);

static int do_readonly(int, char**, ro_func);

#if HAVE_UTIME
static int do_old_date = 0;
#endif /* HAVE_UTIME */
static int do_create = 0;
static int do_ranlib = 0;
static int do_update = 0;
static int do_backup = 0;
static int before = 0;
static char *where = NULL;

static int
do_delete(int argc, char **argv) {
	Elf *arf;
	Elf *elf;
	Elf_Arhdr *ah;
	Elf_Cmd cmd;
	ardesc_t *ap;
	const char *name;
	int modified;
	int err;
	int fd;

	modified = 0;
	if ((fd = open(arch, O_RDWR)) == -1) {
		file_error(arch, "open: %s", strerror(errno));
		return 1;
	}
	if (!(arf = elf_begin(fd, ELF_C_READ, NULL))) {
		file_error(arch, "elf_begin: %s", elf_errmsg(-1));
		close(fd);
		return 1;
	}
	ap = ar_create();
	err = 0;
	elf_errno();
	cmd = ELF_C_READ;
	while (!err && (elf = elf_begin(fd, cmd, arf))) {
		if (!(ah = elf_getarhdr(elf))) {
			file_error(arch, "elf_getarhdr: %s", elf_errmsg(-1));
			err = 1;
		}
		else if (!(name = ah->ar_name)) {
			file_error(arch, "bad member name `%s'", ah->ar_rawname);
			err = 1;
		}
		else if (name[0] == '/') {
			/* skip */;
		}
		else if (argc && (name = name_in_table(argc, argv, name))) {
			if (verbose) {
				printf("d - %s\n", name);
			}
			modified = 1;
		}
		else if (ar_add_elf(ap, elf, *ah)) {
			file_error(ah->ar_name, "could not copy file");
			err = 1;
		}
		cmd = elf_next(elf);
		elf_end(elf);
	}
	if (!err && (err = elf_errno())) {
		file_error(arch, "elf_begin: %s", elf_errmsg(err));
	}
	elf_end(arf);
	close(fd);
	if (!err && modified) {
		err = ar_update_archive(ap, arch, do_backup);
	}
	ar_destroy(ap);
	return err != 0;
}

static int
do_move(int argc, char **argv) {
	/* XXX: add this */
	/* verbose: "m - %s\n", cmd_line_argument */
	error("option -m not supported yet");
	return 1;
}

static const char*
ar_basename(const char *name) {
	const char *s;

	return (s = strrchr(name, '/')) ? s + 1 : name;
}

static int
do_replace(int argc, char **argv, int quick) {
	/* verbose: "r - %s\n", cmd_line_arg ("a -" if file is new) */
	Elf *arf;
	Elf *elf;
	Elf_Arhdr *ah;
	Elf_Cmd cmd;
	ardesc_t *ap;
	const char *name;
	int err;
	int fd;
	int i;

	if (argc == 0) {
		return 0;
	}
	if (where) {
		/* XXX: add this */
		error("options -a, -b and -i not supported yet");
		return 1;
	}
	err = 0;
	ap = ar_create();
	if ((fd = open(arch, O_RDWR)) == -1) {
		if (errno != ENOENT) {
			file_error(arch, "open: %s", strerror(errno));
			ar_destroy(ap);
			return 1;
		}
		if (!do_create) {
			printf("%s does not exist, creating it\n", arch);
		}
	}
	else {
		if (!(arf = elf_begin(fd, ELF_C_READ, NULL))) {
			file_error(arch, "elf_begin: %s", elf_errmsg(-1));
			ar_destroy(ap);
			close(fd);
			return 1;
		}
		elf_errno();
		cmd = ELF_C_READ;
		while (!err && (elf = elf_begin(fd, cmd, arf))) {
			int copy = 0;
			struct stat st;

			if (!(ah = elf_getarhdr(elf))) {
				file_error(arch, "elf_getarhdr: %s", elf_errmsg(-1));
				err = 1;
			}
			else if (!(name = ah->ar_name)) {
				file_error(arch, "bad member name `%s'", ah->ar_rawname);
				err = 1;
			}
			else if (name[0] == '/') {
				/* skip */;
			}
			else if (quick) {
				copy = 1;
			}
			else if (!argc || !(name = name_from_table(&argc, argv, name))) {
				copy = 1;
			}
			else if (stat(name, &st) == -1) {
				file_error(name, "stat: %s", strerror(errno));
				err = 1;
			}
			else if (do_update && st.st_mtime < ah->ar_date) {
				copy = 1;
			}
			else if (ar_add_file(ap, ah->ar_name, name)) {
				err = 1;
			}
			else if (verbose) {
				printf("r - %s\n", name);
			}
			if (copy && ar_add_elf(ap, elf, *ah)) {
				file_error(ah->ar_name, "could not copy file");
				err = 1;
			}
			cmd = elf_next(elf);
			elf_end(elf);
		}
		if (!err && (err = elf_errno())) {
			file_error(arch, "elf_begin: %s", elf_errmsg(err));
		}
		elf_end(arf);
		close(fd);
	}
	for (i = 0; i < argc; i++) {
		name = argv[i];
		if (ar_add_file(ap, ar_basename(name), name)) {
			err = 1;
		}
		else if (verbose) {
			printf("a - %s\n", name);
		}
	}
	if (!err) {
		err = ar_update_archive(ap, arch, do_backup);
	}
	ar_destroy(ap);
	return err != 0;
}

static const char*
name_in_table(int argc, char **argv, const char *name) {
	const char *s;
	int i;

	for (i = 0; i < argc; i++) {
		s = ar_basename(argv[i]);
		if (strcmp(s, name) == 0) {
			return argv[i];
		}
	}
	return NULL;
}

static const char*
name_from_table(int *argc, char **argv, const char *name) {
	const char *s;
	int i;

	for (i = 0; i < *argc; i++) {
		s = ar_basename(argv[i]);
		if (strcmp(s, name) == 0) {
			s = argv[i];
			while (++i < *argc) {
				argv[i - 1] = argv[i];
			}
			--*argc;
			return s;
		}
	}
	return NULL;
}

static int
ro_print(Elf *elf, Elf_Arhdr *ah, const char *name) {
	const char *ptr;
	size_t len;

	if (!(ptr = elf_rawfile(elf, &len))) {
		file_error(ah->ar_name, "elf_rawfile: %s", elf_errmsg(-1));
		return 1;
	}
	if (verbose) {
		printf("\n<%s>\n\n", name);
	}
	if (len) {
		if (fwrite(ptr, 1, len, stdout) != len || fflush(stdout)) {
			file_error(name, "write: %s", strerror(errno));
			return 1;
		}
	}
	return 0;
}

static int
ro_table(Elf *elf, Elf_Arhdr *ah, const char *name) {
	if (verbose) {
		static const char *modestrs[] = {
			"---", "--x", "-w-", "-wx",
			"r--", "r-x", "rw-", "rwx",
		};
#if HAVE_STRFTIME
#if HAVE_STRFTIME_PERCENT_E
		static const char format[] = "%b %e %H:%M %Y";
#else /* HAVE_STRFTIME_PERCENT_E */
		static const char format[] = "%b %d %H:%M %Y";
#endif /* HAVE_STRFTIME_PERCENT_E */
#else /* HAVE_STRFTIME */
		static const char format[] = "%s %2d %02d:%02d %d";
		static const char *mon_names[] = {
			"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
		};
#endif /* HAVE_STRFTIME */
		char date[256];
		struct tm *tm;

		tm = localtime(&ah->ar_date); 
		assert(tm);
#if HAVE_STRFTIME
		strftime(date, sizeof(date), format, tm);
#else /* HAVE_STRFTIME */
		sprintf(date, format, mon_names[tm->tm_mon], tm->tm_mday,
				tm->tm_hour, tm->tm_min, tm->tm_year + 1900);
#endif /* HAVE_STRFTIME */
		printf("%s%s%s %u/%u %6u %s %s\n",
			   modestrs[(ah->ar_mode >> 6) & 7u],
			   modestrs[(ah->ar_mode >> 3) & 7u],
			   modestrs[(ah->ar_mode >> 0) & 7u],
			   (unsigned)ah->ar_uid, (unsigned)ah->ar_gid,
			   (unsigned)ah->ar_size, date, name);
	}
	else {
		printf("%s\n", name);
	}
	return 0;
}

static int
ro_extract(Elf *elf, Elf_Arhdr *ah, const char *name) {
	const char *ptr;
	size_t len;
	int ofd;

	if (!(ptr = elf_rawfile(elf, &len))) {
		file_error(ah->ar_name, "elf_rawfile: %s", elf_errmsg(-1));
		return 1;
	}
#if defined(O_CREAT) && defined(O_TRUNC)
	if ((ofd = open(name, O_WRONLY | O_CREAT | O_TRUNC, 0600)) == -1) {
#else
	if ((ofd = creat(name, 0600)) == -1) {
#endif
		file_error(name, "open: %s", strerror(errno));
		return 1;
	}
	if (verbose) {
		printf("x - %s\n", name);
	}
	if (len && xwrite(ofd, ptr, len)) {
		close(ofd);
		unlink(name);
		return 1;
	}
	close(ofd);
	chown(name, ah->ar_uid, ah->ar_gid);
	chmod(name, ah->ar_mode & 07777);
#if HAVE_UTIME
	if (do_old_date) {
		struct utimbuf ut;

		ut.modtime = ah->ar_date;
		ut.actime = ah->ar_date;
		utime(name, &ut);
	}
#elif HAVE_UTIMES
	if (do_old_date) {
		struct timeval tv[2];

		tv[0].tv_sec = ah->ar_date;
		tv[0].tv_usec = 0;
		tv[1].tv_sec = ah->ar_date;
		tv[1].tv_usec = 0;
		utimes(name, tv);
	}
#endif /* HAVE_UTIME */
	return 0;
}

static int
ro_nothing(Elf *elf, Elf_Arhdr *ah, const char *name) {
	return 0;
}

static int
do_readonly(int argc, char **argv, ro_func func) {
	Elf *arf;
	Elf *elf;
	Elf_Arhdr *ah;
	Elf_Cmd cmd;
	ardesc_t *ap;
	const char *name;
	int err;
	int fd;

	ap = do_ranlib ? ar_create() : NULL;
	if ((fd = open(arch, ap ? O_RDWR : O_RDONLY)) == -1) {
		file_error(arch, "open: %s", strerror(errno));
		ar_destroy(ap);
		return 1;
	}
	if (!(arf = elf_begin(fd, ELF_C_READ, NULL))) {
		file_error(arch, "elf_begin: %s", elf_errmsg(-1));
		ar_destroy(ap);
		close(fd);
		return 1;
	}
	err = 0;
	elf_errno();
	cmd = ELF_C_READ;
	while (!err && (elf = elf_begin(fd, cmd, arf))) {
		if (!(ah = elf_getarhdr(elf))) {
			file_error(arch, "elf_getarhdr: %s", elf_errmsg(-1));
			err = 1;
		}
		else if (!(name = ah->ar_name)) {
			file_error(arch, "bad member name `%s'", ah->ar_rawname);
			err = 1;
		}
		else if (name[0] == '/') {
			/* skip */;
		}
		else {
			if (!argc || (name = name_in_table(argc, argv, name))) {
				err = func(elf, ah, name);
			}
			if (ap && ar_add_elf(ap, elf, *ah)) {
				file_error(ah->ar_name, "could not copy file");
				err = 1;
			}
		}
		cmd = elf_next(elf);
		elf_end(elf);
	}
	if (!err && (err = elf_errno())) {
		file_error(arch, "elf_begin: %s", elf_errmsg(err));
	}
	elf_end(arf);
	close(fd);
	if (!err && ap) {
		err = ar_update_archive(ap, arch, do_backup);
		ar_destroy(ap);
	}
	return err != 0;
}

static const char usage[] =
	"usage: %s [-V] [-]{dmpqrtx}[aBbcilosuv] [member...] archive [file...]\n";

static void
Usage(const char *progname, int x) {
	fprintf(stderr, usage, progname);
	exit(x);
}

static void
Usage_ranlib(const char *progname, int x) {
	fprintf(stderr, "usage: %s [-V] [file]\n", progname);
	exit(x);
}

int
ranlib_main(const char *progname, int argc, char **argv) {
	int i = 1;
	char *key;

	if (i >= argc) {
		Usage_ranlib(progname, 1);
	}
	key = argv[i];
	if (*key == '-') {
		if (*++key == 'V' && !key[1]) {
			show_version("ranlib");
			exit(0);
		}
		if (key[0] != '-' || key[1]) {
			error("invalid option `%s'", argv[i]);
			Usage_ranlib(progname, 1);
		}
		i++;
		if (i >= argc) {
			exit(0);
		}
	}
	if (i >= argc) {
		error("missing archive name");
		Usage_ranlib(progname, 1);
	}
	if (i + 1 < argc) {
		error("too many arguments");
		Usage_ranlib(progname, 1);
	}
	do_ranlib = 1;
	arch = argv[i++];
	if (elf_version(EV_CURRENT) == EV_NONE) {
		error("libelf version mismatch");
		exit(1);
	}
	i = do_readonly(argc - i, &argv[i], ro_nothing);
	exit(i);
}

int
main(int argc, char **argv) {
	const char *progname;
	int operation = 0;
	int i = 1;
	char *key;
	int c;
	size_t len;

	progname = argv[0] ? ar_basename(argv[0]) : "ar";
	setprogname(progname);

	if (argv[0]
	 && (len = strlen(argv[0])) >= 6
	 && strcmp(argv[0] + len - 6, "ranlib") == 0) {
		ranlib_main(progname, argc, argv);
		/* never returns */
	}

	if (i >= argc) {
		Usage(progname, 1);
	}
	key = argv[i++];
	if (*key == '-') {
		if (*++key == 'V' && !key[1]) {
			show_version("ar");
			exit(0);
		}
	}
	if (*key == '\0' || !strchr("dmpqrtx", *key)) {
		Usage(progname, 1);
	}
	operation = *key++;
	while (*key) {
		switch ((c = *key++)) {
			case 'i':
				c = 'b';
				/* flow through */
			case 'a':
			case 'b':
				if (i >= argc) {
					error("`%c' requires an argument", c);
					Usage(progname, 1);
				}
				if (operation != 'r' && operation != 'm') {
					error("`%c' not allowed with `%c'", c, operation);
					Usage(progname, 1);
				}
				if (where) {
					error("multiple `a', `b' or `i' modifiers");
					Usage(progname, 1);
				}
				where = argv[i++];
				before = c;
				break;
			case 'u':
				if (operation != 'r') {
					error("`%c' not allowed with `%c'", c, operation);
					Usage(progname, 1);
				}
				do_update = 1;
				break;
			case 'c': do_create = 1; break;
			case 'l': break;
#if HAVE_UTIME || HAVE_UTIMES
			case 'o': do_old_date = 1; break;
#endif /* HAVE_UTIME || HAVE_UTIMES */
			case 's': do_ranlib = 1; break;
			case 'v': verbose = 1; break;
			case 'B': do_backup = 1; break;
			default: Usage(progname, 1);
		}
	}
	if (i >= argc) {
		error("missing archive name");
		Usage(progname, 1);
	}
	arch = argv[i++];
	if (where && i < argc && name_in_table(argc, argv, where)) {
		static const char *arr[] = { "after", "before" };

		error("cannot insert %s %s itself", where, arr[before == 'b']);
		exit(1);
	}
	if (elf_version(EV_CURRENT) == EV_NONE) {
		error("libelf version mismatch");
		exit(1);
	}
	switch (operation) {
		case 'd': i = do_delete(argc - i, &argv[i]); break;
		case 'm': i = do_move(argc - i, &argv[i]); break;
		case 'q': i = do_replace(argc - i, &argv[i], 1); break;
		case 'r': i = do_replace(argc - i, &argv[i], 0); break;
		case 'p': i = do_readonly(argc - i, &argv[i], ro_print); break;
		case 't': i = do_readonly(argc - i, &argv[i], ro_table); break;
		case 'x': i = do_readonly(argc - i, &argv[i], ro_extract); break;
		default: assert(0);
	}
	exit(i);
}
