/*
 * ldmap.c -- section-to-segment mapping
 * Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

static const char rcsid[] = "@(#) $Id: ldmap.c,v 1.11 2003/02/07 16:19:33 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gelf.h>
#include <ar.h>
#include <assert.h>

#include <ld/ld.h>
#include <ld/ldmisc.h>

static struct segment defsegs[] = {
	/* next, scns, name, phdr */
	{ &defsegs[1], NULL, "text",
		{ .p_type = PT_LOAD, .p_flags = PF_R | PF_X }, },
	{ &defsegs[2], NULL, "data",
		{ .p_type = PT_LOAD, .p_flags = PF_R | PF_W | PF_X }, },
	{ &defsegs[3], NULL, "note",
		{ .p_type = PT_NOTE, }, },
	{ NULL,        NULL, NULL,
		{ .p_type = PT_NULL, }, },
};

struct segment *const text_segment = &defsegs[0];
struct segment *const data_segment = &defsegs[1];
struct segment *const note_segment = &defsegs[2];

static struct scnmap defmaps[] = {
	/* next, seg, name, type, flags, fmask, files, nfiles */
	{ &defmaps[1], &defsegs[0], NULL, SHT_NULL, SHF_ALLOC,
		SHF_ALLOC | SHF_WRITE, NULL, 0 },
	{ &defmaps[2], &defsegs[1], NULL, SHT_NULL, SHF_ALLOC | SHF_WRITE,
		SHF_ALLOC | SHF_WRITE, NULL, 0 },
	{ &defmaps[3], &defsegs[2], NULL, SHT_NOTE, 0, 0, NULL, 0 },
	/* catch-all */
	{ NULL,        &defsegs[3], NULL, SHT_NULL, 0, 0, NULL, 0 },
};

struct segment *segments = &defsegs[0];
struct scnmap *section_map = &defmaps[0];

struct segment*
ld_map_section(const char *fn, const char *name, const GElf_Shdr *sh) {
	GElf_Word type;
	struct scnmap *map;
	const char *base;
	size_t i;

	assert(fn);
	assert(name);
	assert(sh);

	if ((base = strrchr(fn, '/'))) {
		base++;
	}
	else {
		base = fn;
	}

	type = sh->sh_type;
	if (type >= SHT_LOUSER && type <= SHT_HIUSER) {
		type = SHT_PROGBITS;
	}

	for (map = section_map; map; map = map->next) {
		if (map->type != SHT_NULL && type != map->type) continue;
		if ((sh->sh_flags ^ map->flags) & map->fmask) continue;
		if (map->name && strcmp(name, map->name) != 0) continue;
		if (!map->nfiles) return map->seg;
		for (i = 0; i < map->nfiles; i++) {
			const char *fname = map->files[i];

			if (*fname == '*') {
				if (strcmp(base, fname + 1) == 0) return map->seg;
			}
			else {
				if (strcmp(fn, fname) == 0) return map->seg;
			}
		}
	}
	fatal("segment catch-all failed");
	/* never reached */
	return NULL;
}

struct outscn*
ld_output_section(const char *fn, const char *name, const GElf_Shdr *sh) {
	struct segment *seg = ld_map_section(fn, name, sh);
	GElf_Word type;
	int state;
	struct outscn *sect, **p;

	type = sh->sh_type;
	if (type >= SHT_LOUSER && type <= SHT_HIUSER) {
		/* Treat user-defined sections like PROGBITS */
		type = SHT_PROGBITS;
	}

	/*
	 * Search existing sections
	 */
	assert(seg);
#ifndef NDEBUG
	if (seg->name)
		debug("* %s : %s(%s)", seg->name, name, fn);
	else
		debug("* (append) : %s(%s)", name, fn);
#endif
	state = 0;
	for (p = &seg->scns; (sect = *p); p = &sect->next) {
		if (type == sect->shdr.sh_type) {
			/*
			 * Type match
			 */
			if (sh->sh_flags == sect->shdr.sh_flags
			 && strcmp(name, sect->name) == 0) {
				/*
				 * Exact match
				 */
				return sect;
			}
			state = 1;
		}
		else if (state) {
			/*
			 * Type does no langer match.
			 * Insert new section here.
			 */
			break;
		}
		else if (sect->shdr.sh_type == SHT_NOBITS) {
			/*
			 * Insert before final NOBITS sections, if any
			 */
			break;
		}
	}
	/*
	 * Create new section
	 */
	sect = xmalloc(sizeof(struct outscn));
	sect->next = *p;
	sect->name = xstrdup(name);
	sect->shdr = *sh;
	sect->shdr.sh_size = 0;
	sect->shdr.sh_addralign = 1;
	sect->buf = NULL;
	sect->rela = NULL;
	sect->rel = NULL;
	sect->nrela = 0;
	sect->nrel = 0;
	sect->scn = NULL;
	*p = sect;
	return sect;
}

struct outscn*
ld_dummy_section(const char *name, GElf_Word type, GElf_Xword flags) {
	struct outscn *sect;

	assert(name);
	/*
	 * Create new section
	 */
	sect = xmalloc(sizeof(struct outscn));
	sect->next = NULL;
	sect->name = xstrdup(name);
	sect->shdr.sh_type = type;
	sect->shdr.sh_flags = flags;
	sect->shdr.sh_size = 0;
	sect->shdr.sh_addralign = 1;
	sect->buf = NULL;
	sect->rela = NULL;
	sect->rel = NULL;
	sect->nrela = 0;
	sect->nrel = 0;
	sect->scn = NULL;
	return sect;
}
