/*
 * ldmain.c -- ld(1) main program
 * Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

static const char rcsid[] = "@(#) $Id: ldmain.c,v 1.36 2003/02/08 18:27:14 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <signal.h>
#include <gelf.h>
#include <assert.h>

#include <ld/ld.h>
#include <ld/ldmisc.h>

Elf *aelf = NULL;
GElf_Ehdr aeh = {{0,},};
char *outfn = "a.out";
char *ld_opt_rpath = NULL;
char ld_opt_dynamic = 0;
char ld_opt_ident = 0;
char ld_opt_ignore_libpath = 0;
char ld_opt_no_size_warn = 0;
char ld_opt_strip = 0;
char ld_opt_weakextract = 0;
char ld_opt_whole_archive = 0;
char ld_opt_zdefs = '\0';
char ld_opt_zmuldefs = '\0';
int ld_target_type = ET_EXEC;

static char **opt_auxfilter = NULL;
static char **opt_filter = NULL;
static char *opt_e = NULL;	/* by default, try `_start' and `main' */
static char *opt_h = NULL;
static char *opt_I = NULL;
static char opt_a = 0;
static char opt_b = 0;
static char opt_Bdynamic = 0;
static char opt_Bsymbolic = 0;
static char opt_G = 0;
static char opt_m = 0;
static char opt_r = 0;
static char opt_V = 0;
static char opt_ztext = '\0';

extern void setprogname(const char *__newname);

#define DEFTARGET(name)	extern void init_target_##name(void);
#include <ld/targets.def>
#undef DEFTARGET

static const char version_string[] =
	"%s (fctools) version " VERSION_STRING "\n"
	"Copyright (C) 2000 - 2003 Michael \"Tired\" Riepe\n"
	"This program is free software; you can redistribute it and/or\n"
	"modify it under the terms of the GNU General Public License.\n"
	"This program is distributed WITHOUT ANY WARRANTY.\n\n"
	;

static const char supported_targets[] =
	"Supported targets:\n"
#define DEFTARGET(name)	"\t" #name "\n"
#include <ld/targets.def>
#undef DEFTARGET
	"\n";

static void
invarg(const char *arg, char opt) {
	error("invalid argument `%s' for option `%c'", arg, opt);
	exit(1);
}

static void
invstatic(const char *str) {
	error("option `%s' invalid in static mode", str);
	exit(1);
}

static void
opt_B(const char *key) {
	switch (key[0]) {
		case 'd':
			if (strcmp(key, "dynamic") == 0) {
				if (ld_opt_dynamic != 'y') {
					invstatic("-B dynamic");
				}
				opt_Bdynamic = 1;
				return;
			}
			break;
		case 's':
			if (strcmp(key, "static") == 0) {
				opt_Bdynamic = 0;
				return;
			}
			if (strcmp(key, "symbolic") == 0) {
				if (ld_opt_dynamic != 'y') {
					invstatic("-B symbolic");
				}
				opt_Bsymbolic = 1;
				return;
			}
			break;
	}
	invarg(key, 'B');
}

static void
opt_d(const char *key) {
	if ((key[0] == 'y' || key[0] == 'n') && key[1] == '\0') {
		ld_opt_dynamic = key[0];
		return;
	}
	invarg(key, 'd');
}

static void
opt_F(const char *key) {
	/* XXX: collect filtees */;
	error("option `-F' not supported yet");
	exit(1);
}

static void
opt_f(const char *key) {
	/* XXX: collect filtees */;
	error("option `-f' not supported yet");
	exit(1);
}

static void
opt_M(const char *map) {
	/* XXX: load mapfile(s)
	if (ld_load_mapfile(map)) {
		exit(1);
	}
	*/
	error("option `-M' not supported yet");
	exit(1);
}

static void
opt_Q(const char *key) {
	if ((key[0] == 'y' || key[0] == 'n') && key[1] == '\0') {
		ld_opt_ident = key[0];
		return;
	}
	invarg(key, 'Q');
}

static void
opt_R(const char *key) {
	if (ld_opt_rpath) {
		size_t len = strlen(ld_opt_rpath);

		ld_opt_rpath = xrealloc(ld_opt_rpath, len + strlen(key) + 2);
		ld_opt_rpath[len++] = ':';
		strcpy(ld_opt_rpath + len, key);
	}
	else {
		ld_opt_rpath = xstrdup(key);
	}
}

static void
opt_z(const char *key) {
	switch (key[0]) {
		case 'a':
			if (strcmp(key, "allextract") == 0) {
				ld_opt_whole_archive = 1;
				ld_opt_weakextract = 0;
				return;
			}
			break;
		case 'd':
			if (strcmp(key, "defaultextract") == 0) {
				ld_opt_whole_archive = 0;
				ld_opt_weakextract = 0;
				return;
			}
			if (strcmp(key, "defs") == 0) {
				ld_opt_zdefs = 'y';
				return;
			}
			break;
		case 'm':
			if (strcmp(key, "muldefs") == 0) {
				ld_opt_zmuldefs = 1;
				return;
			}
			break;
		case 'n':
			if (strcmp(key, "nodefs") == 0) {
				ld_opt_zdefs = 'n';
				return;
			}
			break;
		case 't':
			if (strcmp(key, "text") == 0) {
				opt_ztext = 'y';
				return;
			}
			if (strcmp(key, "textoff") == 0) {
				opt_ztext = 'n';
				return;
			}
			if (strcmp(key, "textwarn") == 0) {
				opt_ztext = 'w';
				return;
			}
			break;
		case 'w':
			if (strcmp(key, "weakextract") == 0) {
				ld_opt_whole_archive = 0;
				ld_opt_weakextract = 1;
				return;
			}
			break;
	}
	invarg(key, 'z');
}

int
opt_arg(char **opt, int argc, char **argv, int i) {
	char c = argv[i][1];

	assert(opt);
	if (argv[i][2]) {
		*opt = &argv[i][2];
		return i;
	}
	if (++i < argc) {
		*opt = argv[i];
		return i;
	}
	error("option `%c' requires an argument", c);
	exit(1);
}

void
cleanup(int status) {
	unlink(outfn);
	exit(status);
}

static void
sighand(int sig) {
	cleanup(1);
}

int
main(int argc, char **argv) {
	char *progname;
	char *s;
	int c;
	int err;
	int fd;
	int i;
	int j;
	int n;
	int files;

	if (*argv && (progname = strrchr(*argv, '/'))) {
		progname++;
	}
	else if (!(progname = *argv)) {
		progname = "ld";
	}
	setprogname(progname);

	if (elf_version(EV_CURRENT) == EV_NONE) {
		fatal("elf_version: %s", elf_errmsg(-1));
		exit(1);
	}

	if (signal(SIGINT, SIG_IGN) != SIG_IGN) {
		signal(SIGINT, sighand);
	}
	if (signal(SIGTERM, SIG_IGN) != SIG_IGN) {
		signal(SIGTERM, sighand);
	}

	ld_add_string(&astrtab, NULL);

	/*
	 * process environment
	 */
	if ((s = getenv("LD_OPTIONS")) && *s) {
		static const char IFS[] = " \t\n";
		char **tmpv;

		s = xstrdup(s);
		i = strspn(s, IFS);
		for (j = argc + 1; s[i]; j++) {
			i += strcspn(&s[i], IFS);
			i += strspn(&s[i], IFS);
		}
		tmpv = xmalloc(j * sizeof(char*));
		tmpv[0] = argv[0];
		for (j = 1, s = strtok(s, IFS); s; s = strtok(NULL, IFS)) {
			tmpv[j++] = s;
		}
		for (i = 1; i < argc; i++) {
			tmpv[j++] = argv[i];
		}
		tmpv[j] = NULL;
		argc = j;
		argv = tmpv;
	}

	/*
	 * process command line options
	 */
	files = 0;
	for (i = 1, j = 0; i < argc; i++) {
		if (argv[i][0] != '-') {
			argv[j++] = argv[i];
			files++;
			continue;
		}
		switch ((c = argv[i][1])) {
			case '-':	/* -- */
				if (argv[i][2]) {
					error("invalid option `%s'", argv[i]);
					exit(1);
				}
				while (++i < argc) {
					argv[j++] = argv[i];
				}
				continue;
			case 'l':	/* -l<libname> */
				files++;
				/* flow through */
			case 'B':	/* -Bdynamic|-Bstatic|-Bsymbolic */
			case 'N':	/* -N<needed> */
			case 'L':	/* -L<dir> */
			case 'u':	/* -u<symbol> */
				if (argv[i][2]) {
					argv[j++] = argv[i];
					continue;
				}
				if (++i < argc) {
					s = xmalloc(3 + strlen(argv[i]));
					s[0] = '-';
					s[1] = c;
					strcpy(s + 2, argv[i]);
					argv[j++] = s;
					continue;
				}
				error("option `%c' requires an argument", c);
				exit(1);
			case 'd':
				i = opt_arg(&s, argc, argv, i);
				opt_d(s);
				continue;
			case 'e':	/* entry point is <optarg> */
				i = opt_arg(&opt_e, argc, argv, i);
				continue;
			case 'F':	/* create filter for <optarg> */
				i = opt_arg(&s, argc, argv, i);
				opt_F(s);
				continue;
			case 'f':	/* create auxiliary filter for <optarg> */
				i = opt_arg(&s, argc, argv, i);
				opt_f(s);
				continue;
			case 'h':	/* SONAME is <optarg> */
				i = opt_arg(&opt_h, argc, argv, i);
				continue;
			case 'I':	/* ELF interpreter is <optarg> */
				i = opt_arg(&opt_I, argc, argv, i);
				continue;
			case 'M':	/* use mapfile <optarg> */
				/* may be used multiple times, or may be a directory */
				i = opt_arg(&s, argc, argv, i);
				opt_M(s);
				continue;
			case 'o':
				i = opt_arg(&outfn, argc, argv, i);
				continue;
			case 'Q':	/* add ident to .comment or not */
				i = opt_arg(&s, argc, argv, i);
				opt_Q(s);
				continue;
			case 'R':	/* run-time linker search path */
				i = opt_arg(&s, argc, argv, i);
				opt_R(s);
				continue;
			case 'z':	/* defs | nodefs | text* */
				i = opt_arg(&s, argc, argv, i);
				opt_z(s);
				continue;
			case 'D':	/* debugging tokens */
				i = opt_arg(&s, argc, argv, i);
				error("option `-%c %s' not supported", c, s);
				exit(1);
			case 'Y':	/* change default library path */
				i = opt_arg(&s, argc, argv, i);
				if (s[0] == 'P' && s[1] == ',') {
					ld_set_deflibpath(s + 2);
					continue;
				}
				error("option `-%c %s' not supported", c, s);
				exit(1);
		}
		for (n = 1; (c = argv[i][n]); n++) {
			switch (c) {
				case 'a':	/* produce ET_EXEC (static) */
					opt_a = 1; continue;
				case 'b':	/* no copy relocations, no PLT entries for undefined functions? */
					opt_b = 1; continue;
				case 'G':	/* produce ET_DYN */
					opt_G = 1; continue;
				case 'i':	/* ignore LD_LIBRARY_PATH */
					ld_opt_ignore_libpath = 1; continue;
				case 'm':	/* produce a link map */
					opt_m = 1; continue;
				case 'r':	/* produce ET_REL */
					opt_r = 1; continue;
				case 's':	/* strip symbol tables */
					ld_opt_strip = 1; continue;
				case 't':
					ld_opt_no_size_warn = 1; continue;
				case 'V':	/* print version to stderr */
					opt_V = 1; continue;
				default:
					error("unrecognized option `%c'", c); exit(1);
			}
		}
	}
	argc = j;
	argv[argc] = NULL;

	/* XXX: dynamic mode disabled for now */
	if (ld_opt_dynamic == 'y') {
		error("dynamic linking not supported (yet)");
		exit(1);
	}
	ld_opt_dynamic = 'n';

	/*
	 * option sanity check
	 */
	if (!ld_opt_dynamic) {
		ld_opt_dynamic = (opt_a || opt_r) ? 'n' : 'y';
	}
	if (ld_opt_dynamic == 'y') {
		if (opt_a || opt_r) {
			error("options `-a' and `-r' are invalid in dynamic mode");
			exit(1);
		}
		if (opt_G) {
			ld_target_type = ET_DYN;
		}
		else if (opt_filter || opt_auxfilter || opt_h) {
			error("options `-F', `-f' and `-h' also need `-G'");
			exit(1);
		}
		opt_Bdynamic = 1;
	}
	else {
		if (opt_G) {
			invstatic("-G");
		}
		if (opt_a && opt_r) {
			error("options `-a' and `-r' are mutual exclusive");
			exit(1);
		}
		opt_a = !opt_r;
		if (opt_b) {
			invstatic("-b");
		}
		if (opt_r) {
			if (opt_I) {
				error("options `-r' and `-I' used together");
				exit(1);
			}
			if (opt_e) {
				error("options `-r' and `-e' used together");
				exit(1);
			}
			ld_target_type = ET_REL;
		}
	}

	if (opt_V) {
		fprintf(stderr, version_string, progname);
		fprintf(stderr, supported_targets);
		if (files == 0) {
			exit(0);
		}
	}

	if (files == 0) {
		error("no input files");
		exit(1);
	}

	ld_init_libpath();

#define DEFTARGET(name)	init_target_##name();
#include <ld/targets.def>
#undef DEFTARGET

	/*
	 * Create output file
	 */
	unlink(outfn);
	if ((fd = open(outfn, O_WRONLY | O_CREAT | O_TRUNC, 0600)) == -1) {
		perror(outfn);
		exit(1);
	}
	if (!(aelf = elf_begin(fd, ELF_C_WRITE, NULL))) {
		file_error(outfn, "elf_begin: %s", elf_errmsg(-1));
		cleanup(1);
	}

	/*
	 * Pass 1: process arguments
	 */
	err = 0;
	for (i = 0; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				case 'B':
					opt_B(&argv[i][2]); continue;
				case 'L':
					ld_add_libdir(&argv[i][2]); continue;
				case 'u':
					ld_undef_symbol(&argv[i][2]); continue;
				case 'l':
					break;
				default:
					warn("ignoring option `%s'", argv[i]);
					continue;
			}
		}
		err |= ld_process_file(argv[i], opt_Bdynamic);
	}
	/*
	 * Can't do anything without a selected target
	 */
	if (!err && !target) {
		error("no object files loaded");
		cleanup(1);
	}

	/*
	 * Pass 2: link input files
	 */
	if (err || errors || ld_output(opt_e)) {
		cleanup(1);
	}

	/*
	 * Finally, write the file
	 */
	if (elf_update(aelf, ELF_C_WRITE) == (off_t)-1) {
		file_error(outfn, "elf_update: %s", elf_errmsg(-1));
		cleanup(1);
	}
	elf_end(aelf);
	close(fd);
	if (ld_target_type == ET_REL) {
		chmod(outfn, 0666 &~ umask(0));
	}
	else {
		chmod(outfn, 0777 &~ umask(0));
	}
	exit(0);
}
