/*
 * ldfile.c -- file-based ld(1) operations
 * Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

static const char rcsid[] = "@(#) $Id: ldfile.c,v 1.10 2003/02/05 15:11:01 michael Exp $";

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <gelf.h>
#include <ar.h>
#include <assert.h>

#include <ld/ld.h>
#include <ld/ldmisc.h>

#define PRIME  263u

static int
ld_add_member(const char *fn, size_t noff, Elf *elf, int all) {
	const size_t blocksize = 1024;
	static char *member = NULL;
	static size_t nmemb = 0;
	Elf_Arhdr *ah;
	char *name;
	size_t len;

	if (!(ah = elf_getarhdr(elf))) {
		file_error(fn, "elf_getarhdr: %s", elf_errmsg(-1));
		return -1;
	}
	if (!(name = ah->ar_name) || !*name) {
		file_error(fn, "bad archive member name");
		return -1;
	}
	if (*name == '/') {
		if (all) {
			return 0;
		}
		file_error(fn, "bad archive member name");
		return -1;
	}
	len = strlen(fn) + strlen(name) + 3;
	if (nmemb < len) {
		nmemb = len + blocksize - 1;
		nmemb = nmemb - nmemb % blocksize;
		member = xrealloc(member, nmemb);
	}
	strcpy(member, fn);
	strcat(member, "(");
	strcat(member, name);
	strcat(member, ")");
	return ld_add_elf_file(member, noff, elf);
}

static int
process_ar(const char *fn, size_t noff, int fd, Elf *arf, int all) {
	Elf *elf;
	int err;

	debug("file=%s	[ archive ]", fn);
	if (all) {
		Elf_Cmd cmd;

		/*
		 * Load all members
		 */
		elf_errno();
		cmd = ELF_C_READ;
		while ((elf = elf_begin(fd, cmd, arf))) {
			err = ld_add_member(fn, noff, elf, all);
			cmd = elf_next(elf);
			elf_end(elf);
			if (err) {
				return -1;
			}
		}
		/*
		 * Check for errors
		 */
		if ((err = elf_errno())) {
			file_error(fn, "elf_begin: %s", elf_errmsg(err));
			return -1;
		}
	}
	else {
		static GElf_Word *chain = NULL;
		GElf_Word buckets[PRIME] = { 0, };
		Elf_Arsym *arsym;
		GElf_Sym *gsym;
		char *name;
		size_t i;
		size_t j;
		size_t narsym;
		unsigned long hash;

		/*
		 * Check symbol table
		 */
		if (!(arsym = elf_getarsym(arf, &narsym)) || !narsym) {
			file_error(fn, "archive has no symbol table");
			return -1;
		}
		if (narsym == 1) {
			/* XXX: is this an error? */
			file_warn(fn, "empty archive symbol table");
			return 0;
		}
		/*
		 * Build a temporary hash table for faster search.
		 * Note: `0' indicates end-of-chain and must not be used
		 * as an index.
		 */
		chain = xrealloc(chain, narsym * sizeof(*chain));
		for (i = 1; i < narsym; i++) {
			hash = arsym[i - 1].as_hash % PRIME;
			chain[i] = buckets[hash];
			buckets[hash] = i;
		}

		/*
		 * Try to resolve all undefined globals.
		 *
		 * Symbols from the newly added files are added at the
		 * end of the global symbol table; that is, ld will get
		 * a chance to resolve them from the archive as well.
		 * An explicit rescan is not necessary.
		 */
		for (i = 1; i < anglobs; i++) {
			gsym = &aglobals[i].sym;
			if (gsym->st_shndx != SHN_UNDEF) {
				continue;
			}
			switch (GELF_ST_BIND(gsym->st_info)) {
				case STB_LOCAL:
					fatal("local symbol in global symbol table");
				case STB_GLOBAL:
					break;
				case STB_WEAK:
					if (!ld_opt_weakextract) {
						continue;
					}
					break;
				default:
					fatal("unrecognized symbol binding %#x",
						(unsigned)GELF_ST_BIND(gsym->st_info));
			}
			name = ld_symbol_name(gsym->st_name);
			hash = aglobals[i].hash;
			debug("trying to resolve `%s' (%u)", name, (unsigned)i);
			for (j = buckets[hash % PRIME]; j; j = chain[j]) {
				assert(j < narsym);
				if (hash == arsym[j - 1].as_hash
				 && strcmp(name, arsym[j - 1].as_name) == 0) {
					break;
				}
			}
			if (j == 0) {
				/*
				 * Not found, try next undefined symbol
				 */
				continue;
			}
			/*
			 * Symbol found, load member
			 */
			debug("found `%s' in library", name);
			if (elf_rand(arf, arsym[j - 1].as_off) == 0) {
				file_error(fn, "elf_rand: %s", elf_errmsg(-1));
				return -1;
			}
			if (!(elf = elf_begin(fd, ELF_C_READ, arf))) {
				file_error(fn, "elf_begin: %s", elf_errmsg(-1));
				return -1;
			}
			err = ld_add_member(fn, noff, elf, all);
			elf_end(elf);
			if (err) {
				return -1;
			}
			assert(aglobals[i].sym.st_shndx != SHN_UNDEF);
		}
	}
	return 0;
}

static int
process_file(const char *fn, size_t noff, int fd) {
	Elf *elf;
	int err;

	if (!(elf = elf_begin(fd, ELF_C_READ, NULL))) {
		file_error(fn, "elf_begin: %s", elf_errmsg(-1));
		return -1;
	}
	if (elf_kind(elf) == ELF_K_AR) {
		err = process_ar(fn, noff, fd, elf, ld_opt_whole_archive);
	}
	else {
		err = ld_add_elf_file(fn, noff, elf);
	}
	elf_end(elf);
	return err;
}

static char *libpath = NULL;	/* first part of LD_LIBRARY_PATH, plus -L */
static char *lib2path = NULL;	/* second part of LD_LIBRARY_PATH */
static size_t nlibpath = 0;		/* copy of strlen(libpath) */

static const char *deflibpath = "/lib:/usr/lib";

void
ld_set_deflibpath(const char *s) {
	assert(s);
	deflibpath = s;
}

void
ld_init_libpath(void) {
	char *s, *t;

	if (ld_opt_ignore_libpath) {
		return;
	}
	if (!(s = getenv("LD_LIBRARY_PATH"))) {
		return;
	}
	s = xstrdup(s);
	if ((t = strchr(s, ';'))) {
		libpath = s;
		nlibpath = t - s;
		*t++ = '\0';
		lib2path = xstrdup(t);
	}
	else {
		lib2path = s;
	}
}

void
ld_add_libdir(const char *dir) {
	const size_t blocksize = 1024;
	size_t len = strlen(dir);
	size_t n;

	n = nlibpath + len + 2 + blocksize - 1;
	if ((nlibpath + blocksize - 1) / blocksize < n / blocksize) {
		libpath = xrealloc(libpath, n - n % blocksize);
	}
	if (nlibpath) {
		libpath[nlibpath++] = ':';
	}
	strcpy(libpath + nlibpath, dir);
	nlibpath += len;
}

int
ld_process_file(const char *name, int solibs_too) {
	const size_t blocksize = 1024;
	static char *file = NULL;
	static size_t flen = 0;
	const char *path;
	int err;
	int fd;
	size_t i;
	size_t j;
	size_t len;

	if (name[0] != '-') {
		debug("trying to open %s", name);
		if ((fd = open(name, O_RDONLY)) != -1) {
			err = process_file(name, 0, fd);
			close(fd);
			return err;
		}
		file_error(name, "open: %s", strerror(errno));
		return -1;
	}
	assert(name[1] == 'l');
	/*
	 * Find library
	 */
	len = strlen(name + 2);
	if (!(path = libpath) && !(path = lib2path)) {
		path = deflibpath;
	}
	for (;;) {
		assert(path);
		i = strspn(path, ":");
		while ((j = strcspn(path + i, ":"))) {
			if (flen < j + len + 8) {
				flen = j + len + 8 + blocksize - 1;
				flen -= flen % blocksize;
				file = xrealloc(file, flen);
			}
			memcpy(file, path + i, j);
			memcpy(file + j, "/lib", 4);
			memcpy(file + j + 4, name + 2, len);
			if (solibs_too) {
				strcpy(file + j + len + 4, ".so");
				debug("trying to open %s", file);
				if ((fd = open(file, O_RDONLY)) != -1) {
					err = process_file(file, j + 1, fd);
					close(fd);
					return err;
				}
			}
			strcpy(file + j + len + 4, ".a");
			debug("trying to open %s", file);
			if ((fd = open(file, O_RDONLY)) != -1) {
				err = process_file(file, j + 1, fd);
				close(fd);
				return err;
			}
			i += j + strspn(path + i + j, ":");
		}
		if (path == deflibpath) {
			file_error(name, "open: %s", strerror(errno));
			return -1;
		}
		if (path == lib2path || (path == libpath && !(path = lib2path))) {
			path = deflibpath;
		}
	}
}
