/*
 * ld.c -- linking routines
 * Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

static const char rcsid[] = "@(#) $Id: ld.c,v 1.25 2003/02/08 06:25:17 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gelf.h>
#include <ar.h>
#include <assert.h>

#include <ld/ld.h>
#include <ld/ldmisc.h>

struct current cur = { NULL, };

static size_t
number_of_elems(GElf_Shdr *sh, Elf_Type type) {
	size_t n = gelf_fsize(cur.elf, type, 1, cur.eh.e_version);

	assert(n);
	if (sh->sh_entsize != n) {
		file_warn(cur.fn, "section `%s' has wrong sh_entsize",
			cur.names + sh->sh_name);
	}
	if (sh->sh_size % n != 0) {
		file_error(cur.fn, "corrupt data in section `%s'",
			cur.names + sh->sh_name);
		return 0;
	}
	return sh->sh_size / n;
}

int
ld_copy_data_sections(void) {
	Elf_Data *data;
	GElf_Shdr *sh;
	const char *name;
	size_t align;
	size_t i;
	size_t off;
	struct outscn *sect;

	debug("ld_copy_data_sections");

	for (i = 1; i < cur.nscns; i++) {
		sh = &cur.scns[i].sh;
		name = cur.names + sh->sh_name;
		if (sh->sh_type == SHT_REL || sh->sh_type == SHT_RELA) {
			continue;
		}
		if (!(cur.scns[i].flags & LDSF_NEEDED)) {
			debug("ignoring section `%s' (%u)", name, (unsigned)i);
			continue;
		}
		/*
		 * Copy section
		 */
		sect = ld_output_section(cur.fn, name, sh);
		cur.scns[i].out = sect;
		assert(sect);
		if (!(data = elf_rawdata(cur.scns[i].scn, NULL))) {
			file_error(cur.fn, "elf_rawdata: %s", elf_errmsg(-1));
			return -1;
		}
		/*
		 * Calculate offset for new data
		 */
		off = sect->shdr.sh_size;
		align = sh->sh_addralign;
		if (sect->shdr.sh_addralign < align) {
			sect->shdr.sh_addralign = align;
		}
		if (align > 1 && off % align) {
			off += align - off % align;
		}
		/*
		 * Copy data
		 */
		if (sh->sh_type != SHT_NOBITS) {
			sect->buf = xrealloc(sect->buf, off + data->d_size);
			memcpy(sect->buf + off, data->d_buf, data->d_size);
		}
		sect->shdr.sh_size = off + data->d_size;
		/*
		 * Remember offset in section
		 */
		cur.scns[i].off = off;
	}
	return 0;
}

int
ld_scan_file(const char *fn, size_t noff, Elf *elf) {
	size_t isym = 0;
	size_t idyn = 0;
	size_t ishndx = 0;
	Elf_Data *data;
	GElf_Shdr *sh;
	size_t i;
	size_t j;

	cur.fn = xrealloc(cur.fn, strlen(fn) + 1);
	strcpy(cur.fn, fn);
	cur.elf = elf;
	if (!gelf_getehdr(elf, &cur.eh)) {
		file_error(fn, "gelf_getehdr: %s", elf_errmsg(-1));
		return -1;
	}
	if (ld_file_type_match()) {
		return -1;
	}
	if (cur.eh.e_shoff == 0) {
		file_warn(fn, "file does not contain any sections");
		return 0;
	}
	cur.nscns = cur.eh.e_shnum;
	cur.shstrndx = cur.eh.e_shstrndx;
	if (cur.nscns == 0) {
		GElf_Shdr shdr;

		if (!gelf_getshdr(elf_getscn(elf, 0), &shdr)) {
			file_error(fn, "gelf_getshdr: %s", elf_errmsg(-1));
			return -1;
		}
		cur.nscns = shdr.sh_size;
		if (cur.nscns < SHN_LORESERVE) {
			file_error(fn, "bad e_shnum value");
			return -1;
		}
		if (cur.shstrndx == SHN_XINDEX) {
			cur.shstrndx = shdr.sh_link;
			if (cur.shstrndx < SHN_LORESERVE) {
				file_error(fn, "bad e_shstrndx value");
				return -1;
			}
		}
	}
	/*
	 * Find symbol table section
	 */
	cur.scns = xrealloc(cur.scns, cur.nscns * sizeof(*cur.scns));
	for (i = 1; i < cur.nscns; i++) {
		cur.scns[i].flags = 0;
		if (!(cur.scns[i].scn = elf_getscn(elf, i))) {
			file_error(fn, "elf_getscn: %s", elf_errmsg(-1));
			return -1;
		}
		if (!gelf_getshdr(cur.scns[i].scn, &cur.scns[i].sh)) {
			file_error(fn, "gelf_getshdr: %s", elf_errmsg(-1));
			return -1;
		}
		switch (cur.scns[i].sh.sh_type) {
			case SHT_SYMTAB:
				if (isym && cur.scns[isym].sh.sh_type == SHT_SYMTAB) {
					file_error(fn, "multiple symbol tables found");
					return -1;
				}
				isym = i;
				break;
			case SHT_SYMTAB_SHNDX:
				j = cur.scns[i].sh.sh_link;
				if (cur.scns[j].sh.sh_type != SHT_SYMTAB) {
					file_error(fn, "SYMTAB_SHNDX link does not point to a symbol table");
					return -1;
				}
				ishndx = i;
				break;
			case SHT_DYNSYM:
				if (cur.eh.e_type == ET_REL) {
					file_error(fn, "dynsym section found in relocatable file");
					return -1;
				}
				assert(cur.eh.e_type == ET_DYN);
				if (isym == 0) {
					isym = i;
				}
				break;
			case SHT_DYNAMIC:
				if (cur.eh.e_type == ET_REL) {
					file_error(fn, "dynamic section found in relocatable file");
					return -1;
				}
				if (idyn) {
					file_error(fn, "multiple dynamic sections found");
					return -1;
				}
				idyn = i;
				break;
		}
		cur.scns[i].off = 0;
		cur.scns[i].out = NULL;
	}
	if (!isym) {
		file_error(fn, "no symbol table");
		return -1;
	}
	assert(!ishndx || cur.scns[ishndx].sh.sh_link == isym);
	/*
	 * Copy symbol table
	 */
	sh = &cur.scns[isym].sh;
	cur.nsyms = number_of_elems(sh, ELF_T_SYM);
	cur.lsyms = sh->sh_info;
	if (cur.nsyms == 0 || cur.nsyms < cur.lsyms) {
		file_error(fn, "corrupt symbol table");
		return -1;
	}
	cur.syms = xrealloc(cur.syms, cur.nsyms * sizeof(GElf_Sym));
	if (!(data = elf_getdata(cur.scns[isym].scn, NULL))) {
		file_error(fn, "elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	for (i = 0; i < cur.nsyms; i++) {
		if (!gelf_getsym(data, i, &cur.syms[i])) {
			file_error(fn, "gelf_getsym: %s", elf_errmsg(-1));
			return -1;
		}
	}
	/*
	 * Copy section indices, if any
	 */
	assert(sizeof(GElf_Word) == sizeof(Elf32_Word));
	assert(sizeof(GElf_Word) == sizeof(Elf64_Word));
	cur.shndx = xrealloc(cur.shndx, cur.nsyms * sizeof(GElf_Word));
	if (ishndx) {
		if (!(data = elf_getdata(cur.scns[ishndx].scn, NULL))) {
			file_error(fn, "elf_getdata: %s", elf_errmsg(-1));
			return -1;
		}
		if (data->d_type == ELF_T_WORD) {
			/*
			 * Translated by libelf
			 */
			if (data->d_size < cur.nsyms * sizeof(GElf_Word)) {
				file_error(fn, "SYMTAB_SHNDX section is too short");
				return -1;
			}
			memcpy(cur.shndx, data->d_buf, cur.nsyms * sizeof(GElf_Word));
		}
		else if (data->d_type == ELF_T_BYTE) {
			Elf_Data src, dst;

			/*
			 * Untranslated - do it ourselves
			 */
			src = *data;
			src.d_type = ELF_T_WORD;
			dst.d_buf = cur.shndx;
			dst.d_size = cur.nsyms * sizeof(GElf_Word);
			dst.d_version = EV_CURRENT;
			if (!gelf_xlatetom(elf, &dst, &src, cur.eh.e_ident[EI_DATA])) {
				file_error(fn, "cannot translate SYMTAB_SHNDX section");
				return -1;
			}
			if (dst.d_size < cur.nsyms * sizeof(GElf_Word)) {
				file_error(fn, "SYMTAB_SHNDX section is too short");
				return -1;
			}
		}
		else {
			file_error(fn, "cannot translate SYMTAB_SHNDX section");
			return -1;
		}
	}
	else {
		/* just in case */
		memset(cur.shndx, 0, cur.nsyms * sizeof(GElf_Word));
	}
	/*
	 * Copy associated string table
	 */
	isym = sh->sh_link;
	if (isym < 1 || isym >= cur.nscns || cur.scns[isym].sh.sh_type != SHT_STRTAB) {
		file_error(fn, "invalid string table index %#x in section header",
				   (unsigned)isym);
		return -1;
	}
	if (!(data = elf_getdata(cur.scns[isym].scn, NULL))) {
		file_error(fn, "elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	cur.strs = xrealloc(cur.strs, data->d_size + 1);
	memcpy(cur.strs, data->d_buf, data->d_size);
	cur.strs[data->d_size] = '\0';		/* just in case */
	for (i = 0; i < cur.nsyms; i++) {
		if (cur.syms[i].st_name >= data->d_size) {
			file_error(fn, "invalid st_name %#x in symbol table entry %u",
					   (unsigned)cur.syms[i].st_name, (unsigned)i);
			return -1;
		}
	}
	/*
	 * Copy section name table
	 */
	isym = cur.shstrndx;
	if (isym < 1 || isym >= cur.nscns || cur.scns[isym].sh.sh_type != SHT_STRTAB) {
		file_error(fn, "invalid string table index %#x in ELF header",
				   (unsigned)isym);
		return -1;
	}
	if (!(data = elf_getdata(cur.scns[isym].scn, NULL))) {
		file_error(fn, "elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	cur.names = xrealloc(cur.names, data->d_size + 1);
	memcpy(cur.names, data->d_buf, data->d_size);
	cur.names[data->d_size] = '\0';		/* just in case */
	for (i = 1; i < cur.nscns; i++) {
		if (cur.scns[i].sh.sh_name >= data->d_size) {
			file_error(fn, "invalid sh_name %#x in section header %u",
					   (unsigned)cur.scns[i].sh.sh_name, (unsigned)i);
			return -1;
		}
	}
	if (cur.eh.e_type == ET_REL) {
		return 0;
	}
	/*
	 * Copy dynamic section
	 */
	assert(cur.eh.e_type == ET_DYN);
	if (!idyn) {
		file_error(fn, "shared library has no dynamic section");
		return -1;
	}
	sh = &cur.scns[idyn].sh;
	cur.ndyns = number_of_elems(sh, ELF_T_DYN);
	if (cur.ndyns < 1) {
		file_error(fn, "corrupt dynamic section");
		return -1;
	}
	cur.dyns = xrealloc(cur.dyns, cur.ndyns * sizeof(*cur.dyns));
	if (!(data = elf_getdata(cur.scns[idyn].scn, NULL))) {
		file_error(fn, "elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	for (i = 0; i < cur.ndyns; i++) {
		if (!gelf_getdyn(data, i, &cur.dyns[i])) {
			file_error(fn, "gelf_getdyn: %s", elf_errmsg(-1));
			return -1;
		}
	}
	/*
	 * Copy associated string table
	 */
	idyn = sh->sh_link;
	if (idyn < 1 || idyn >= cur.nscns || cur.scns[idyn].sh.sh_type != SHT_STRTAB) {
		file_error(fn, "invalid string table index %#x in section header",
				   (unsigned)idyn);
		return -1;
	}
	if (!(data = elf_getdata(cur.scns[idyn].scn, NULL))) {
		file_error(fn, "elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	cur.dstrs = xrealloc(cur.dstrs, data->d_size + 1);
	memcpy(cur.dstrs, data->d_buf, data->d_size);
	cur.dstrs[data->d_size] = '\0';			/* just in case */
	cur.ndstrs = data->d_size;
	/*
	 * Look for DT_SONAME
	 */
	cur.soname = NULL;
	for (i = 0; i < cur.ndyns; i++) {
		if (cur.dyns[i].d_tag != DT_SONAME) {
			continue;
		}
		if (cur.soname) {
			file_error(fn, "multiple SONAMEs");
			return -1;
		}
		if (cur.dyns[i].d_un.d_val >= cur.ndstrs) {
			file_error(fn, "invalid d_un.d_val %#x in SONAME entry",
					   (unsigned)cur.dyns[i].d_un.d_val);
			return -1;
		}
		cur.soname = cur.dstrs + cur.dyns[i].d_un.d_val;
	}
	if (!cur.soname) {
		cur.soname = cur.fn + noff;
	}
	return 0;
}

#if 0
int
ld_reloc_deps(size_t i, int *found) {
	Elf_Data *data;
	GElf_Shdr *sh;
	size_t j;
	size_t k;
	size_t n;

	assert(found);
	if (!(data = elf_getdata(cur.scns[i].scn, NULL))) {
		file_error(cur.fn, "elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	sh = &cur.scns[i].sh;
	n = number_of_elems(sh, data->d_type);
	for (i = 0; i < n; i++) {
		GElf_Rela rela;

		if (data->d_type == ELF_T_REL) {
			if (!gelf_getrel(data, i, (GElf_Rel*)&rela)) {
				file_error(cur.fn, "gelf_getrel: %s", elf_errmsg(-1));
				return -1;
			}
		}
		else {
			assert(data->d_type == ELF_T_RELA);
			if (!gelf_getrela(data, i, &rela)) {
				file_error(cur.fn, "gelf_getrela: %s", elf_errmsg(-1));
				return -1;
			}
		}
		j = GELF_R_SYM(rela.r_info);
		if (j < 1 || j >= cur.nsyms) {
			continue;
		}
		k = cur.syms[j].st_shndx;
		if (k == SHN_XINDEX) {
			k = cur.shndx[j];
		}
		if (k < 1 || k >= cur.nscns) {
			continue;
		}
		if (cur.scns[k].flags & LDSF_NEEDED) {
			continue;
		}
		debug("need section `%s' (%u) [reloc]",
			cur.names + cur.scns[k].sh.sh_name, (unsigned)k);
		cur.scns[k].flags |= LDSF_NEEDED;
		*found = 1;
	}
	return 0;
}
#endif

static int
may_strip_section(const char *name, GElf_Shdr *sh) {
	if (sh->sh_type == SHT_NULL) return 1;		/* unused header */
	if (sh->sh_flags & SHF_ALLOC) return 0;
	switch (sh->sh_type) {
		case SHT_DYNSYM:
		case SHT_SYMTAB:
		case SHT_SYMTAB_SHNDX:
			/*
			 * We handle these separately
			 */
			return 1;
		case SHT_PROGBITS:
			if (!ld_opt_strip) return 0;
			/*
			 * Strip debugging sections
			 */
			if (strcmp(name, ".line") == 0) return 1;
			if (strncmp(name, ".debug", 6) == 0) return 1;
			if (strncmp(name, ".stab", 5) == 0) return 1;
			return 0;
		case SHT_STRTAB:
			if (ld_opt_strip) return 1;
			/*
			 * Keep debugging strings
			 */
			if (strncmp(name, ".stab", 5) == 0) return 0;
			return 1;
	}
	/*
	 * Most sections can't be stripped
	 */
	return 0;
}

int
ld_needed_sections(void) {
	static size_t *chain = NULL;
	GElf_Shdr *sh;
	const char *name;
	int found;
	size_t i;
	size_t j;

	assert(cur.eh.e_type == ET_REL);

	/*
	 * Consistency check
	 */
	for (i = 1; i < cur.nscns; i++) {
		sh = &cur.scns[i].sh;
		if (sh->sh_type != SHT_REL && sh->sh_type != SHT_RELA) {
			continue;
		}
		j = sh->sh_link;
		if (j == i || j < 1 || j >= cur.nscns
		 /* XXX: assuming ET_REL files here */
		 || cur.scns[j].sh.sh_type != SHT_SYMTAB) {
			file_error(cur.fn, "bad sh_link in reloc section");
			return -1;
		}
		j = sh->sh_info;
		if (j == i || j < 1 || j >= cur.nscns) {
			file_error(cur.fn, "bad sh_info in reloc section");
			return -1;
		}
	}

	chain = xrealloc(chain, cur.nscns * sizeof(*chain));
	for (i = 0; i < cur.nscns; i++) {
		chain[i] = 0;
	}
	do {
		found = 0;
		for (i = 1; i < cur.nscns; i++) {
			if (cur.scns[i].flags & LDSF_NEEDED) {
				continue;
			}
			sh = &cur.scns[i].sh;
			name = cur.names + sh->sh_name;
			if (sh->sh_type == SHT_DYNSYM
			 || sh->sh_type == SHT_SYMTAB
			 || sh->sh_type == SHT_SYMTAB_SHNDX) {
				/*
				 * Handled in ld_scan_file()
				 */
				continue;
			}
			if (may_strip_section(name, sh)) {
				continue;
			}
			if (sh->sh_type == SHT_REL || sh->sh_type == SHT_RELA) {
				if (!(cur.scns[sh->sh_info].flags & LDSF_NEEDED)) {
					/*
					 * Skip relocations for stripped section (e.g. .rel.stab)
					 */
					continue;
				}
				/*
				 * Remember used relocation sections
				 */
				chain[i] = chain[0];
				chain[0] = i;
			}
			cur.scns[i].flags |= LDSF_NEEDED;
			found = 1;
		}
#if 0
		/*
		 * Process reloc chain
		 * XXX: isn't symbol processing (below) enough?
		 */
		for (i = 0; (j = chain[i]); i = j) {
			if (ld_reloc_deps(j, &found)) {
				return -1;
			}
			chain[i] = 0;
		}
		if (!found) {
			/*
			 * Process symbol table entries
			 */
			for (i = 1; i < cur.nsyms; i++) {
				GElf_Sym sym = cur.syms[i];

				j = sym.st_shndx;
				if (j == SHN_XINDEX) {
					j = cur.shndx[i];
				}
				if (j < 1 || j >= cur.nscns) {
					continue;
				}
				if (GELF_ST_BIND(sym.st_info) == STB_LOCAL) {
					continue;
				}
				if (cur.scns[j].flags & LDSF_NEEDED) {
					continue;
				}
				debug("need additional section `%s' (%u) [symbol %s]",
					cur.names + cur.scns[j].sh.sh_name,
					(unsigned)j, cur.strs + sym.st_name);
				cur.scns[j].flags |= LDSF_NEEDED;
				found = 1;
			}
		}
#endif
	}
	while (found);
	return 0;
}

int
ld_copy_relocs(void) {
	Elf_Data *data;
	GElf_Shdr *sh;
	GElf_Xword sym;
	const char *name;
	size_t i;
	size_t j;
	size_t n;
	size_t off;
	struct outscn *sect;

	debug("ld_copy_relocs");

	for (i = 1; i < cur.nscns; i++) {
		sh = &cur.scns[i].sh;
		name = cur.names + sh->sh_name;
		if (sh->sh_type != SHT_REL && sh->sh_type != SHT_RELA) {
			continue;
		}
		if (!(cur.scns[i].flags & LDSF_NEEDED)) {
			debug("ignoring section `%s' (%u)", name, (unsigned)i);
			continue;
		}
		if (!(sect = cur.scns[sh->sh_info].out)) {
			fatal("reference to ignored section %u in reloc pass",
				(unsigned)sh->sh_info);
		}
		if (!(data = elf_getdata(cur.scns[i].scn, NULL))) {
			file_error(cur.fn, "elf_getdata: %s", elf_errmsg(-1));
			return -1;
		}
		/*
		 * Copy data
		 */
		assert(data->d_type == ELF_T_REL || data->d_type == ELF_T_RELA);
		if (sh->sh_size == 0) {
			file_warn(cur.fn, "empty relocation section `%s'", name);
			continue;
		}
		if (!(n = number_of_elems(sh, data->d_type))) {
			return -1;
		}
		off = cur.scns[sh->sh_info].off;
		if (sh->sh_type == SHT_RELA) {
			GElf_Rela *rela;

			sect->nrela += n;
			sect->rela = xrealloc(sect->rela, sect->nrela * sizeof(*rela));
			rela = sect->rela + (sect->nrela - n);
			for (j = 0; j < n; j++, rela++) {
				if (!gelf_getrela(data, j, rela)) {
					file_error(cur.fn, "gelf_getrela: %s", elf_errmsg(-1));
					return -1;
				}
				if ((sym = GELF_R_SYM(rela->r_info))) {
					if (GELF_ST_TYPE(cur.syms[sym].st_info) == STT_SECTION) {
						if (!(sym = ld_section_symbol(sym))) {
							return -1;
						}
					}
					else if (!(sym = cur.stable[sym])) {
						file_error(cur.fn, "relocation references invalid symbol");
						return -1;
					}
					rela->r_info = GELF_R_INFO(sym, GELF_R_TYPE(rela->r_info));
				}
				if (off) {
					rela->r_offset += off;
				}
			}
		}
		else {
			GElf_Rel *rel;

			sect->nrel += n;
			sect->rel = xrealloc(sect->rel, sect->nrel * sizeof(*rel));
			rel = sect->rel + (sect->nrel - n);
			for (j = 0; j < n; j++, rel++) {
				if (!gelf_getrel(data, j, rel)) {
					file_error(cur.fn, "gelf_getrel: %s", elf_errmsg(-1));
					return -1;
				}
				if ((sym = GELF_R_SYM(rel->r_info))) {
					if (GELF_ST_TYPE(cur.syms[sym].st_info) == STT_SECTION) {
						if (!(sym = ld_section_symbol(sym))) {
							return -1;
						}
					}
					else if (!(sym = cur.stable[sym])) {
						file_error(cur.fn, "relocation references invalid symbol");
						return -1;
					}
					rel->r_info = GELF_R_INFO(sym, GELF_R_TYPE(rel->r_info));
				}
				if (off) {
					rel->r_offset += off;
				}
			}
		}
	}
	return 0;
}

int
ld_add_elf_object(void) {
	if (ld_needed_sections()) {
		return -1;
	}
	if (ld_copy_data_sections()) {
		return -1;
	}
	if (ld_copy_symbols()) {
		return -1;
	}
	if (ld_copy_relocs()) {
		return -1;
	}
	debug("added `%s'", cur.fn);
	return 0;
}

struct target *target = NULL;

int
ld_file_type_match(void) {
	if (target == NULL) {
		target = ld_select_target(&cur.eh);
		if (target == NULL) {
			error("target machine (%u) not supported",
				(unsigned)cur.eh.e_machine);
			cleanup(1);
		}
		if (!gelf_newehdr(aelf, cur.eh.e_ident[EI_CLASS])) {
			file_error(outfn, "gelf_newehdr: %s", elf_errmsg(-1));
			return -1;
		}
		if (!gelf_getehdr(aelf, &aeh)) {
			file_error(outfn, "gelf_getehdr: %s", elf_errmsg(-1));
			return -1;
		}
		aeh.e_ident[EI_CLASS] = cur.eh.e_ident[EI_CLASS];
		aeh.e_ident[EI_DATA] = cur.eh.e_ident[EI_DATA];
		aeh.e_type = ld_target_type;
		aeh.e_machine = cur.eh.e_machine;
		aeh.e_version = EV_CURRENT;
		aeh.e_entry = 0;
		aeh.e_flags = 0;
		aeh.e_shstrndx = 0;
		if (!gelf_update_ehdr(aelf, &aeh)) {
			file_error(outfn, "gelf_update_ehdr: %s", elf_errmsg(-1));
			return -1;
		}
	}
	if (aeh.e_ident[EI_CLASS] != cur.eh.e_ident[EI_CLASS]
	 || aeh.e_ident[EI_DATA] != cur.eh.e_ident[EI_DATA]
	 || aeh.e_machine != cur.eh.e_machine) {
		file_error(cur.fn, "file type mismatch");
		return -1;
	}
	if (cur.eh.e_version > EV_CURRENT) {
		file_warn(cur.fn, "file has future ELF version");
	}
	return 0;
}

int
ld_add_elf_file(const char *fn, size_t noff, Elf *elf) {
	if (ld_scan_file(fn, noff, elf)) {
		return -1;
	}
	if (cur.nscns == 0) {
		return 0;
	}
	if (cur.eh.e_type == ET_REL) {
		debug("file=%s	[ ET_REL ]", cur.fn);
		return ld_add_elf_object();
	}
#if 0 /* not yet */
	if (ld_opt_dynamic && cur.eh.e_type == ET_DYN) {
		debug("file=%s	[ ET_DYN ]", cur.fn);
		return ld_add_elf_shlib();
	}
#endif
	file_error(cur.fn, "not a relocatable file");
	return -1;
}
