/*
 * fcpu.c -- F-CPU target for ld(1)
 * Copyright (C) 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	 02111-1307	 USA
 */

static const char rcsid[] = "@(#) $Id: fcpu.c,v 1.6 2003/02/08 13:10:15 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <gelf.h>
#include <assert.h>

#include <ld/ld.h>
#include <ld/ldmisc.h>

#include <as/elf_FCPU.h>

extern int fcpu_apply_rela(struct target *tgt, struct outscn *sect);

static struct target targets[] = {
	{
		.name = "fcpu",
		.class = ELFCLASS64,
		.data = ELFDATA2LSB,
		.machine = EM_FCPU,
		.page_size = (GElf_Addr)1 << 20,	/* 1 MB */
		.load_address = (GElf_Addr)1 << 32,	/* 4 GB */
		.rel_size = sizeof(Elf64_Rel),
		.rela_size = sizeof(Elf64_Rela),
		.copy_rel = generic_copy_rel,
		.copy_rela = generic_copy_rela,
		.apply_rel = NULL,
		.apply_rela = fcpu_apply_rela,
	},
};

void
init_target_fcpu(void) {
	size_t i;

	for (i = 0; i < sizeof(targets) / sizeof(*targets); i++) {
		ld_register_target(&targets[i]);
	}
}

/* DO NOT REMOVE THIS!
DEFTARGET(fcpu)
*/

/*
 * Implementation
 */

#include <fcpu_opcodes/fcpu_opcodes.h>	/* for INSTRUCTION_BIG_ENDIAN */

int
fcpu_apply_rela(struct target *tgt, struct outscn *sect) {
	Elf_Data *data;
	GElf_Addr addr;
	GElf_Addr orig;
	GElf_Addr value;
	GElf_Rela *rela;
	GElf_Word index;
	GElf_Word type;
	const char *name;
	size_t i;
	size_t relsize;
	unsigned char *place;

	if (!(data = elf_getdata(sect->scn, NULL))) {
		error("elf_getdata: %s", elf_errmsg(-1));
		return -1;
	}
	assert(data->d_type == ELF_T_BYTE);
	for (i = 0; i < sect->nrela; i++) {
		rela = &sect->rela[i];
		type = GELF_R_TYPE(rela->r_info);
		index = GELF_R_SYM(rela->r_info);
		if (type == R_FCPU_NONE) {
			continue;
		}
		assert(sect->shdr.sh_addr);
		addr = sect->shdr.sh_addr + rela->r_offset;
		if (index && index >= anlocs) {
			/*
			 * Calculate global symbol index
			 */
			index = SYMBOLS_MAX - index;
			assert(0 < index && index < anglobs);
			value = aglobals[index].sym.st_value;
			name = ld_symbol_name(aglobals[index].sym.st_name);
		}
		else {
			value = alocals[index].sym.st_value;
			name = ld_symbol_name(alocals[index].sym.st_name);
		}
		debug("relocating: type=%u sym=`%s' (%lx) addr=%#Lx value=%#Lx addend=%#Lx",
			(unsigned)type, name, (unsigned long)GELF_R_SYM(rela->r_info),
			(unsigned long long)addr, (unsigned long long)value,
			(unsigned long long)rela->r_addend);
		value += rela->r_addend;

		/*
		 * PC-relative relocations
		 */
		switch (type) {
			case R_FCPU_PC17:
			case R_FCPU_PC8:
			case R_FCPU_LE_PC16:
			case R_FCPU_BE_PC16:
			case R_FCPU_LE_PC32:
			case R_FCPU_BE_PC32:
			case R_FCPU_LE_PC64:
			case R_FCPU_BE_PC64:
			case R_FCPU_U_PC16:
			case R_FCPU_U_PC32:
			case R_FCPU_U_PC48:
			case R_FCPU_U_PC64:
				value -= addr; break;
		}

		/*
		 * Relocation sizes
		 */
		relsize = 4;
		switch (type) {
			case R_FCPU_8:
			case R_FCPU_PC8:
				relsize = 1; break;
			case R_FCPU_LE_16:
			case R_FCPU_LE_PC16:
			case R_FCPU_BE_16:
			case R_FCPU_BE_PC16:
				relsize = 2; break;
			case R_FCPU_LE_32:
			case R_FCPU_LE_PC32:
			case R_FCPU_BE_32:
			case R_FCPU_BE_PC32:
				relsize = 4; break;
			case R_FCPU_LE_64:
			case R_FCPU_LE_PC64:
			case R_FCPU_BE_64:
			case R_FCPU_BE_PC64:
				relsize = 8; break;
			case R_FCPU_U_16:
			case R_FCPU_U_PC16:
				break;
			case R_FCPU_U_32:
			case R_FCPU_U_PC32:
				value >>= 16; break;
			case R_FCPU_U_48:
			case R_FCPU_U_PC48:
				value >>= 32; break;
			case R_FCPU_U_64:
			case R_FCPU_U_PC64:
				value >>= 48; break;
		}
		if (rela->r_offset + relsize > data->d_size) {
			error("relocation offset out of range");
			return -1;
		}
		place = (unsigned char*)data->d_buf + rela->r_offset;
		orig = ld_get_data(place, 4, INSTRUCTION_BIG_ENDIAN);

		/*
		 * Apply relocation
		 */
		switch (type) {
			case R_FCPU_PC17:
				if ((GElf_Addr)(value + 0x10000) >= 0x20000) {
					error("displacement out of range - cannot relocate");
					return -1;
				}
				value = ((value << 6) & 0x7fffc0) | (orig &~ 0x7fffc0);
				ld_put_data(place, 4, value, INSTRUCTION_BIG_ENDIAN);
				break;

			case R_FCPU_8:
			case R_FCPU_PC8:
			case R_FCPU_LE_16:
			case R_FCPU_LE_PC16:
			case R_FCPU_LE_32:
			case R_FCPU_LE_PC32:
			case R_FCPU_LE_64:
			case R_FCPU_LE_PC64:
				ld_put_data(place, relsize, value, 0);
				break;

			case R_FCPU_BE_16:
			case R_FCPU_BE_PC16:
			case R_FCPU_BE_32:
			case R_FCPU_BE_PC32:
			case R_FCPU_BE_64:
			case R_FCPU_BE_PC64:
				ld_put_data(place, relsize, value, 1);
				break;

			case R_FCPU_U_PC16:
			case R_FCPU_U_16:
			case R_FCPU_U_PC32:
			case R_FCPU_U_32:
			case R_FCPU_U_PC48:
			case R_FCPU_U_48:
			case R_FCPU_U_PC64:
			case R_FCPU_U_64:
				value = ((value << 6) & 0x3fffc0) | (orig &~ 0x3fffc0);
				ld_put_data(place, 4, value, INSTRUCTION_BIG_ENDIAN);
				break;

			default:
				error("unrecognized relocation type (%u)", (unsigned)type);
				return -1;
		}
	}
	return 0;
}
