/*
 * syscalls.c -- ELF-based F-CPU emulator, system calls
 * Copyright (C) 2002, 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static const char rcsid[] = "@(#) $Id: syscalls.c,v 1.3 2003/02/08 22:23:26 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
extern void *malloc(), *memset();
extern size_t strlen();
extern int memcmp();
#endif

#include <stdio.h>
#include <assert.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#else
extern int close();
extern char **environ;
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#else
extern int errno;
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#else
extern int open();
#endif
#ifndef O_RDONLY
#define O_RDONLY 0
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif
#ifndef MAP_FAILED
#define MAP_FAILED	(-1)
#endif

#if HAVE_SETJMP_H
#include <setjmp.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#endif

#if HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#if HAVE_GRP_H
#include <grp.h>
#endif

#include <emu/emu.h>
#include <emu/syscalls.h>

#define INITIAL_STACK_POINTER	0xffffffff00000000ull
#define STACKSIZE				(1ull << 20)	/* 1 MB */
#define PAGESIZE				4096u

extern void emulator_setmap(U64, char*, U64, int);
extern void emulator_delmap(U64, U64);
extern void *emulator_map(U64, U64, int);
extern int do_brk(U64 newbrk);
extern U64 do_sbrk(I64 delta);

static jmp_buf jb;

static U64
check_arg(U64 val, U64 shortval, int ecode) {
	if (val == shortval) return val;
	longjmp(jb, ecode);
}

static void*
check_ptr(U64 addr, U64 size, size_t algn, int prot) {
	void *p;

	if (algn > 1 && addr % algn) longjmp(jb, EINVAL);
	p = emulator_map(addr, size, prot);
	if (p) return p;
	ex(EX_NONE);
	longjmp(jb, EINVAL);
}

static char*
check_str(U64 addr) {
	size_t i = 0;
	char *s;

	/* XXX: this is slow */
	s = check_ptr(addr, 1, 0, PROT_READ);
	while (s[i++]) {
		check_ptr(addr + i, 1, 0, PROT_READ);
	}
	return s;
}

static char**
copy_vec(U64 addr) {
	size_t i, count;
	char **vec;
	U64 *p;

	/* XXX: this is slow */
	for (count = 0; ; count++) {
		p = check_ptr(addr + 8 * count, 8, 8, PROT_READ);
		assert(p);
		if (*p == 0) {
			break;
		}
	}
	if (!(vec = malloc((count + 1) * sizeof(char*)))) {
		longjmp(jb, ENOMEM);
	}
	for (i = 0; i < count; i++) {
		p = emulator_map(addr + 8 * i, 8, PROT_READ);
		vec[i] = check_str(*p);
	}
	vec[i] = NULL;
	return vec;
}

#define arg(n) \
	(r(n+1).C(o,0))
#define checkarg(n, type) \
	check_arg(arg(n), (type)arg(n), EINVAL)
#define checkrptr(n, size, algn) \
	check_ptr(arg(n), size, algn, PROT_READ)
#define checkwptr(n, size, algn) \
	check_ptr(arg(n), size, algn, PROT_WRITE)
#define checkrwptr(n, size, algn) \
	check_ptr(arg(n), size, algn, PROT_READ | PROT_WRITE)
#define checkstr(n) \
	check_str(arg(n))
#define copyvec(n) \
	copy_vec(arg(n))

#define checkint(x) checkarg((x), int)

#define SYSCALL(name, impl) static U64 sys_##name(void) impl
#include <emu/syscalls.def>
#undef SYSCALL

static U64 (*syscall_table[])(void) = {
#define SYSCALL(name, impl) [SYS_##name] = sys_##name,
#include <emu/syscalls.def>
#undef SYSCALL
};

const size_t SYS_num = sizeof(syscall_table) / sizeof(*syscall_table);

U64
do_syscall(void) {
	if (!(errno = setjmp(jb))) {
		U64 sel = r(1).C(o,0);
		U64 (*f)(void);

		if (sel < SYS_num && (f = syscall_table[sel])) {
			errno = 0;
			return f();
		}
		errno = ENOSYS;
	}
	return -1;
}
