/*
 * emumain.c -- F-CPU instruction-level emulator main program
 * Copyright (C) 2002, 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static const char rcsid[] = "@(#) $Id: emumain.c,v 1.5 2003/02/01 14:01:36 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
void *malloc();
unsigned long strtoul();
#endif

#include <stdio.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#else
int read(), write();
int getopt(), optind;
char *optarg;
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#else
extern int errno;
#endif

#if HAVE_MATH_H
#include <math.h>
#else
int isnan();
double sqrt(), log(), exp();
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#else
int open(), close();
#endif
#ifndef O_RDONLY
#define O_RDONLY 0
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_MMAN_H
#include <sys/mman.h>
#ifndef MAP_FAILED
#define MAP_FAILED (-1)
#endif
#if !defined(MAP_ANONYMOUS) && defined(MAP_ANON)
#define MAP_ANONYMOUS	MAP_ANON
#endif
#endif

#include <fcpu_opcodes/fcpu_opcodes.h>
#include <emu/emu.h>

/* target -> host memory mapping */
unsigned char *addrbase;	/* F-CPU core memory is mapped here */
size_t ramsize;				/* total amount of RAM available for emulation */

void*
memmap(U64 virtaddr, U64 align, U64 len, int write_mode) {
	/* no MMU for now */
	if (len > ramsize) {
		ex(EX_ADDRESS);
		return NULL;
	}
	if (virtaddr >= ramsize) {
		ex(EX_ADDRESS);
		return NULL;
	}
	if (virtaddr + len > ramsize) {
		ex(EX_ADDRESS);
		return NULL;
	}
	if (align > 1 && virtaddr % align) {
		ex(EX_ALIGNMENT);
	}
	return addrbase + virtaddr;
}

U32
fetch(U64 ip) {
	unsigned char *p = memmap(ip, 4, 4, 0);
	U32 insn = 0;

	if (!p) return 0xffffffff;
#if INSTRUCTION_BIG_ENDIAN
	insn = (insn << 8) | p[0];
	insn = (insn << 8) | p[1];
	insn = (insn << 8) | p[2];
	insn = (insn << 8) | p[3];
#else
	insn = (insn << 8) | p[3];
	insn = (insn << 8) | p[2];
	insn = (insn << 8) | p[1];
	insn = (insn << 8) | p[0];
#endif
	return insn;
}

void
show(unsigned long ip, unsigned insn) {
	char buf[1024];

	if (fcpu_decode_instruction(buf, sizeof(buf), insn) < 0) {
		abort();
	}
	fprintf(stderr, "%08lx  %02x%02x%02x%02x  %s\n", ip, (U8)(insn>>24),
			(U8)(insn>>16), (U8)(insn>>8), (U8)(insn>>0), buf);
}

void
exception(unsigned long ip, unsigned insn, unsigned code) {
	static const char *msgs[EX_number] = {
		[EX_ACCESS]    = "access rights violation",
		[EX_ADDRESS]   = "address out of range",
		[EX_ALIGNMENT] = "misaligned address",
		[EX_INVALID]   = "invalid instruction",
		[EX_ZERO]      = "divide by 0",
		[EX_RANGE]     = "value out of range",
		[EX_HALT]      = "halted",
		[EX_NOHAND]    = "no exception handler",
	};

	if (code >= EX_number || !msgs[code]) {
		fprintf(stderr, "\n*** exception %u ***\n", code);
	}
	else {
		fprintf(stderr, "\n*** %s ***\n", msgs[code]);
	}
}

#define fcpu_io(rw, func, a0, a1, a2, a3)	\
	do { \
		int fd = a1; \
		size_t n = a3; \
		void *p = memmap(a2, 0, a3, rw); \
		if (fd != a1 || n != a3 || !p || excode != EX_NONE) { \
			a0 = -EINVAL; \
			ex(EX_NONE); \
			break; \
		} \
		n = func(fd, p, n); \
		if (n == (size_t)-1) { \
			a0 = -errno; \
			break; \
		} \
		a0 = n; \
	} while (0)

/* provide simple I/O via syscall */
int syscall_handler(U32 opcode) {
	if (UIMM16 != 0) {
		/* let emulator handle other syscalls */
		return 0;
	}
	if (r(1).C(q,0) != r(1).C(o,0)) {
		/* syscall number too big */
		r(1).C(o,0) = -ENOSYS;
		return 1;
	}
	switch (r(1).C(q,0)) {
		case 0:
			fprintf(stderr, "program exited with status %u\n", r(2).C(b,0));
			exit(0);
		case 1:
			fcpu_io(1, read, r(1).C(o,0), r(2).C(o,0), r(3).C(o,0), r(4).C(o,0));
			break;
		case 2:
			fcpu_io(0, write, r(1).C(o,0), r(2).C(o,0), r(3).C(o,0), r(4).C(o,0));
			break;
		default:
			r(1).C(o,0) = -ENOSYS;
			break;
	}
	return 1;
}

static const char usage[] = "usage: emu [-i] [-m mbytes] binfile\n";

#if INSTRUCTION_BIG_ENDIAN
#define INSN(x)	(char)((x)>>24),(char)((x)>>16),(char)((x)>>8),(char)(x)
#else
#define INSN(x)	(char)(x),(char)((x)>>8),(char)((x)>>16),(char)((x)>>24)
#endif

#define X(op)	((op)<<24)

char smoke_test[] = {
	/* loadaddri $target-.-4, r1 */
	INSN(X(OP_LOADADDRI) | 28 << 6 | 1 << 0),
	/* jmp r1, r3 */
	INSN(X(OP_JMP) | 1 << 6 | 3 << 0),
	/* .string */
	's','m','o','k','e',' ','t','e',
	's','t',' ','s','u','c','c','e',
	's','s','f','u','l','!','\n',0,
	/* inc r0, r2 */
	INSN(X(OP_INC) | ISIZE_64BIT | 0 << 6 | 2 << 0),
	/* loadconsx.0 $23, r4 */
	INSN(X(OP_LOADCONSX) | 23 << 6 | 4 << 0),
	/* loadconsx.0 $2, r1 */
	INSN(X(OP_LOADCONSX) | 2 << 6 | 1 << 0),
	/* syscall $0, r0 */
	INSN(X(OP_SYSCALL) | 0 << 6 | 0 << 0),
	/* nop */
	INSN(X(OP_NOP)),
	/* halt */
	INSN(X(OP_HALT)),
};

#undef X

int
main(int argc, char **argv) {
	int interactive = 0;
	int trace = 0;
	unsigned char *p;
	struct stat st;
	int x;
	U64 ip;
	U32 insn;

	ramsize = 0;
	while ((x = getopt(argc, argv, "him:")) != EOF) {
		switch (x) {
			case 'i':
				interactive = 1;
				break;
			case 'm':
				ramsize = strtoul(optarg, NULL, 0) << 20;
				break;
			case 'h':
			case '?':
				fprintf(stderr, usage);
				exit(x != 'h');
		}
	}
	if (optind != argc && optind + 1 != argc) {
		fprintf(stderr, usage);
		exit(1);
	}
	if (!ramsize) {
		ramsize = 32 << 20;
	}
	fprintf(stderr, "RAM size set to %lu MB\n", (unsigned long)ramsize >> 20);
#if HAVE_MMAP && defined(MAP_ANONYMOUS)
	addrbase = mmap(0, ramsize, PROT_READ | PROT_WRITE,
					MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);
	if (addrbase == (unsigned char*)MAP_FAILED) {
#else
	if (1) {
#endif
		addrbase = malloc(ramsize);
		if (addrbase == NULL) {
			perror("malloc");
			exit(1);
		}
	}
	if (optind < argc) {
		if ((x = open(argv[optind], O_RDONLY)) == -1) {
			perror(argv[optind]);
			exit(1);
		}
		if (fstat(x, &st)) {
			perror("fstat");
			exit(1);
		}
		if (st.st_size == 0) {
			fprintf(stderr, "%s: empty file\n", argv[optind]);
			exit(1);
		}
#if HAVE_MMAP
		p = mmap(addrbase, st.st_size, PROT_READ | PROT_WRITE,
				 MAP_PRIVATE | MAP_FIXED, x, 0);
		if (p == (unsigned char*)MAP_FAILED) {
#else
		if (1) {
#endif
			size_t n = 0, nread = 0;

			fprintf(stderr, "warning: mmap failed, trying read\n");
			while (nread < ramsize
			&&     (n = read(x, addrbase + nread, ramsize - nread)) > 0) {
				nread += n;
			}
			if (n == (size_t)-1) {
				perror("read");
				exit(1);
			}
		}
		else if (p != addrbase) {
			fprintf(stderr, "mapping failed\n");
			exit(1);
		}
		close(x);
	}
	else {
		memcpy(addrbase, &smoke_test, sizeof(smoke_test));
	}

	fputs("F-CPU instruction-level emulator " VERSION_STRING "\n"
		"Copyright (C) 2002, 2003 Michael \"Tired\" Riepe\n"
		"This program is free software; you can redistribute it and/or\n"
		"modify it under the terms of the GNU General Public License.\n"
		"This program is distributed WITHOUT ANY WARRANTY.\n\n",
		stderr);
	fprintf(stderr, "Emulated F-CPU has %u-bit registers\n", 8 * MAXSIZE);
	fprintf(stderr, "and uses %s write semantics.\n\n", PARTIAL_WRITES ? "old" : "new");
	fprintf(stderr, "Enter \"?\" for help.\n\n");
	initemu();
	syscall_hook = syscall_handler;

	for (;;) {
		int done = 0;
		unsigned i;

		ex(EX_NONE);
		ip = regs.r_pc.C(o,0);
		insn = fetch(ip);
		if (!interactive) {
			while (!excode) {
				if (trace) {
					show(ip, insn);
				}
				regs.r_pc.C(o,0) = ip + 4;
				emulate1(insn);
				if (excode) break;
				ip = regs.r_pc.C(o,0);
				insn = fetch(ip);
			}
			regs.r_pc.C(o,0) = ip;
			interactive = 1;
			trace = 0;
		}
		if (excode) {
			exception(ip, insn, excode);
			ex(EX_NONE);
		}
		show(ip, insn);
		do {
			char line[1024], *s;
			int regno;

			fprintf(stderr, "-");
			if (!fgets(line, sizeof(line), stdin)) {
				exit(0);
			}
			switch (*line) {
				case '\n':
					continue;
				case 'g':
					if (line[1] == '=') {
						regs.r_pc.C(o,0) = strtoul(line + 2, NULL, 16);
					}
					interactive = 0;
					done = 1;
					break;
				case 'l':
					if (line[1] != '\n') {
						ip = strtoul(line + 1, NULL, 16);
						insn = fetch(ip);
					}
					for (i = 0; i < 16; i++) {
						show(ip, insn);
						insn = fetch(ip += 4);
					}
					ex(EX_NONE);
					break;
				case 'q':
					exit(0);
				case 'r':
					regno = strtoul(line + 1, &s, 10);
					if (s == line + 1 || regno > 63u) break;
					printf("r%d=%llx\n", regno, r(regno).C(o,0));
					break;
				case 's':
					if (line[1] == '=') {
						regs.r_pc.C(o,0) = strtoul(line + 2, NULL, 16);
					}
					ip = regs.r_pc.C(o,0);
					insn = fetch(ip);
					if (excode) {
						exception(ip, insn, excode);
						ex(EX_NONE);
						show(ip, insn);
						break;
					}
					regs.r_pc.C(o,0) = ip + 4;
					emulate1(insn);
					if (excode) {
						exception(ip, insn, excode);
						regs.r_pc.C(o,0) = ip;
						ex(EX_NONE);
						show(ip, insn);
						break;
					}
					done = 1;
					break;
				case 't':
					if (line[1] == '=') {
						regs.r_pc.C(o,0) = strtoul(line + 2, NULL, 16);
					}
					interactive = 0;
					trace = 1;
					done = 1;
					break;
				case '?':
					fprintf(stderr, "usage:\n"
						"  ?          show this help\n"
						"  g[=addr]   run (from addr)\n"
						"  l[addr]    disassemble (from addr)\n"
						"  q          quit\n"
						"  r{n}       show contents of register {n}\n"
						"  s[=addr]   single-step (from addr)\n"
						"  t[=addr]   trace (from addr)\n"
					);
					break;
				default:
					fprintf(stderr, "huh?\n");
			}
		}
		while (!done);
	}
}
