/*
 * elfemu.c -- ELF-based F-CPU emulator
 * Copyright (C) 2002, 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static const char rcsid[] = "@(#) $Id: elfemu.c,v 1.15 2003/02/09 00:21:18 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
extern void *malloc(), *realloc(), *memset();
extern size_t strlen();
extern int memcmp();
#endif

#include <stdio.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#else
extern int close();
#endif

extern char **environ;

#if HAVE_ERRNO_H
#include <errno.h>
#else
extern int errno;
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#else
extern int open();
#endif
#ifndef O_RDONLY
#define O_RDONLY 0
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_MMAN_H
#include <sys/mman.h>
#ifndef MAP_FAILED
#define MAP_FAILED	(-1)
#endif
#if !defined(MAP_ANONYMOUS) && defined(MAP_ANON)
#define MAP_ANONYMOUS	MAP_ANON
#endif
#endif

#include <libelf.h>
#include <as/elf_FCPU.h>

#include <fcpu_opcodes/fcpu_opcodes.h>
#include <emu/emu.h>
#include <emu/syscalls.h>

#define INITIAL_STACK_POINTER	0xffffffff00000000ull
#define STACKSIZE				(1ull << 20)	/* 1 MB */
#define PAGESIZE				4096u

const size_t pagesize = PAGESIZE;	/* XXX: get it from host! */

extern void emulator_setmap(U64, char*, U64, int);
extern void emulator_delmap(U64, U64);
extern void *emulator_map(U64, U64, int);

static void
elf_error(int err) {
	fprintf(stderr, "error: %s\n", elf_errmsg(err));
}

static void
put_data(void *p, U64 v) {
	unsigned i;

	for (i = 0; i < 8; i++) {
		((char*)p)[i] = v;
		v >>= 8;
	}
}

static U64 bss_start = 0;
static U64 brklvl = 0;
static char *bss_ptr = NULL;
static size_t bss_len = 0;

int
do_brk(U64 newbrk) {
	U64 bss_addr = bss_start &~ (U64)pagesize;	/* aligned address */
	U64 len;
	char *p;

	/*
	 * Check argument
	 */
	if (newbrk < bss_start) {
		errno = EINVAL;
		return -1;
	}

	/*
	 * Calculate required heap length
	 */
	len = newbrk - bss_addr;
	if (len % pagesize) {
		len += pagesize - len % pagesize;
	}
	if (len < pagesize) {
		len = pagesize;
	}

	/*
	 * Check for overflow
	 */
	if ((size_t)len != len) {
		errno = ENOMEM;
		return -1;
	}

	/*
	 * Allocate memory if necessary
	 */
	if ((size_t)len > bss_len) {
		size_t len2 = (size_t)len;

		p = bss_ptr ? realloc(bss_ptr, len2) : malloc(len2);
		if (p == NULL) {
			/* out of memory */
			/* note: memory at bss_ptr is still available */
			errno = ENOMEM;
			return -1;
		}
		/*
		 * Initialize fresh part
		 */
		memset(p + bss_len, 0, len2 - bss_len);
		bss_len = len2;
		bss_ptr = p;
	}

	/*
	 * Change mapping
	 */
	emulator_delmap(bss_start, brklvl - bss_start);
	emulator_setmap(bss_start, bss_ptr + (size_t)(bss_start - bss_addr),
		newbrk - bss_start, PROT_READ | PROT_WRITE | PROT_EXEC);
	brklvl = newbrk;
	return 0;
}

U64
do_sbrk(I64 delta) {
	U64 save = brklvl;

	if (do_brk(brklvl + delta)) {
		return -1;
	}
	return save;
}

int
load_elf(const char *name, int fd, char **argp, char **envp) {
	Elf *elf;
	Elf64_Ehdr *ehdr;
	Elf64_Phdr *phdr;
	U64 *sp;
	U64 end_bss = 0;
	U64 end_data = 0;
	U64 end_text = 0;
	char *cp;
	char *p;
	char *s;
	int prot;
	size_t argc;
	size_t envc;
	size_t i;
	size_t sz;

	/* open ELF file and check it */
	if (!(elf = elf_begin(fd, ELF_C_READ, (Elf*)0))) {
		elf_error(-1);
		return -1;
	}
	if (!(p = elf_getident(elf, &sz))) {
		elf_error(-1);
		elf_end(elf);
		return -1;
	}
	if (sz < EI_NIDENT
	 || memcmp(p, "\177ELF", 4) != 0
	 || p[EI_CLASS] != ELFCLASS64
	 || p[EI_DATA] != ELFDATA2LSB
	 || p[EI_VERSION] != EV_CURRENT) {
		errno = ENOEXEC;
		perror(name);
		elf_end(elf);
		return -1;
	}
	if (!(ehdr = elf64_getehdr(elf))) {
		elf_error(-1);
		elf_end(elf);
		return -1;
	}
	if (ehdr->e_type != ET_EXEC
	 || ehdr->e_machine != EM_FCPU) {
		errno = ENOEXEC;
		perror(name);
		elf_end(elf);
		return -1;
	}
	if (!(phdr = elf64_getphdr(elf))) {
		elf_error(-1);
		elf_end(elf);
		return -1;
	}

	/* bail out if this is not a statically linked executable */
	for (i = 0; i < ehdr->e_phnum; i++) {
		if (phdr[i].p_type == PT_INTERP || phdr[i].p_type == PT_DYNAMIC) {
			/* not supported yet */
			errno = ENOEXEC;
			perror(name);
			elf_end(elf);
			return -1;
		}
	}

	/* map ELF segments, one by one */
	for (i = 0; i < ehdr->e_phnum; i++) {
		if (phdr[i].p_type == PT_LOAD) {
			U64 off;
			U64 len;

			prot = 0;
			if (phdr[i].p_flags & PF_R) prot |= PROT_READ;
			if (phdr[i].p_flags & PF_W) prot |= PROT_WRITE;
			if (phdr[i].p_flags & PF_X) prot |= PROT_EXEC;
			off = phdr[i].p_offset & (pagesize - 1);
			len = off + phdr[i].p_filesz;
			p = (char*)mmap(0, len, prot,
#if defined(MAP_DENYWRITE) && defined(MAP_EXECUTABLE) /* Linux only */
				MAP_DENYWRITE | MAP_EXECUTABLE |
#endif
				MAP_PRIVATE, fd, phdr[i].p_offset - off);
			if (p == (char*)MAP_FAILED) {
				perror("mmap(segment)");
				elf_end(elf);
				return -1;
			}
			emulator_setmap(phdr[i].p_vaddr - off, p, len, prot);
			off = phdr[i].p_vaddr + phdr[i].p_filesz;
			if ((prot & PROT_EXEC) && end_text < off) end_text = off;
			if (end_data < off) end_data = off;
			off = phdr[i].p_vaddr + phdr[i].p_memsz;
			if (end_bss < off) end_bss = off;
		}
	}

	/* prepare BSS segment */
	if (end_data > end_bss) abort();
	bss_start = brklvl = end_data;
	if (do_brk(end_bss)) {
		perror("do_brk");
		elf_end(elf);
		return -1;
	}

	/* setup stack */
	prot = PROT_READ | PROT_WRITE | PROT_EXEC;
	p = malloc(STACKSIZE);
	if (p == NULL) {
		perror("malloc(stack)");
		elf_end(elf);
		return -1;
	}
	emulator_setmap(INITIAL_STACK_POINTER - STACKSIZE, p, STACKSIZE, prot);

	/* copy arguments */
	sz = 0;
	for (envc = 0; envp[envc]; envc++) {
		sz += strlen(envp[envc]) + 1;
	}
	for (argc = 0; argp[argc]; argc++) {
		sz += strlen(argp[argc]) + 1;
	}
	sz = (sz + 7) &~ 7;
	if (sz + 8 * (argc + envc + 2) > STACKSIZE) {
		errno = ENOMEM;
		perror(name);
		elf_end(elf);
		return -1;
	}
	cp = p + STACKSIZE - sz;
	sp = (U64*)cp;
	sp -= envc + 1;
	for (i = 0; i < envc; i++) {
		put_data(sp + i, (cp - p) + (INITIAL_STACK_POINTER - STACKSIZE));
		for (s = envp[i]; (*cp++ = *s++); );
	}
	put_data(sp + i, 0);
	sp -= argc + 1;
	for (i = 0; i < argc; i++) {
		put_data(sp + i, (cp - p) + (INITIAL_STACK_POINTER - STACKSIZE));
		for (s = argp[i]; (*cp++ = *s++); );
	}
	put_data(sp + i, 0);

	/* setup registers */
	r(1).C(o,0) = argc;
	r(2).C(o,0) = ((char*)sp - p) + (INITIAL_STACK_POINTER - STACKSIZE);
	sp += argc + 1;
	r(3).C(o,0) = ((char*)sp - p) + (INITIAL_STACK_POINTER - STACKSIZE);
	r(62) = r(2);
	regs.r_pc.C(o,0) = ehdr->e_entry;

	/* done */
	elf_end(elf);
	return 0;
}

struct vma {
	U64 vma_addr;
	U64 vma_len;
	struct vma *vma_next;
	char *vma_ptr;
	int vma_prot;
};

static struct vma *vma_list = NULL;
static struct vma *vma_freelist = NULL;

void
emulator_setmap(U64 addr, char *ptr, U64 len, int prot) {
	struct vma **p, *q, *r;

	/* look for matching vma */
	for (p = &vma_list; (q = *p); p = &q->vma_next) {
		if (addr <= q->vma_addr) {
			/* found */
			break;
		}
		if (addr < q->vma_addr + q->vma_len) {
			/* overlap with previous vma; cut it off */
			q->vma_len = addr - q->vma_addr;
		}
	}
	/* remove shadowed areas */
	while (q && addr + len >= q->vma_addr + q->vma_len) {
		*p = q->vma_next;
		q->vma_next = vma_freelist;
		vma_freelist = q;
		q = *p;
	}
	/* cut off overlapping vma */
	if (q && addr + len > q->vma_addr) {
		q->vma_len -= addr + len - q->vma_addr;
		q->vma_ptr += addr + len - q->vma_addr;
		q->vma_addr = addr + len;
	}
	if (addr && prot) {
		if ((r = vma_freelist)) {
			vma_freelist = r->vma_next;
		}
		else if (!(r = malloc(sizeof(struct vma)))) {
			perror("malloc(vma)");
			exit(1);
		}
		r->vma_addr = addr;
		r->vma_len = len;
		r->vma_next = q;
		r->vma_ptr = ptr;
		r->vma_prot = prot;
		*p = r;
	}
}

void
emulator_delmap(U64 addr, U64 len) {
	emulator_setmap(addr, NULL, len, 0);
}

static struct vma*
find_vma(U64 addr) {
	struct vma *p;

	for (p = vma_list; p; p = p->vma_next) {
		if (addr < p->vma_addr) {
			break;
		}
		if (addr < p->vma_addr + p->vma_len) {
			return p;
		}
	}
	return NULL;
}

void*
emulator_map(U64 addr, U64 len, int prot) {
	struct vma *p;

	p = find_vma(addr);
	if (p == NULL || addr + len > p->vma_addr + p->vma_len) {
		ex(EX_ADDRESS);
		return NULL;
	}
	if (prot &~ p->vma_prot) {
		ex(EX_ACCESS);
		return NULL;
	}
	return p->vma_ptr + (addr - p->vma_addr);
}

void*
memmap(U64 virtaddr, U64 align, U64 len, int write_mode) {
	int prot;

	if (align > 1 && virtaddr % align) {
		ex(EX_ALIGNMENT);
		return NULL;
	}
	prot = write_mode ? PROT_READ | PROT_WRITE : PROT_READ;
	return emulator_map(virtaddr, len, prot);
}

U32
fetch(U64 ip) {
	unsigned char *p;
	U32 insn = 0;

	if (ip % 4) {
		ex(EX_ALIGNMENT);
		return 0xffffffff;
	}
	p = emulator_map(ip, 4, PROT_EXEC);
	if (p == NULL) {
		return 0xffffffff;
	}
#if INSTRUCTION_BIG_ENDIAN
	insn = (insn << 8) | p[0];
	insn = (insn << 8) | p[1];
	insn = (insn << 8) | p[2];
	insn = (insn << 8) | p[3];
#else
	insn = (insn << 8) | p[3];
	insn = (insn << 8) | p[2];
	insn = (insn << 8) | p[1];
	insn = (insn << 8) | p[0];
#endif
	return insn;
}

void
show(unsigned long ip, unsigned insn) {
	char buf[1024];

	if (fcpu_decode_instruction(buf, sizeof(buf), insn) < 0) {
		abort();
	}
	fprintf(stderr, "%08lx  %02x%02x%02x%02x  %s\n", ip, (U8)(insn>>24),
			(U8)(insn>>16), (U8)(insn>>8), (U8)(insn>>0), buf);
}

void
exception(unsigned long ip, unsigned insn, unsigned code) {
	static const char *msgs[EX_number] = {
		[EX_ACCESS]    = "access rights violation",
		[EX_ADDRESS]   = "address out of range",
		[EX_ALIGNMENT] = "misaligned address",
		[EX_INVALID]   = "invalid instruction",
		[EX_ZERO]      = "divide by 0",
		[EX_RANGE]     = "value out of range",
		[EX_HALT]      = "halted",
		[EX_NOHAND]    = "no exception handler",
	};

	if (code >= EX_number || !msgs[code]) {
		fprintf(stderr, "\n*** exception %u ***\n", code);
	}
	else {
		fprintf(stderr, "\n*** %s ***\n", msgs[code]);
	}
}

int
syscall_handler(U32 opcode) {
	U64 res;

	if (UIMM16 != 0) {
		/* let emulator handle other syscalls */
		return 0;
	}
	res = do_syscall();
	if (excode == EX_NONE) {
		if (res == -1) {
			res = -errno;
		}
		r(1).C(o,0) = res;
	}
	return 1;
}

static void
usage(const char *pname) {
	fprintf(stderr, "usage: %s [-v] program [arg ...]\n", pname);
	exit(1);
}

int
main(int argc, char **argv) {
	int verbose = 0;
	U64 ip;
	U32 insn;
	int fd;
	int i;

	if (elf_version(EV_CURRENT) == EV_NONE) {
		elf_error(-1);
		exit(1);
	}
	for (i = 1; i < argc; i++) {
		if (argv[i][0] != '-') break;
		switch (argv[i][1]) {
			case '-': i++; break;
			case 'v': verbose = 1; break;
			default: usage(*argv);
		}
	}
	if (i >= argc) {
		usage(*argv);
	}
	argv += i;
	argc -= i;
	if ((fd = open(argv[0], O_RDONLY)) == -1) {
		perror(argv[0]);
		exit(1);
	}
	initemu();
	syscall_hook = syscall_handler;
	if (load_elf(argv[0], fd, argv, environ)) {
		close(fd);
		exit(1);
	}
	do {
		ip = regs.r_pc.C(o,0);
		insn = fetch(ip);
		if (excode != EX_NONE) break;
		if (verbose) show(ip, insn);
		regs.r_pc.C(o,0) = ip + 4;
		emulate1(insn);
	}
	while (excode == EX_NONE);
	exception(ip, insn, excode);
	show(ip, insn);
	exit(1);
}
