/*
 * disasm.c -- Simple Standalone F-CPU Disassembler
 * Copyright (C) 2002, 2003 Michael Riepe <michael@stud.uni-hannover.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static const char rcsid[] = "@(#) $Id: disasm.c,v 1.6 2003/01/29 03:38:50 michael Exp $";

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
void *malloc();
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#else
int close(), read(), write();
#endif

#include <stdio.h>

#if HAVE_FCNTL_H
#include <fcntl.h>
#else
int open();
#endif
#ifndef O_RDONLY
#define O_RDONLY 0
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif
#ifndef MAP_FAILED
#define MAP_FAILED (-1)
#endif

#include <fcpu_opcodes/fcpu_opcodes.h>

void
show(unsigned long ip, unsigned insn) {
	char buf[1024];

	if (fcpu_decode_instruction(buf, sizeof(buf), insn) < 0) {
		abort();
	}
	printf("%08lx  %02x%02x%02x%02x  %s\n", ip,
			(unsigned char)(insn>>24), (unsigned char)(insn>>16),
			(unsigned char)(insn>>8), (unsigned char)(insn>>0), buf);
}

int
main(int argc, char **argv) {
	unsigned char *p;
	struct stat st;
	size_t i;
	int x;

	if (argc != 2) {
		fprintf(stderr, "usage: disasm file\n");
		exit(1);
	}
	if ((x = open(*++argv, O_RDONLY)) == -1) {
		perror(*argv);
		exit(1);
	}
	if (fstat(x, &st)) {
		perror("fstat");
		exit(1);
	}
	if (st.st_size == 0) {
		fprintf(stderr, "%s: empty file\n", *argv);
		exit(1);
	}
#if HAVE_MMAP
	p = mmap(0, st.st_size, PROT_READ, MAP_SHARED, x, 0);
	if (p == (unsigned char*)MAP_FAILED) {
#else
	if (1) {
#endif
		size_t nread = 0;

		if (!(p = malloc(st.st_size))) {
			perror("malloc");
		}
		for (i = 0; nread < st.st_size; nread += i) {
			i = read(x, p + nread, st.st_size - nread);
			if (i == (size_t)-1) {
				perror("read");
				exit(1);
			}
			if (i == 0) {
				if (nread != st.st_size) {
					fprintf(stderr, "warning: file size changed\n");
					st.st_size = nread;
				}
				break;
			}
		}
		close(x);
	}

	fprintf(stderr, "F-CPU disassembler v0.1\n");
	fprintf(stderr, "Copyright (C) 2002, 2003 Michael \"Tired\" Riepe\n\n");

	for (i = 0; i < st.st_size; i += 4) {
		unsigned insn = 0;

#if INSTRUCTION_BIG_ENDIAN
		insn = (insn << 8) | p[i+0];
		insn = (insn << 8) | p[i+1];
		insn = (insn << 8) | p[i+2];
		insn = (insn << 8) | p[i+3];
#else
		insn = (insn << 8) | p[i+3];
		insn = (insn << 8) | p[i+2];
		insn = (insn << 8) | p[i+1];
		insn = (insn << 8) | p[i+0];
#endif
		show(i, insn);
	}
	exit(0);
}
