-- regfile_interleaved.vhdl -- F-CPU Register File Implementations
-- Copyright (C) 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: regfile_interleaved.vhdl,v 1.3 2003/06/27 20:33:27 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;

architecture Interleaved_SRAM of RegFile is
	constant LOG_BANKS : integer := 3;	-- do not change
	constant NUM_BANKS : integer := 2 ** LOG_BANKS;

	use work.SRAM;	-- make instantiated entity visible

	component SRAM
		generic (
			LOG_LINES : integer := 3;	-- 8 lines
			WIDTH : integer := 8		-- 8 bits/line
		);
		port (
			RA : in std_ulogic_vector(LOG_LINES-1 downto 0);
			WA : in std_ulogic_vector(LOG_LINES-1 downto 0);
			RE : in std_ulogic;
			WE : in std_ulogic;
			WD : in std_ulogic_vector(WIDTH-1 downto 0);
		--
			RD : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	function decode3to8 (X : in std_ulogic_vector(2 downto 0);
						 E : in std_ulogic)
						return std_ulogic_vector is
		variable yy : std_ulogic_vector(7 downto 0);
	begin
		yy(0) := not X(2) and not X(1) and not X(0) and E;
		yy(1) := not X(2) and not X(1) and     X(0) and E;
		yy(2) := not X(2) and     X(1) and not X(0) and E;
		yy(3) := not X(2) and     X(1) and     X(0) and E;
		yy(4) :=     X(2) and not X(1) and not X(0) and E;
		yy(5) :=     X(2) and not X(1) and     X(0) and E;
		yy(6) :=     X(2) and     X(1) and not X(0) and E;
		yy(7) :=     X(2) and     X(1) and     X(0) and E;
		return yy;
	end decode3to8;

	type F_REGISTER_DATA is array(NUM_BANKS-1 downto 0) of F_VECTOR;

	function mux8 (X : in F_REGISTER_DATA;
				   Y : in std_ulogic_vector(2 downto 0)) return F_VECTOR is
	begin
		case Y is
			when "000" => return X(0);
			when "001" => return X(1);
			when "010" => return X(2);
			when "011" => return X(3);
			when "100" => return X(4);
			when "101" => return X(5);
			when "110" => return X(6);
			when "111" => return X(7);
			when others => return (others => 'X');
		end case;
	end mux8;

	signal WD, RD : F_REGISTER_DATA;
	signal WA, RA : std_ulogic_vector(3*NUM_BANKS-1 downto 0);
	signal WE, RE : std_ulogic_vector(NUM_BANKS-1 downto 0);
begin
	banks : for bank in NUM_BANKS-1 downto 0 generate
		inst : SRAM
			generic map (LOG_LINES => LOG_BANKS, WIDTH => UMAX)
			port map (
				RA => RA(3*bank+2 downto 3*bank),
				WA => WA(3*bank+2 downto 3*bank),
				RD => RD(bank),
				WD => WD(bank),
				RE => RE(bank),
				WE => WE(bank)
			);
	end generate;

	-- write logic
	write : process (WD0, WD1, WA0, WA1, WE0, WE1)
		variable w0, w1 : std_ulogic_vector(NUM_BANKS-1 downto 0);
	begin
		w0 := decode3to8(WA0(2 downto 0), WE0);
		w1 := decode3to8(WA1(2 downto 0), WE1);
		WE <= w0 or w1;

		for bank in NUM_BANKS-1 downto 0 loop
			if to_X01(w0(bank)) = '1' then
				WA(3*bank+2 downto 3*bank) <= WA0(5 downto 3);
				WD(bank) <= WD0;
			else
				WA(3*bank+2 downto 3*bank) <= WA1(5 downto 3);
				WD(bank) <= WD1;
			end if;
		end loop;
	end process;

	-- read logic
	read : process (RD, RA0, RA1, RA2, RE0, RE1, RE2)
		variable r0, r1, r2 : std_ulogic_vector(NUM_BANKS-1 downto 0);
	begin
		r0 := decode3to8(RA0(2 downto 0), RE0);
		r1 := decode3to8(RA1(2 downto 0), RE1);
		r2 := decode3to8(RA2(2 downto 0), RE2);
		RE <= r0 or r1 or r2;

		for bank in NUM_BANKS-1 downto 0 loop
			if to_X01(r0(bank)) = '1' then
				RA(3*bank+2 downto 3*bank) <= RA0(5 downto 3);
			elsif to_X01(r1(bank)) = '1' then
				RA(3*bank+2 downto 3*bank) <= RA1(5 downto 3);
			else
				RA(3*bank+2 downto 3*bank) <= RA2(5 downto 3);
			end if;
		end loop;

		if to_X01(RE0) = '1' then
			RD0 <= mux8(RD, to_X01(RA0(2 downto 0)));
		else
			RD0 <= (others => 'Z');
		end if;
		if to_X01(RE1) = '1' then
			RD1 <= mux8(RD, to_X01(RA1(2 downto 0)));
		else
			RD1 <= (others => 'Z');
		end if;
		if to_X01(RE2) = '1' then
			RD2 <= mux8(RD, to_X01(RA2(2 downto 0)));
		else
			RD2 <= (others => 'Z');
		end if;
	end process;
end Interleaved_SRAM;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
