-- rop2_test.vhdl -- Testbench for ROP2 Execution Unit
-- Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: rop2_test.vhdl,v 1.4 2003/06/27 20:33:26 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

use work.fcpu_config.all;

entity Rop2_test is
	generic (WIDTH : natural := 64);
end Rop2_test;

architecture Arch_1 of Rop2_test is
	use work.Rop2;	-- make instantiated entity visible

	component Rop2
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			C : in std_ulogic_vector(WIDTH-1 downto 0);
			Func : in std_ulogic_vector(2 downto 0);
			Mode : in std_ulogic_vector(1 downto 0);
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Z : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal M : std_ulogic_vector(7 downto 0);
	signal A, B, C, Y, Z : std_ulogic_vector(WIDTH-1 downto 0);
	signal Clk : std_ulogic := '0';
	signal Rst : std_ulogic := '0';
	signal En : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_vector (lbl : string;
							x : std_ulogic_vector;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_vector;

	procedure print_signals is
	begin
		print_vector("A", A);
		print_vector("B", B);
		print_vector("C", C);
		print_vector("M", M);
		print_vector("Y", Y);
		print_vector("Z", Z);
	end print_signals;

	procedure do_report (lbl : in string; x, y : in std_ulogic_vector) is
	begin
		writestr("WHOA THERE!!!");
		print_signals;
		print_vector(lbl, x);
		print_vector(lbl, y, " /= ");
	end do_report;

	procedure check_numeric (lbl : in string;
							 x : in std_ulogic_vector;
							 y : in natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y rem 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : in string;
						   a, b : in std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length
			report "bad args in check_logic" severity failure;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : Rop2
		generic map (WIDTH => WIDTH)
		port map (
			A => A,
			B => B,
			C => C,
			Func => M(7 downto 5),
			Mode => M(4 downto 3),
			U => M(2 downto 0),
			Clk => Clk,
			Rst => Rst,
			En => En,
		--
			Y => Y,
			Z => Z
		);

	-- driver process
	process
		constant std_0 : std_ulogic := '0';
		constant std_1 : std_ulogic := '1';

		procedure print_mode (simd : in natural) is
			variable lout : line;
		begin
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
		end print_mode;

		procedure test_direct is
			variable av, bv, cv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable lout : line;
		begin
			M <= "XXXXXXXX";
			M(4 downto 3) <= ROP2_DIRECT_MODE;
			writestr("*** testing rop2 direct instructions ***");
			av := (others => 'X');
			bv := (others => 'X');
			cv := (others => 'X');
			tmp := (others => '-');
			for i in 0 to WIDTH-1 loop
				for aa in std_0 to std_1 loop
					for bb in std_0 to std_1 loop
						av(i) := aa;
						bv(i) := bb;
						for f in 0 to 7 loop
							M(7 downto 5) <= std_ulogic_vector(to_unsigned(f, 3));
							case f is
								when 0 => tmp(i) := aa and bb;
								when 1 => tmp(i) := aa and not bb;
								when 2 => tmp(i) := aa xor bb;
								when 3 => tmp(i) := aa or bb;
								when 4 => tmp(i) := aa nor bb;
								when 5 => tmp(i) := aa xor not bb;
								when 6 => tmp(i) := aa or not bb;
								when 7 => tmp(i) := aa nand bb;
								when others => assert false;
							end case;
							A <= av; B <= bv; C <= cv; wait for 1 ns;
							check_logic("Y", Y, tmp);
						end loop;
					end loop;
				end loop;
				av(i) := 'X';
				bv(i) := 'X';
				tmp(i) := '-';
			end loop;
		end test_direct;

		procedure test_and is
			variable av, bv, cv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right, simd : natural;
		begin
			M <= "XXXXXXXX";
			M(7 downto 5) <= ROP2_AND;	-- arbitrary choice
			M(4 downto 3) <= ROP2_AND_MODE;
			writestr("*** testing and combine mode ***");
			for gran in 0 to 3 loop
				simd := 2 ** (gran + 3);
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				print_mode(simd);
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk * simd;
					left := right + simd - 1;
					av := (others => 'X');
					bv := (others => 'X');
					cv := (others => 'X');
					tmp := (others => '-');
					av(left downto right) := (left downto right => '1');
					bv(left downto right) := (left downto right => '1');
					tmp(left downto right) := (left downto right => '1');
					A <= av; B <= bv; C <= cv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					tmp(left downto right) := (left downto right => '0');
					for i in right to left loop
						av(i) := '0';
						bv(i) := '0';
						A <= av; B <= bv; C <= cv; wait for 1 ns;
						check_logic("Y", Y, tmp);
						av(i) := '1';
						bv(i) := '1';
					end loop;
					av(left downto right) := (left downto right => 'X');
					bv(left downto right) := (left downto right => 'X');
					tmp(left downto right) := (left downto right => '-');
				end loop;
			end loop;
		end test_and;

		procedure test_or is
			variable av, bv, cv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right, simd : natural;
		begin
			M <= "XXXXXXXX";
			M(7 downto 5) <= ROP2_AND;	-- arbitrary choice
			M(4 downto 3) <= ROP2_OR_MODE;
			writestr("*** testing or combine mode ***");
			for gran in 0 to 3 loop
				simd := 2 ** (gran + 3);
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				print_mode(simd);
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk * simd;
					left := right + simd - 1;
					av := (others => 'X');
					bv := (others => 'X');
					cv := (others => 'X');
					tmp := (others => '-');
					av(left downto right) := (left downto right => '0');
					bv(left downto right) := (left downto right => '0');
					tmp(left downto right) := (left downto right => '0');
					A <= av; B <= bv; C <= cv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					tmp(left downto right) := (left downto right => '1');
					for i in right to left loop
						av(i) := '1';
						bv(i) := '1';
						A <= av; B <= bv; C <= cv; wait for 1 ns;
						check_logic("Y", Y, tmp);
						av(i) := '0';
						bv(i) := '0';
					end loop;
					av(left downto right) := (left downto right => 'X');
					bv(left downto right) := (left downto right => 'X');
					tmp(left downto right) := (left downto right => '-');
				end loop;
			end loop;
		end test_or;

		procedure test_mux is
			variable av, bv, cv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		begin
			M <= "XXXXXXXX";
			M(7 downto 5) <= ROP2_AND;	-- same opcode as AND
			M(4 downto 3) <= ROP2_MUX_MODE;
			writestr("*** testing mux instruction ***");
			av := (others => 'X');
			bv := (others => 'X');
			cv := (others => 'X');
			tmp := (others => '-');
			for i in 0 to WIDTH-1 loop
				for aa in std_0 to std_1 loop
					for bb in std_0 to std_1 loop
						for cc in std_0 to std_1 loop
							av(i) := aa;
							bv(i) := bb;
							cv(i) := cc;
							if bb = '1' then
								tmp(i) := aa;
							else
								tmp(i) := cc;
							end if;
							A <= av; B <= bv; C <= cv; wait for 1 ns;
							check_logic("Y", Y, tmp);
						end loop;
					end loop;
				end loop;
				av(i) := 'X';
				bv(i) := 'X';
				cv(i) := 'X';
				tmp(i) := '-';
			end loop;
		end test_mux;

		procedure test_muxr is
			variable av, bv, cv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		begin
			M <= "XXXXXXXX";
			M(7 downto 5) <= ROP2_ANDN;	-- same opcode as ANDN
			M(4 downto 3) <= ROP2_MUX_MODE;
			writestr("*** testing muxr instruction ***");
			av := (others => 'X');
			bv := (others => 'X');
			cv := (others => 'X');
			tmp := (others => '-');
			for i in 0 to WIDTH-1 loop
				for aa in std_0 to std_1 loop
					for bb in std_0 to std_1 loop
						for cc in std_0 to std_1 loop
							av(i) := aa;
							bv(i) := bb;
							cv(i) := cc;
							if bb = '1' then
								tmp(i) := cc;
							else
								tmp(i) := aa;
							end if;
							A <= av; B <= bv; C <= cv; wait for 1 ns;
							check_logic("Y", Y, tmp);
						end loop;
					end loop;
				end loop;
				av(i) := 'X';
				bv(i) := 'X';
				cv(i) := 'X';
				tmp(i) := '-';
			end loop;
		end test_muxr;

		procedure test_bitop is
			variable av, bv, cv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right, simd : natural;
			variable lout : line;
		begin
			M <= "XXXXXXXX";
			M(4 downto 3) <= ROP2_DIRECT_MODE;
			writestr("*** testing bitop instructions ***");
			for gran in 0 to 3 loop
				simd := 2 ** (gran + 3);
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				print_mode(simd);
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk * simd;
					left := right + simd - 1;
					av := (others => 'X');
					bv := (others => 'X');
					cv := (others => 'X');
					tmp := (others => '-');
					for i in 0 to simd-1 loop
						bv(right+gran+2 downto right) :=
							std_ulogic_vector(to_unsigned(i, gran+3));
						for aa in std_0 to std_1 loop
							av(right+i) := aa;
							for f in 0 to 7 loop
								M(7 downto 5) <= std_ulogic_vector(to_unsigned(f, 3));
								tmp(left downto right) := (left downto right => '-');
								case f is
									when 0 =>
										tmp(left downto right) := (left downto right => '0');
										tmp(right+i) := aa;
									when 1|4 =>
										tmp(right+i) := '0';
									when 2 =>
										tmp(right+i) := not aa;
									when 3 =>
										tmp(right+i) := '1';
									when 5 =>
										tmp(right+i) := aa;
									when 6 =>
										tmp(left downto right) := (left downto right => '1');
										tmp(right+i) := aa;
									when 7 =>
										tmp(left downto right) := (left downto right => '1');
										tmp(right+i) := not aa;
									when others =>
										assert false;
								end case;
								A <= av; B <= bv; C <= cv; wait for 1 ns;
								check_logic("Z", Z, tmp);
							end loop;
							av(right+i) := 'X';
						end loop;
					end loop;
				end loop;
			end loop;
		end test_bitop;
	begin
		test_direct;
		test_and;
		test_or;
		test_mux;
		test_muxr;
		test_bitop;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
