-- inc64_test.vhdl -- Testbench for 64-Bit Increment Unit
-- Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: inc64_test.vhdl,v 1.5 2003/06/27 20:33:26 michael Exp $

--pragma synthesis_off

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

--use work.Bit_Manipulation.all;

entity Inc64_test is
	generic (WIDTH : natural := 64);
end Inc64_test;

architecture Arch_1 of Inc64_test is
	use work.Inc64;	-- make instantiated entity visible

	component Inc64
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			Inc : in std_ulogic;
			Dec : in std_ulogic;
			Neg : in std_ulogic;
			Xabs : in std_ulogic;	-- `Abs' is reserved
			Lsb1 : in std_ulogic;
			Lsb0 : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			X : out std_ulogic_vector(WIDTH-1 downto 0);
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Z : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal M : std_ulogic_vector(8 downto 0) := (others => '0');
	signal A, X, Y, Z : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal Clk : std_ulogic := '0';
	signal Rst : std_ulogic := '0';
	signal En : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_vector (lbl : string;
							x : std_ulogic_vector;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_vector;

	procedure print_signals is
	begin
		print_vector("A", A);
		print_vector("M", M);
		print_vector("X", X);
		print_vector("Y", Y);
		print_vector("Z", Z);
	end print_signals;

	procedure do_error (lbl : string;
						a_x, a_y : std_ulogic_vector) is
	begin
		writestr("WHOA THERE!!!");
		print_signals;
		print_vector(lbl, a_x);
		print_vector(lbl, a_y, " /= ");
	end do_error;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y rem 2**x'length, x'length));
		if x /= tmp then
			do_error(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length
			report "bad args in check_logic" severity failure;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_error(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : Inc64
		generic map (WIDTH => WIDTH)
		port map (
			A => A,
			Inc => M(0),
			Dec => M(1),
			Neg => M(2),
			Xabs => M(3),
			Lsb1 => M(4),
			Lsb0 => M(5),
			U => M(8 downto 6),
			Clk => Clk,
			Rst => Rst,
			En => En,
			X => X,
			Y => Y,
			Z => Z
		);

	-- driver process
	process
		constant std_ulogic_0 : std_ulogic := '0';
		constant std_ulogic_1 : std_ulogic := '1';

		procedure do_clock is
		begin
			wait for 1 ns;
			Clk <= '1';
			wait for 1 ns;
			Clk <= '0';
		end do_clock;

		procedure print_mode (simd : in natural) is
			variable lout : line;
		begin
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
		end print_mode;

		function myinv (X : in std_ulogic_vector) return std_ulogic_vector is
			constant L : natural := X'length;
			variable xx : std_ulogic_vector(L-1 downto 0);
		begin
			xx := X;
			for i in L-1 downto 0 loop
				if xx(i) /= '-' then
					xx(i) := not xx(i);
				end if;
			end loop;
			return xx;
		end myinv;

		procedure test_inc is
			variable av : std_ulogic_vector(WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing inc ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX000001";
				for i in 6 to 8 loop
					if i < gran + 6 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					tmp := (others => '-');
					for index in right to left loop
						av(index) := '0';
						tmp(index) := '1';
						A <= av; wait for 1 ns;
						check_logic("X", X, tmp);
						do_clock;
						check_logic("Y", Y, tmp);
						av(index) := '1';
						tmp(index) := '0';
						A <= av; wait for 1 ns;
						check_logic("X", X, tmp);
						do_clock;
						check_logic("Y", Y, tmp);
					end loop;
				end loop;
			end loop;
		end test_inc;

		procedure test_dec is
			variable av : std_ulogic_vector(WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing dec ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX000010";
				for i in 6 to 8 loop
					if i < gran + 6 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					tmp := (others => '-');
					for index in right to left loop
						av(index) := '1';
						tmp(index) := '0';
						A <= av; wait for 1 ns;
						check_logic("X", X, tmp);
						do_clock;
						check_logic("Y", Y, tmp);
						av(index) := '0';
						tmp(index) := '1';
						A <= av; wait for 1 ns;
						check_logic("X", X, tmp);
						do_clock;
						check_logic("Y", Y, tmp);
					end loop;
				end loop;
			end loop;
		end test_dec;

		procedure test_neg is
			variable av : std_ulogic_vector(WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing neg ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX000100";
				for i in 6 to 8 loop
					if i < gran + 6 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					tmp := (others => '-');
					for index in right to left loop
						av(index) := '1';
						tmp(index) := '1';
						A <= av; wait for 1 ns;
						check_logic("X", X, tmp);
						do_clock;
						check_logic("Y", Y, tmp);
						av(index) := '0';
						tmp(index) := '0';
						A <= av; wait for 1 ns;
						check_logic("X", X, tmp);
						do_clock;
						check_logic("Y", Y, tmp);
					end loop;
				end loop;
			end loop;
		end test_neg;

		procedure test_abs is
			variable av : std_ulogic_vector(WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing abs ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX001000";
				for i in 6 to 8 loop
					if i < gran + 6 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					tmp := (others => '-');
					av(left) := '0';
					tmp(left) := '0';
					for index in left-1 downto right loop
						for bit in std_ulogic_0 to std_ulogic_1 loop
							av(index) := bit;
							tmp(index) := bit;
							A <= av; do_clock;
							check_logic("Y", Y, tmp);
							av(index) := 'X';
							tmp(index) := '-';
						end loop;
					end loop;
					av := (others => 'X');
					tmp := (others => '-');
					av(left downto right) := (left downto right => '0');
					tmp(left downto right) := (left downto right => '0');
					-- special case: `sign surprise'
					av(left) := '1';
					tmp(left) := '1';
					A <= av; do_clock;
					check_logic("Y", Y, tmp);
					-- other negative numbers
					tmp(left) := '0';
					for index in left-1 downto right loop
						av(index) := '1';
						tmp(index) := '1';
						A <= av; do_clock;
						check_logic("Y", Y, tmp);
						tmp(index) := '0';
					end loop;
				end loop;
			end loop;
		end test_abs;

		procedure test_nabs is
			variable av : std_ulogic_vector(WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing nabs ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX001000";
				for i in 6 to 8 loop
					if i < gran + 6 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					tmp := (others => '-');
					av(left) := '1';
					tmp(left) := '1';
					for index in left-1 downto right loop
						for bit in std_ulogic_0 to std_ulogic_1 loop
							av(index) := bit;
							tmp(index) := bit;
							A <= av; do_clock;
							check_logic("Z", Z, tmp);
							av(index) := 'X';
							tmp(index) := '-';
						end loop;
					end loop;
					av := (others => 'X');
					tmp := (others => '-');
					av(left downto right) := (left downto right => '0');
					tmp(left downto right) := (left downto right => '0');
					tmp(left) := '1';
					for index in left-1 downto right loop
						av(index) := '1';
						tmp(index) := '1';
						A <= av; do_clock;
						check_logic("Z", Z, tmp);
						av(index) := '0';
					end loop;
				end loop;
			end loop;
		end test_nabs;

		procedure test_lsb (bit : in std_ulogic) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
			variable lout : line;
		begin
			write(lout, string'("*** testing lsb"));
			write(lout, bit);
			write(lout, string'(" ***"));
			writeline(output, lout);
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXXXX0000";
				M(4) <= bit;
				M(5) <= not bit;
				for i in 6 to 8 loop
					if i < gran + 6 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					av(left downto right) := (left downto right => not bit);
					tmp := (others => '-');
					tmp(left downto right) := (left downto right => '0');
					A <= av; do_clock;
					check_logic("Y", Y, tmp);
					check_logic("Z", Z, tmp);
					for index in left downto right loop
						av(index) := bit;
						tmp(left downto right) := std_ulogic_vector(
							to_unsigned(index - right + 1, simd));
						A <= av; do_clock;
						check_logic("Y", Y, tmp);
						tmp(left downto right) := (left downto right => '0');
						tmp(index) := '1';
						check_logic("Z", Z, tmp);
						av(index) := 'X';
					end loop;
				end loop;
			end loop;
		end test_lsb;
	begin
		test_inc;
		test_dec;
		test_neg;
		test_abs;
		test_nabs;
		test_lsb('0');
		test_lsb('1');

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

--pragma synthesis_on

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
