-- im64test4.vhdl - F-CPU Integer Multiplication Unit Testbench #4
-- Copyright (C) 2000 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: im64test4.vhdl,v 1.6 2003/06/27 20:33:26 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

use work.IMul64_testconf.all;

entity IMul64_test4 is
end IMul64_test4;

architecture Arch_1 of IMul64_test4 is
	use work.IMul64;	-- make instantiated entity visible

	component IMul64
		generic (
			WIDTH : natural := 64;	-- DO NOT CHANGE!
			PIPELINED : integer := 0
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			X : in std_ulogic_vector(WIDTH-1 downto 0);
			Sig : in std_ulogic;
			Mac : in std_ulogic_vector(2 downto 0);
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y08l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=18 (3 stages)
			Y08h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=18 (3 stages)
			Y16l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=24 (4 stages)
			Y16h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=24 (4 stages)
			Y32l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=29 (5 stages)
			Y32h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=29 (5 stages)
			Y64l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=34 (6 stages)
			Y64h : out std_ulogic_vector(WIDTH-1 downto 0)	-- d=34 (6 stages)
		);
	end component;

	constant WIDTH : natural := 64;

	signal A, B : std_ulogic_vector(WIDTH-1 downto 0);
	signal X : std_ulogic_vector(WIDTH-1 downto 0);
	signal M : std_ulogic_vector(6 downto 0);
	signal Y08l, Y08h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y16l, Y16h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y32l, Y32h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y64l, Y64h : std_ulogic_vector(WIDTH-1 downto 0);
	signal L, H : std_ulogic_vector(WIDTH-1 downto 0);

	signal Clk, Rst, En : std_ulogic;

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 x_1, x_2 : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("X := ")); write(lout, X); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("H := ")); write(lout, H); writeline(output, lout);
		write(lout, string'("L := ")); write(lout, L); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x_1);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, x_2);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	Clk <= '0'; Rst <= '0'; En <= '1';

	mut: IMul64
		generic map (PIPELINED => 0)
		port map (
			A => A, B => B, X => X, Sig => M(0),
			Mac => M(3 downto 1), U => M(6 downto 4),
			Clk => Clk, Rst => Rst, En => En,
			Y08l => Y08l, Y08h => Y08h,
			Y16l => Y16l, Y16h => Y16h,
			Y32l => Y32l, Y32h => Y32h,
			Y64l => Y64l, Y64h => Y64h
		);

	-- output selector
	L <= Y64l when M(6) = '1'
	else Y32l when M(5) = '1'
	else Y16l when M(4) = '1'
	else Y08l;
	H <= Y64h when M(6) = '1'
	else Y32h when M(5) = '1'
	else Y16h when M(4) = '1'
	else Y08h;

	test : process
		constant bits : natural := 4;
		variable prod, res : integer;
		variable lout : line;
		variable av, bv, xv : std_ulogic_vector(WIDTH-1 downto 0);
		variable pv, rv : std_ulogic_vector(2*WIDTH-1 downto 0);
		variable gran, left, right : natural;
	begin
		A <= (others => '0');
		B <= (others => '0');
		for level in MIN_LEVEL to MAX_LEVEL loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit unsigned amac ***"));
			writeline(output, lout);
			M <= "0001000";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;
			av := (others => '0');
			bv := (others => '0');
			xv := (others => '0');
			pv := (others => '0');
			pv(WIDTH) := '1';
			for ai in 1 to 2**bits-1 loop
				av(gran-1 downto 0) :=
					std_ulogic_vector(to_unsigned(ai, gran));
				for bi in 1 to 2**bits-1 loop
					bv(gran-1 downto 0) :=
						std_ulogic_vector(to_unsigned(bi, gran));
					xv(gran-1 downto 0) :=
						std_ulogic_vector(to_signed(-ai*bi, gran));
					A <= av;
					B <= bv;
					X <= xv;
					wait for 1 ns;
					rv := H & L;
					check_logic("Y", rv, pv);
				end loop;
			end loop;
		end loop;

		writestr("*** Simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
