-- idiv64_test1.vhdl - F-CPU Integer Divider Testbench #1
-- Copyright (C) 2000 - 2004 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: idiv64_test1.vhdl,v 1.13 2004/01/23 15:30:52 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

use work.Bit_Manipulation.all;

entity IDiv64_test1 is
	generic (WIDTH : natural := 64);
end IDiv64_test1;

architecture Arch_1 of IDiv64_test1 is
	use work.IDiv64;	-- make instantiated entity visible

	component IDiv64
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			Sig : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Z : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal A, B : std_ulogic_vector(WIDTH-1 downto 0);
	signal M : std_ulogic_vector(3 downto 0);
	signal Y, Z : std_ulogic_vector(WIDTH-1 downto 0);
	signal Clk, Rst, En : std_ulogic;

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string; x_1, x_2 : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("Y := ")); write(lout, Y); writeline(output, lout);
		write(lout, string'("Z := ")); write(lout, Z); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x_1);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, x_2);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	function match (a, b : std_ulogic_vector) return boolean is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			return false;
		end loop;
		return true;
	end match;

	procedure check_logic (lbl : string; a, b : std_ulogic_vector) is
	begin
		if not match(a, b) then
			do_report(lbl, a, b);
			wait;
		end if;
	end check_logic;

	procedure pulse (signal x : out std_ulogic; n : in natural := 1) is
	begin
		for i in 1 to n loop
			wait for 1 ns; x <= '1';
			wait for 1 ns; x <= '0';
		end loop;
	end pulse;
begin
	mut: IDiv64
		generic map (WIDTH => WIDTH)
		port map (
			A => A, B => B, Sig => M(0), U => M(3 downto 1),
			Clk => Clk, Rst => Rst, En => En, Y => Y, Z => Z
		);

	test : process
		variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		variable simd, left, right : natural;
		variable lout : line;
	begin
		Clk <= '0'; Rst <= '0'; En <= '0'; pulse(Rst);
		for gran in 0 to 3 loop
			simd := 2 ** (gran + 3);
			M <= "XXX0";
			for i in 1 to 3 loop
				if i - 1 >= gran then
					M(i) <= '0';
				else
					M(i) <= '1';
				end if;
			end loop;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit unsigned mode ***"));
			writeline(output, lout);
			for chunk in 0 to WIDTH/simd-1 loop
				right := chunk * simd;
				left := right + simd - 1;
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				for i in simd-1 downto 1-simd loop
					av(left downto right) := (left downto right => '0');
					bv(left downto right) := (left downto right => '0');
					tmp(left downto right) := (left downto right => '0');
					if i >= 0 then
						av(right + i) := '1';
						bv(right) := '1';
						tmp(right + i) := '1';
					else
						av(right) := '1';
						bv(right - i) := '1';
					end if;
					A <= av; B <= bv;
					En <= '1'; pulse(Clk);
					En <= '0'; pulse(Clk, simd + 3);
					check_logic("Y", Y, tmp);
				end loop;
			end loop;
			M(0) <= '1';
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit signed mode (+/+) ***"));
			writeline(output, lout);
			for chunk in 0 to WIDTH/simd-1 loop
				right := chunk * simd;
				left := right + simd - 1;
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				for i in simd-2 downto 2-simd loop
					av(left downto right) := (left downto right => '0');
					bv(left downto right) := (left downto right => '0');
					tmp(left downto right) := (left downto right => '0');
					if i >= 0 then
						av(right + i) := '1';
						bv(right) := '1';
						tmp(right + i) := '1';
					else
						av(right) := '1';
						bv(right - i) := '1';
					end if;
					A <= av; B <= bv;
					En <= '1'; pulse(Clk);
					En <= '0'; pulse(Clk, simd + 3);
					check_logic("Y", Y, tmp);
				end loop;
			end loop;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit signed mode (-/-) ***"));
			writeline(output, lout);
			for chunk in 0 to WIDTH/simd-1 loop
				right := chunk * simd;
				left := right + simd - 1;
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				for i in simd-1 downto 1-simd loop
					av(left downto right) := (left downto right => '0');
					bv(left downto right) := (left downto right => '0');
					tmp(left downto right) := (left downto right => '0');
					if i >= 0 then
						av(right + i) := '1';
						bv(right) := '1';
						tmp(right + i) := '1';
					else
						av(right) := '1';
						bv(right - i) := '1';
					end if;
					av(left downto right) := neg(av(left downto right));
					bv(left downto right) := neg(bv(left downto right));
					A <= av; B <= bv;
					En <= '1'; pulse(Clk);
					En <= '0'; pulse(Clk, simd + 3);
					check_logic("Y", Y, tmp);
				end loop;
			end loop;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit signed mode (+/-) ***"));
			writeline(output, lout);
			for chunk in 0 to WIDTH/simd-1 loop
				right := chunk * simd;
				left := right + simd - 1;
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				for i in simd-2 downto 1-simd loop
					av(left downto right) := (left downto right => '0');
					bv(left downto right) := (left downto right => '0');
					tmp(left downto right) := (left downto right => '0');
					if i >= 0 then
						av(right + i) := '1';
						bv(right) := '1';
						tmp(right + i) := '1';
					else
						av(right) := '1';
						bv(right - i) := '1';
					end if;
					bv(left downto right) := neg(bv(left downto right));
					tmp(left downto right) := neg(tmp(left downto right));
					A <= av; B <= bv;
					En <= '1'; pulse(Clk);
					En <= '0'; pulse(Clk, simd + 3);
					check_logic("Y", Y, tmp);
				end loop;
			end loop;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit signed mode (-/+) ***"));
			writeline(output, lout);
			for chunk in 0 to WIDTH/simd-1 loop
				right := chunk * simd;
				left := right + simd - 1;
				av := (others => 'X');
				bv := (others => 'X');
				tmp := (others => '-');
				for i in simd-1 downto 2-simd loop
					av(left downto right) := (left downto right => '0');
					bv(left downto right) := (left downto right => '0');
					tmp(left downto right) := (left downto right => '0');
					if i >= 0 then
						av(right + i) := '1';
						bv(right) := '1';
						tmp(right + i) := '1';
					else
						av(right) := '1';
						bv(right - i) := '1';
					end if;
					av(left downto right) := neg(av(left downto right));
					tmp(left downto right) := neg(tmp(left downto right));
					A <= av; B <= bv;
					En <= '1'; pulse(Clk);
					En <= '0'; pulse(Clk, simd + 3);
					check_logic("Y", Y, tmp);
				end loop;
			end loop;
		end loop;

		writestr("*** Simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
