-- omega_network_test.vhdl - omega network testbench
-- Copyright (C) 2002 - 2004 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: omega_network_test.vhdl,v 1.1 2004/01/25 05:41:27 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;
use work.Bit_Manipulation.all;
use work.Omega_Network.all;

entity Omega_Network_test is
end Omega_Network_test;

architecture Arch_1 of Omega_Network_test is
	constant WIDTH : natural := 64;

	signal A : std_ulogic_vector(WIDTH-1 downto 0);
	signal X : std_ulogic_vector(WIDTH/2-1 downto 0);
	signal Y : std_ulogic_vector(WIDTH-1 downto 0);

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_vector (lbl : string;
							x : std_ulogic_vector;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_vector;

	procedure print_signals is
		variable lout : line;
	begin
		print_vector("A", A);
		print_vector("X", X);
		print_vector("Y", Y);
	end print_signals;

	procedure do_report (lbl : string;
						 x, y : std_ulogic_vector) is
	begin
		writestr("WHOA THERE!!!");
		print_signals;
		print_vector(lbl, x);
		print_vector(lbl, y, " /= ");
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- driver process
	run : process
		constant std_ulogic_0 : std_ulogic := '0';
		constant std_ulogic_1 : std_ulogic := '1';
		variable av, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		variable xv : std_ulogic_vector(WIDTH/2-1 downto 0);
		variable j, k : natural;
	begin
		writestr("*** testing omega network stage (forward) ***");
		av := (others => 'X');
		xv := (others => 'X');
		tmp := (others => '-');
		for i in 0 to WIDTH-1 loop
			-- i->j routing
			j := 2 * i;
			if j < WIDTH then
				k := j + 1;
			else
				k := j - WIDTH;
				j := k + 1;
			end if;
			-- straight connection
			xv(j / 2) := '0';
			for bit in std_ulogic_0 to std_ulogic_1 loop
				av(i) := bit;
				tmp(j) := bit;
				A <= av; X <= xv; Y <= omega_F_1(av, xv);
				wait for 1 ns; check_logic("Y", Y, tmp);
				tmp(j) := '-';
			end loop;
			-- X-ed connection
			xv(j / 2) := '1';
			for bit in std_ulogic_0 to std_ulogic_1 loop
				av(i) := bit;
				tmp(k) := bit;
				A <= av; X <= xv; Y <= omega_F_1(av, xv);
				wait for 1 ns; check_logic("Y", Y, tmp);
				tmp(k) := '-';
			end loop;
			-- cleanup
			xv(j / 2) := 'X';
			av(i) := 'X';
		end loop;

		writestr("*** testing omega network stage (reversed) ***");
		av := (others => 'X');
		xv := (others => 'X');
		tmp := (others => '-');
		for i in 0 to WIDTH-1 loop
			-- i<-j routing
			j := 2 * i;
			if j < WIDTH then
				k := j + 1;
			else
				k := j - WIDTH;
				j := k + 1;
			end if;
			-- straight connection
			xv(j / 2) := '0';
			for bit in std_ulogic_0 to std_ulogic_1 loop
				av(j) := bit;
				tmp(i) := bit;
				A <= av; X <= xv; Y <= omega_R_1(av, xv);
				wait for 1 ns; check_logic("Y", Y, tmp);
				av(j) := 'X';
			end loop;
			-- X-ed connection
			xv(j / 2) := '1';
			for bit in std_ulogic_0 to std_ulogic_1 loop
				av(k) := bit;
				tmp(i) := bit;
				A <= av; X <= xv; Y <= omega_R_1(av, xv);
				wait for 1 ns; check_logic("Y", Y, tmp);
				av(k) := 'X';
			end loop;
			-- cleanup
			xv(j / 2) := 'X';
			tmp(i) := '-';
		end loop;

		writestr("*** testing unsigned normalize function ***");
		av := (others => 'X');
		tmp := (others => '-');
		tmp(WIDTH-1) := '1';
		for i in 1 to WIDTH loop
			av(WIDTH-i) := '1';
			A <= av; Y <= normalize(av, '0');
			wait for 1 ns; check_logic("Y", Y, tmp);
			for j in 0 to WIDTH-i-1 loop
				assert av(j) = 'X';
				for bit in std_ulogic_0 to std_ulogic_1 loop
					av(j) := bit;
					tmp(j+i-1) := bit;
					A <= av; Y <= normalize(av, '0');
					wait for 1 ns; check_logic("Y", Y, tmp);
				end loop;
				av(j) := 'X';
				tmp(j+i-1) := '-';
			end loop;
			av(WIDTH-i) := '0';
			tmp(i-1) := '0';
		end loop;
		A <= av; Y <= normalize(av, '0');
		wait for 1 ns; check_logic("Y", Y, tmp);

		writestr("*** testing signed normalize function ***");
		writestr("*** testing positive arguments ***");
		av := (others => 'X');
		av(WIDTH-1) := '0';
		tmp := (others => '-');
		tmp(WIDTH-1) := '0';
		tmp(WIDTH-2) := '1';
		for i in 2 to WIDTH loop
			av(WIDTH-i) := '1';
			A <= av; Y <= normalize(av, '1');
			wait for 1 ns; check_logic("Y", Y, tmp);
			for j in 0 to WIDTH-i-1 loop
				assert av(j) = 'X';
				for bit in std_ulogic_0 to std_ulogic_1 loop
					av(j) := bit;
					tmp(j+i-2) := bit;
					A <= av; Y <= normalize(av, '1');
					wait for 1 ns; check_logic("Y", Y, tmp);
				end loop;
				av(j) := 'X';
				tmp(j+i-2) := '-';
			end loop;
			av(WIDTH-i) := '0';
			tmp(i-2) := '0';
		end loop;
		A <= av; Y <= normalize(av, '1');
		wait for 1 ns; check_logic("Y", Y, tmp);
		writestr("*** testing negative arguments ***");
		av := (others => 'X');
		av(WIDTH-1) := '1';
		tmp := (others => '-');
		tmp(WIDTH-1) := '1';
		tmp(WIDTH-2) := '0';
		for i in 2 to WIDTH loop
			av(WIDTH-i) := '0';
			A <= av; Y <= normalize(av, '1');
			wait for 1 ns; check_logic("Y", Y, tmp);
			for j in 0 to WIDTH-i-1 loop
				assert av(j) = 'X';
				for bit in std_ulogic_0 to std_ulogic_1 loop
					av(j) := bit;
					tmp(j+i-2) := bit;
					A <= av; Y <= normalize(av, '1');
					wait for 1 ns; check_logic("Y", Y, tmp);
				end loop;
				av(j) := 'X';
				tmp(j+i-2) := '-';
			end loop;
			av(WIDTH-i) := '1';
			tmp(i-2) := '0';
		end loop;
		A <= av; Y <= normalize(av, '1');
		wait for 1 ns; check_logic("Y", Y, tmp);

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
