-- bit_manipulation_test.vhdl - miscelleanous bit manipulation functions
-- Copyright (C) 2002 - 2004 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: bit_manipulation_test.vhdl,v 1.4 2004/01/25 03:12:00 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;
use work.Bit_Manipulation.all;

entity Bit_Manipulation_test is
end Bit_Manipulation_test;

architecture Arch_1 of Bit_Manipulation_test is
	constant WIDTH : natural := 64;

	signal A : std_ulogic_vector(WIDTH-1 downto 0) := (others => '1');
	signal Y : std_ulogic_vector(WIDTH-1 downto 0);

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 x, y : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("Y := ")); write(lout, Y); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, y);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- driver process
	run : process
		variable av, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		constant std_ulogic_0 : std_ulogic := '0';
		constant std_ulogic_1 : std_ulogic := '1';
	begin
		writestr("*** testing bit_reverse function ***");
		for w in 1 to WIDTH loop
			av := (others => 'X');
			tmp := (others => '-');
			for i in 0 to w-1 loop
				for bit in std_ulogic_0 to std_ulogic_1 loop
					av(i) := bit;
					tmp(w-1-i) := bit;
					Y(w-1 downto 0) <= bit_reverse(av(w-1 downto 0));
					A <= av; wait for 1 ns;
					check_logic("Y", Y, tmp);
				end loop;
				av(i) := 'X';
				tmp(w-1-i) := '-';
			end loop;
		end loop;

		writestr("*** testing bit_extract function ***");
		for w in 1 to WIDTH loop
			for n in 1 to w loop
				for o in 0 to w-1 loop
					av := (others => 'X');
					for i in 0 to w-1 loop
						next when n * i + o >= w;
						for bit in std_ulogic_0 to std_ulogic_1 loop
							av(n * i + o) := bit;
							tmp := (others => '-');
							tmp(i) := bit;
							Y((w-o-1)/n downto 0) <=
								bit_extract(av(w-1 downto 0), n, o);
							A <= av; wait for 1 ns;
							check_logic("Y", Y, tmp);
						end loop;
						av(i) := 'X';
					end loop;
				end loop;
			end loop;
		end loop;

		writestr("*** testing cascade_and function ***");
		for w in 1 to WIDTH loop
			for i in 0 to w-1 loop
				av := (others => 'X');
				tmp := (others => '-');
				av(i) := '0';
				tmp(w-1 downto i) := (w-1 downto i => '0');
				Y(w-1 downto 0) <= cascade_and(av(w-1 downto 0));
				A <= av; wait for 1 ns;
				check_logic("Y", Y, tmp);
			end loop;
			for i in 0 to w-1 loop
				av := (others => 'X');
				tmp := (others => '-');
				av(i downto 0) := (i downto 0 => '1');
				tmp(i downto 0) := (i downto 0 => '1');
				Y(w-1 downto 0) <= cascade_and(av(w-1 downto 0));
				A <= av; wait for 1 ns;
				check_logic("Y", Y, tmp);
			end loop;
		end loop;

		writestr("*** testing cascade_or function ***");
		for w in 1 to WIDTH loop
			for i in 0 to w-1 loop
				av := (others => 'X');
				tmp := (others => '-');
				av(i) := '1';
				tmp(w-1 downto i) := (w-1 downto i => '1');
				Y(w-1 downto 0) <= cascade_or(av(w-1 downto 0));
				A <= av; wait for 1 ns;
				check_logic("Y", Y, tmp);
			end loop;
			for i in 0 to w-1 loop
				av := (others => 'X');
				tmp := (others => '-');
				av(i downto 0) := (i downto 0 => '0');
				tmp(i downto 0) := (i downto 0 => '0');
				Y(w-1 downto 0) <= cascade_or(av(w-1 downto 0));
				A <= av; wait for 1 ns;
				check_logic("Y", Y, tmp);
			end loop;
		end loop;

		writestr("*** testing reduce_and function ***");
		for w in 1 to WIDTH loop
			av := (others => 'X');
			tmp := (others => '-');
			tmp(0) := '0';
			for i in 0 to w-1 loop
				av(i) := '0';
				Y(0) <= reduce_and(av(w-1 downto 0));
				A <= av; wait for 1 ns;
				check_logic("Y", Y, tmp);
				av(i) := 'X';
			end loop;
			av := (others => 'X');
			av(w-1 downto 0) := (w-1 downto 0 => '1');
			tmp(0) := '1';
			Y(0) <= reduce_and(av(w-1 downto 0));
			A <= av; wait for 1 ns;
			check_logic("Y", Y, tmp);
		end loop;

		writestr("*** testing reduce_or function ***");
		for w in 1 to WIDTH loop
			av := (others => 'X');
			tmp := (others => '-');
			tmp(0) := '1';
			for i in 0 to w-1 loop
				av(i) := '1';
				Y(0) <= reduce_or(av(w-1 downto 0));
				A <= av; wait for 1 ns;
				check_logic("Y", Y, tmp);
				av(i) := 'X';
			end loop;
			av := (others => 'X');
			av(w-1 downto 0) := (w-1 downto 0 => '0');
			tmp(0) := '0';
			Y(0) <= reduce_or(av(w-1 downto 0));
			A <= av; wait for 1 ns;
			check_logic("Y", Y, tmp);
		end loop;

		writestr("*** testing lshift function ***");
		for w in 1 to WIDTH loop
			for n in 0 to w loop
				av := (others => 'X');
				for i in 0 to w-1 loop
					for bit in std_ulogic_0 to std_ulogic_1 loop
						av(i) := bit;
						tmp := (others => '-');
						if n + i < w then
							tmp(i + n) := bit;
						end if;
						Y(w-1 downto 0) <= lshift(av(w-1 downto 0), n, 'X');
						A <= av; wait for 1 ns;
						check_logic("Y", Y, tmp);
					end loop;
					av(i) := 'X';
				end loop;
				tmp := (others => '-');
				for ext in std_ulogic_0 to std_ulogic_1 loop
					tmp(n-1 downto 0) := (n-1 downto 0 => ext);
					Y(w-1 downto 0) <= lshift(av(w-1 downto 0), n, ext);
					A <= av; wait for 1 ns;
					check_logic("Y", Y, tmp);
				end loop;
			end loop;
		end loop;

		writestr("*** testing rshift function ***");
		for w in 1 to WIDTH loop
			for n in 0 to w loop
				av := (others => 'X');
				for i in 0 to w-1 loop
					for bit in std_ulogic_0 to std_ulogic_1 loop
						av(i) := bit;
						tmp := (others => '-');
						if i >= n then
							tmp(i - n) := bit;
						end if;
						Y(w-1 downto 0) <= rshift(av(w-1 downto 0), n, 'X');
						A <= av; wait for 1 ns;
						check_logic("Y", Y, tmp);
					end loop;
					av(i) := 'X';
				end loop;
				tmp := (others => '-');
				for ext in std_ulogic_0 to std_ulogic_1 loop
					tmp(w-1 downto w-n) := (w-1 downto w-n => ext);
					Y(w-1 downto 0) <= rshift(av(w-1 downto 0), n, ext);
					A <= av; wait for 1 ns;
					check_logic("Y", Y, tmp);
				end loop;
			end loop;
		end loop;

		writestr("*** testing lrotate function ***");
		for w in 1 to WIDTH loop
			for n in 0 to w loop
				av := (others => 'X');
				for i in 0 to w-1 loop
					for bit in std_ulogic_0 to std_ulogic_1 loop
						av(i) := bit;
						tmp := (others => '-');
						tmp((i + n) mod w) := bit;
						Y(w-1 downto 0) <= lrotate(av(w-1 downto 0), n);
						A <= av; wait for 1 ns;
						check_logic("Y", Y, tmp);
					end loop;
					av(i) := 'X';
				end loop;
			end loop;
		end loop;

		writestr("*** testing rrotate function ***");
		for w in 1 to WIDTH loop
			for n in 0 to w loop
				av := (others => 'X');
				for i in 0 to w-1 loop
					for bit in std_ulogic_0 to std_ulogic_1 loop
						av(i) := bit;
						tmp := (others => '-');
						tmp((w + i - n) mod w) := bit;
						Y(w-1 downto 0) <= rrotate(av(w-1 downto 0), n);
						A <= av; wait for 1 ns;
						check_logic("Y", Y, tmp);
					end loop;
					av(i) := 'X';
				end loop;
			end loop;
		end loop;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
