-- bit_manipulation.vhdl - miscellaneous bit manipulation functions
-- Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: bit_manipulation.vhdl,v 1.13 2004/01/23 15:30:02 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

package Bit_Manipulation is
	-- reverse bits in a vector
	function bit_reverse (A : in std_ulogic_vector) return std_ulogic_vector;
	-- extract 1 bit of N, starting at offset O
	function bit_extract (A : in std_ulogic_vector;
						  N : in positive;
						  O : in natural := 0) return std_ulogic_vector;
	-- duplicate all bits in a vector
	function bit_duplicate (A : in std_ulogic_vector;
							N : in positive) return std_ulogic_vector;
	-- duplicate vector
	function vector_duplicate (A : in std_ulogic_vector;
							   N : in positive) return std_ulogic_vector;
	-- AND cascade
	function cascade_and (A : in std_ulogic_vector) return std_ulogic_vector;
	-- OR cascade
	function cascade_or (A : in std_ulogic_vector) return std_ulogic_vector;
	-- n:1 AND
	function reduce_and (A : in std_ulogic_vector) return std_ulogic;
	-- n:1 XOR
	function reduce_xor (A : in std_ulogic_vector) return std_ulogic;
	-- n:1 OR
	function reduce_or (A : in std_ulogic_vector) return std_ulogic;
	-- left shift w/ carry-in
	function lshift (A : in std_ulogic_vector;
					 N : in natural;
					 C : in std_ulogic) return std_ulogic_vector;
	-- left shift w/o carry-in
	function lshift (A : in std_ulogic_vector;
					 N : in natural) return std_ulogic_vector;
	-- arithmetic left shift
	function lshifta (A : in std_ulogic_vector;
					  N : in natural) return std_ulogic_vector;
	-- right shift w/ carry-in
	function rshift (A : in std_ulogic_vector;
					 N : in natural;
					 C : in std_ulogic) return std_ulogic_vector;
	-- right shift w/o carry-in
	function rshift (A : in std_ulogic_vector;
					 N : in natural) return std_ulogic_vector;
	-- arithmetic right shift
	function rshifta (A : in std_ulogic_vector;
					  N : in natural) return std_ulogic_vector;
	-- left rotate
	function lrotate (A : in std_ulogic_vector;
					  N : in natural) return std_ulogic_vector;
	-- right rotate
	function rrotate (A : in std_ulogic_vector;
					  N : in natural) return std_ulogic_vector;
	-- increment
	function incr (A : in std_ulogic_vector) return std_ulogic_vector;
	-- decrement
	function decr (A : in std_ulogic_vector) return std_ulogic_vector;
	-- negation
	function neg (A : in std_ulogic_vector) return std_ulogic_vector;
end Bit_Manipulation;

package body Bit_Manipulation is
	function bit_reverse (A : in std_ulogic_vector) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa, yy : std_ulogic_vector(L-1 downto 0);
	begin
--pragma synthesis_off
		assert L > 0;
--pragma synthesis_on
		aa := A;
		for i in aa'range loop
			yy(i) := aa(L - 1 - i);
		end loop;
		return yy;
	end bit_reverse;

	function bit_extract (A : in std_ulogic_vector;
						  N : in positive;
						  O : in natural := 0) return std_ulogic_vector is
		constant L : natural := A'length;
		constant L2 : natural := (L - O + N - 1) / N;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(L2-1 downto 0);
	begin
--pragma synthesis_off
		assert L > O;
--pragma synthesis_on
		aa := A;
		for i in L2-1 downto 0 loop
			yy(i) := aa(N*i+O);
		end loop;
		return yy;
	end bit_extract;

	function bit_duplicate (A : in std_ulogic_vector;
							N : in positive) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(N*L-1 downto 0);
	begin
--pragma synthesis_off
		assert L > 0;
		assert N > 0;
--pragma synthesis_on
		aa := A;
		for i in N*L-1 downto 0 loop
			yy(i) := aa(i/N);
		end loop;
		return yy;
	end bit_duplicate;

	function vector_duplicate (A : in std_ulogic_vector;
							   N : in positive) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(N*L-1 downto 0);
	begin
--pragma synthesis_off
		assert L > 0;
		assert N > 0;
--pragma synthesis_on
		aa := A;
		for i in N*L-1 downto 0 loop
			yy(i) := aa(i rem L);
		end loop;
		return yy;
	end vector_duplicate;

	function cascade_and (A : in std_ulogic_vector) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa, bb : std_ulogic_vector(L-1 downto 0);
		variable k1, k2, k3 : integer;
		variable step : natural;
	begin
--pragma synthesis_off
		assert L > 0;
--pragma synthesis_on
		aa := A;
		for i in 0 to 15 loop	-- should be enough
			step := 4 ** i;
			exit when step >= L;
			for j in aa'range loop
				k1 := j - j mod (4 * step) + step - 1;
				k2 := k1 + step;
				k3 := k2 + step;
				case (j / step) mod 4 is
					when 3 =>
						bb(j) := aa(j) and aa(k1) and aa(k2) and aa(k3);
					when 2 =>
						bb(j) := aa(j) and aa(k1) and aa(k2);
					when 1 =>
						bb(j) := aa(j) and aa(k1);
					when others =>
						bb(j) := aa(j);
				end case;
			end loop;
			aa := bb;
		end loop;
		return aa;
	end cascade_and;

	function cascade_or (A : in std_ulogic_vector) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa, bb : std_ulogic_vector(L-1 downto 0);
		variable k1, k2, k3 : integer;
		variable step : natural;
	begin
--pragma synthesis_off
		assert L > 0;
--pragma synthesis_on
		aa := A;
		for i in 0 to 15 loop	-- should be enough
			step := 4 ** i;
			exit when step >= L;
			for j in aa'range loop
				k1 := j - j mod (4 * step) + step - 1;
				k2 := k1 + step;
				k3 := k2 + step;
				case (j / step) mod 4 is
					when 3 =>
						bb(j) := aa(j) or aa(k1) or aa(k2) or aa(k3);
					when 2 =>
						bb(j) := aa(j) or aa(k1) or aa(k2);
					when 1 =>
						bb(j) := aa(j) or aa(k1);
					when others =>
						bb(j) := aa(j);
				end case;
			end loop;
			aa := bb;
		end loop;
		return aa;
	end cascade_or;

	function reduce_and (A : in std_ulogic_vector) return std_ulogic is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable k, len : natural;
	begin
--pragma synthesis_off
		assert L > 0;
--pragma synthesis_on
		aa := A;
		len := L;
		for j in 0 to 15 loop	-- should be enough
			exit when len = 1;
			k := len / 4;
			for i in 0 to k-1 loop
				aa(i) := aa(4*i+0) and aa(4*i+1) and aa(4*i+2) and aa(4*i+3);
			end loop;
			case len mod 4 is
				when 3 => aa(k) := aa(4*k+0) and aa(4*k+1) and aa(4*k+2);
				when 2 => aa(k) := aa(4*k+0) and aa(4*k+1);
				when 1 => aa(k) := aa(4*k+0);
				when others => null;
			end case;
			len := (len + 3) / 4;
		end loop;
		return aa(0);
	end reduce_and;

	function reduce_xor (A : in std_ulogic_vector) return std_ulogic is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable k, len : natural;
	begin
--pragma synthesis_off
		assert L > 0;
--pragma synthesis_on
		aa := A;
		len := L;
		for j in 0 to 31 loop	-- should be enough
			exit when len = 1;
			k := len / 2;
			for i in 0 to k-1 loop
				aa(i) := aa(2*i+0) xor aa(2*i+1);
			end loop;
			case len mod 2 is
				when 1 => aa(k) := aa(2*k+0);
				when others => null;
			end case;
			len := (len + 1) / 2;
		end loop;
		return aa(0);
	end reduce_xor;

	function reduce_or (A : in std_ulogic_vector) return std_ulogic is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable k, len : natural;
	begin
--pragma synthesis_off
		assert L > 0;
--pragma synthesis_on
		aa := A;
		len := L;
		for j in 0 to 15 loop	-- should be enough
			exit when len = 1;
			k := len / 4;
			for i in 0 to k-1 loop
				aa(i) := aa(4*i+0) or aa(4*i+1) or aa(4*i+2) or aa(4*i+3);
			end loop;
			case len mod 4 is
				when 3 => aa(k) := aa(4*k+0) or aa(4*k+1) or aa(4*k+2);
				when 2 => aa(k) := aa(4*k+0) or aa(4*k+1);
				when 1 => aa(k) := aa(4*k+0);
				when others => null;
			end case;
			len := (len + 3) / 4;
		end loop;
		return aa(0);
	end reduce_or;

	function lshift (A : in std_ulogic_vector;
					 N : in natural;
					 C : in std_ulogic) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
	begin
		aa := A;
		yy := (others => C);
		if N < L then
			yy(L-1 downto N) := aa(L-N-1 downto 0);
		end if;
		return yy;
	end lshift;

	function lshift (A : in std_ulogic_vector;
					 N : in natural) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
	begin
		aa := A;
		yy := (others => '0');
		if N < L then
			yy(L-1 downto N) := aa(L-N-1 downto 0);
		end if;
		return yy;
	end lshift;

	function lshifta (A : in std_ulogic_vector;
					  N : in natural) return std_ulogic_vector is
	begin
		return lshift(A, N, A(A'right));
	end lshifta;

	function rshift (A : in std_ulogic_vector;
					 N : in natural;
					 C : in std_ulogic) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
	begin
		aa := A;
		yy := (others => C);
		if N < L then
			yy(L-N-1 downto 0) := aa(L-1 downto N);
		end if;
		return yy;
	end rshift;

	function rshift (A : in std_ulogic_vector;
					 N : in natural) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
	begin
		aa := A;
		yy := (others => '0');
		if N < L then
			yy(L-N-1 downto 0) := aa(L-1 downto N);
		end if;
		return yy;
	end rshift;

	function rshifta (A : in std_ulogic_vector;
					  N : in natural) return std_ulogic_vector is
	begin
		return rshift(A, N, A(A'left));
	end rshifta;

	function lrotate (A : in std_ulogic_vector;
					  N : in natural) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
	begin
		aa := A;
		for i in L-1 downto 0 loop
			yy(i) := aa((i + L - N) rem L);
		end loop;
		return yy;
	end lrotate;

	function rrotate (A : in std_ulogic_vector;
					  N : in natural) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable yy : std_ulogic_vector(L-1 downto 0);
	begin
		aa := A;
		for i in L-1 downto 0 loop
			yy(i) := aa((i + N) rem L);
		end loop;
		return yy;
	end rrotate;

	-- increment/decrement

	function incr (A : in std_ulogic_vector) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable bb : std_ulogic_vector(L-1 downto 0);
	begin
		aa := A;
		bb := lshift(cascade_and(aa), 1, '1');
		return aa xor bb;
	end incr;

	function decr (A : in std_ulogic_vector) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable bb : std_ulogic_vector(L-1 downto 0);
	begin
		aa := A;
		bb := lshift(cascade_and(not aa), 1, '1');
		return aa xor bb;
	end decr;

	-- negation

	function neg (A : in std_ulogic_vector) return std_ulogic_vector is
		constant L : natural := A'length;
		variable aa : std_ulogic_vector(L-1 downto 0);
		variable bb : std_ulogic_vector(L-1 downto 0);
	begin
		aa := not A;
		bb := lshift(cascade_and(aa), 1, '1');
		return aa xor bb;
	end neg;
end Bit_Manipulation;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
