-- asu.vhdl - Add/Subtract Execution Unit for the F-CPU.
-- Copyright (C) 2000, 2001, 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: asu.vhdl,v 1.15 2003/04/15 23:25:30 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;

use work.FCPU_config.all;

entity EU_ASU is
	port(
		-- operands
		Din_0    : in F_VECTOR;
		Din_1    : in F_VECTOR;
		-- subtract flag (should be derived from opcode)
		Subtract : in std_ulogic;
		-- flag bits (directly copied from instruction word)
		Flags    : in std_ulogic_vector(13 downto 8);
		-- SIMD mode bits (decoded)
		Size     : in std_ulogic_vector(LOGMAXSIZE-1 downto 0);
		-- clock/reset/enable inputs
		Clk      : in std_ulogic;
		Rst      : in std_ulogic;
		En       : in std_ulogic;
	--
		-- 8-bit result
		Dout_0   : out F_VECTOR;
		-- 8-bit carry output
		Dout_1   : out F_VECTOR;
		-- 64-bit result
		Dout_2   : out F_VECTOR;
		-- 64-bit carry output
		Dout_3   : out F_VECTOR
	);
end EU_ASU;

architecture Struct_1 of EU_ASU is
	component IAdd
		generic (
			WIDTH : natural := 64;
			PIPELINED : integer := 0
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			Sub : in std_ulogic;
			Sat : in std_ulogic;
			Sig : in std_ulogic;
			Inc : in std_ulogic;
			Avg : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y8l : out std_ulogic_vector(WIDTH-1 downto 0);
			Y8h : out std_ulogic_vector(WIDTH-1 downto 0);
			Yl : out std_ulogic_vector(WIDTH-1 downto 0);
			Yh : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal dummy : std_ulogic := '0';
begin
--pragma synthesis_off
	assert UMAX mod 64 = 0
		report "UMAX must be an integer multiple of 64"
		severity failure;
--pragma synthesis_on

	adder : IAdd
		generic map (WIDTH => UMAX, PIPELINED => 1)
		port map (
			A => Din_0,
			B => Din_1,
			Sub => Subtract,
			Sat => Flags(12),	-- as defined in the manual
			Sig => dummy,
			Inc => dummy,
			Avg => dummy,
			U => Size(2 downto 0),
			Clk => Clk,
			Rst => Rst,
			En => En,
			Y8l => Dout_0,
			Y8h => Dout_1,
			Yl => Dout_2,
			Yh => Dout_3
		);
end Struct_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
