-- im64test2.vhdl - F-CPU Integer Multiplication Unit Testbench #2
-- Copyright (C) 2000, 2001, 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: im64test2.vhdl,v 1.12 2003/04/17 15:55:16 michael Exp $

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

use work.IMul64_testconf.all;

entity IMul64_test2 is
end IMul64_test2;

architecture Arch_1 of IMul64_test2 is
	component IMul64
		generic (
			WIDTH : natural := 64;	-- DO NOT CHANGE!
			PIPELINED : integer := 0
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			X : in std_ulogic_vector(WIDTH-1 downto 0);
			Sig : in std_ulogic;
			Mac : in std_ulogic_vector(2 downto 0);
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y08l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=18 (3 stages)
			Y08h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=18 (3 stages)
			Y16l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=24 (4 stages)
			Y16h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=24 (4 stages)
			Y32l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=29 (5 stages)
			Y32h : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=29 (5 stages)
			Y64l : out std_ulogic_vector(WIDTH-1 downto 0);	-- d=34 (6 stages)
			Y64h : out std_ulogic_vector(WIDTH-1 downto 0)	-- d=34 (6 stages)
		);
	end component;

	constant WIDTH : natural := 64;

	signal A, B : std_ulogic_vector(WIDTH-1 downto 0);
	signal X : std_ulogic_vector(WIDTH-1 downto 0);
	signal M : std_ulogic_vector(6 downto 0);
	signal Y08l, Y08h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y16l, Y16h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y32l, Y32h : std_ulogic_vector(WIDTH-1 downto 0);
	signal Y64l, Y64h : std_ulogic_vector(WIDTH-1 downto 0);
	signal L, H : std_ulogic_vector(WIDTH-1 downto 0);

	signal Clk, Rst, En : std_ulogic;

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 x_1, x_2 : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("X := ")); write(lout, X); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("H := ")); write(lout, H); writeline(output, lout);
		write(lout, string'("L := ")); write(lout, L); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, x_1);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, x_2);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	Clk <= '0'; Rst <= '0'; En <= '1';

	mut: IMul64
		generic map (PIPELINED => 0)
		port map (
			A => A, B => B, X => X, Sig => M(0),
			Mac => M(3 downto 1), U => M(6 downto 4),
			Clk => Clk, Rst => Rst, En => En,
			Y08l => Y08l, Y08h => Y08h,
			Y16l => Y16l, Y16h => Y16h,
			Y32l => Y32l, Y32h => Y32h,
			Y64l => Y64l, Y64h => Y64h
		);

	-- output selector
	L <= Y64l when M(6) = '1'
	else Y32l when M(5) = '1'
	else Y16l when M(4) = '1'
	else Y08l;
	H <= Y64h when M(6) = '1'
	else Y32h when M(5) = '1'
	else Y16h when M(4) = '1'
	else Y08h;

	test : process
		variable prod, res : integer;
		variable lout : line;
		variable pv, rv : std_ulogic_vector(2*WIDTH-1 downto 0);
		variable gran, left, right : natural;
	begin
if true then
		for level in MIN_LEVEL to MAX_LEVEL loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit low-part products ***"));
			writeline(output, lout);
			M <= "000000X";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;
			A <= (others => 'X');
			B <= (others => 'X');
			X <= (others => 'X');
			for slice in 0 to 64/gran-1 loop
				write(lout, string'("--- Slice "));
				write(lout, slice);
				write(lout, string'(" / "));
				write(lout, 64 / gran);
				write(lout, string'(" ---"));
				writeline(output, lout);
				right := gran * slice;
				left := right + gran - 1;
				A(left downto right) <= (others => '0');
				B(left downto right) <= (others => '0');
				wait for 1 ns;
				pv(left downto right) := (others => '0');
				check_logic("L", L(left downto right), pv(left downto right));
				for i in 0 to gran-1 loop
					A(right+i) <= '1';
					for j in 0 to gran-i-1 loop
						B(right+j) <= '1';
						wait for 1 ns;
						pv(right+i+j) := '1';
						check_logic("L", L(left downto right), pv(left downto right));
						pv(right+i+j) := '0';
						B(right+j) <= '0';
					end loop;
					A(right+i) <= '0';
				end loop;
				A(left downto right) <= (others => 'X');
				B(left downto right) <= (others => 'X');
			end loop;
		end loop;
end if;

if true then
		for level in MIN_LEVEL to MAX_LEVEL loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit high-part products (unsigned) ***"));
			writeline(output, lout);
			M <= "0000000";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;
			A <= (others => 'X');
			B <= (others => 'X');
			X <= (others => 'X');
			for slice in 0 to 64/gran-1 loop
				write(lout, string'("--- Slice "));
				write(lout, slice);
				write(lout, string'(" / "));
				write(lout, 64 / gran);
				write(lout, string'(" ---"));
				writeline(output, lout);
				right := gran * slice;
				left := right + gran - 1;
				A(left downto right) <= (others => '0');
				B(left downto right) <= (others => '0');
				wait for 1 ns;
				pv(left downto right) := (others => '0');
				check_logic("H", H(left downto right), pv(left downto right));
				for i in 1 to gran-1 loop
					A(right+i) <= '1';
					for j in gran-i to gran-1 loop
						B(right+j) <= '1';
						wait for 1 ns;
						pv(right+i+j-gran) := '1';
						check_logic("H", H(left downto right), pv(left downto right));
						pv(right+i+j-gran) := '0';
						B(right+j) <= '0';
					end loop;
					A(right+i) <= '0';
				end loop;
				A(left downto right) <= (others => 'X');
				B(left downto right) <= (others => 'X');
			end loop;
		end loop;
end if;

if true then
		for level in MIN_LEVEL to MAX_LEVEL loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit signed products ***"));
			writeline(output, lout);
			M <= "0000001";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;
			A <= (others => 'X');
			B <= (others => 'X');
			X <= (others => 'X');
			for slice in 0 to 64/gran-1 loop
				write(lout, string'("--- Slice "));
				write(lout, slice);
				write(lout, string'(" / "));
				write(lout, 64 / gran);
				write(lout, string'(" ---"));
				writeline(output, lout);
				right := gran * slice;
				left := right + gran - 1;
				A(left downto right) <= (others => '0');
				B(left downto right) <= (others => '0');
				wait for 1 ns;
				pv(left downto right) := (others => '0');
				check_logic("H", H(left downto right), pv(left downto right));
				for i in 2 to gran-2 loop
					A(right+i) <= '1';
					for j in gran-i to gran-2 loop
						B(right+j) <= '1';
						wait for 1 ns;
						pv(right+i+j-gran) := '1';
						check_logic("H", H(left downto right), pv(left downto right));
						pv(right+i+j-gran) := '0';
						B(right+j) <= '0';
					end loop;
					A(right+i) <= '0';
				end loop;
				A(left downto right) <= (others => '1');
				B(left downto right) <= (others => '0');
				wait for 1 ns;
				rv(2*left+1 downto 2*right) := H(left downto right) & L(left downto right);
				pv(2*left+1 downto 2*right) := (others => '0');
				check_logic("Y", rv(2*left+1 downto 2*right), pv(2*left+1 downto 2*right));
				for i in 0 to gran-1 loop
					pv(2*left+1 downto 2*right+i) := (others => '1');
					pv(2*right+i-1 downto 2*right) := (others => '0');
					for j in 0 to gran-2 loop
						B(right+j) <= '1';
						wait for 1 ns;
						rv(2*left+1 downto 2*right) := H(left downto right) & L(left downto right);
						check_logic("Y", rv(2*left+1 downto 2*right), pv(2*left+1 downto 2*right));
						pv(2*right+i+j) := '0';
						B(right+j) <= '0';
					end loop;
					A(right+i) <= '0';
				end loop;
				A(left downto right) <= (others => '0');
				B(left downto right) <= (others => '1');
				wait for 1 ns;
				rv(2*left+1 downto 2*right) := H(left downto right) & L(left downto right);
				pv(2*left+1 downto 2*right) := (others => '0');
				check_logic("Y", rv(2*left+1 downto 2*right), pv(2*left+1 downto 2*right));
				for i in 0 to gran-1 loop
					pv(2*left+1 downto 2*right+i) := (others => '1');
					pv(2*right+i-1 downto 2*right) := (others => '0');
					for j in 0 to gran-2 loop
						A(right+j) <= '1';
						wait for 1 ns;
						rv(2*left+1 downto 2*right) := H(left downto right) & L(left downto right);
						check_logic("Y", rv(2*left+1 downto 2*right), pv(2*left+1 downto 2*right));
						pv(2*right+i+j) := '0';
						A(right+j) <= '0';
				end loop;
					B(right+i) <= '0';
				end loop;
				A(left downto right) <= (others => '1');
				pv(2*left+1 downto 2*right) := (others => '0');
				for i in 0 to gran-1 loop
					B(left downto right) <= (others => '1');
					for j in 0 to gran-1 loop
						wait for 1 ns;
						rv(2*left+1 downto 2*right) := H(left downto right) & L(left downto right);
						pv(2*right+i+j) := '1';
						check_logic("Y", rv(2*left+1 downto 2*right), pv(2*left+1 downto 2*right));
						pv(2*right+i+j) := '0';
						B(right+j) <= '0';
					end loop;
					A(right+i) <= '0';
				end loop;
				A(left downto right) <= (others => 'X');
				B(left downto right) <= (others => 'X');
			end loop;
		end loop;
end if;

if true then
		A <= (others => '0');
		B <= (others => '0');
		for level in MIN_LEVEL to MAX_LEVEL loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit macl/mach pass-through ***"));
			writeline(output, lout);
			M <= "000000X";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;
			X <= (others => '0');
			pv := (others => '0');
			for i in 0 to 127 loop
				X(i mod 64) <= '1';
				M(i / 64 + 1) <= '1';
				wait for 1 ns;
				rv := H & L;
				right := gran * (i / gran / 2)
					   + i mod gran
					   + 64 * ((i / gran) mod 2);
				pv(right) := '1';
				check_logic("Y", rv, pv);
				pv(right) := '0';
				M(i / 64 + 1) <= '0';
				X(i mod 64) <= '0';
			end loop;
		end loop;
end if;

if true then
		A <= (others => '0');
		B <= (others => '0');
		for level in MIN_LEVEL to MAX_LEVEL loop
			gran := 8 * 2 ** level;
			write(lout, string'("*** Testing "));
			write(lout, gran);
			write(lout, string'("-bit alternative mac pass-through ***"));
			writeline(output, lout);
			M <= "000100X";
			for i in 4 to 6 loop
				if level + 4 > i then
					M(i) <= '1';
				end if;
			end loop;
			X <= (others => '0');
			pv := (others => '0');
			for i in 0 to 63 loop
				X(i) <= '1';
				wait for 1 ns;
				rv := H & L;
				pv(i) := '1';
				check_logic("Y", rv, pv);
				pv(i) := '0';
				X(i) <= '0';
			end loop;
		end loop;
end if;

		writestr("*** Simulation complete ***");
		wait;
	end process;
end Arch_1;

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
