-- icmp64_test.vhdl -- Testbench for 64-Bit Integer Compare Unit
-- Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: icmp64_test.vhdl,v 1.6 2003/04/17 15:52:54 michael Exp $

--pragma synthesis_off

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

entity ICmp64_test is
	generic (WIDTH : natural := 64);
end ICmp64_test;

architecture Arch_1 of ICmp64_test is
	component ICmp64
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			Comp : in std_ulogic;
			Sort : in std_ulogic;
			Msb0 : in std_ulogic;
			Msb1 : in std_ulogic;
			Sig : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Z : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	signal M : std_ulogic_vector(7 downto 0) := (others => '0');
	signal A, B, Y, Z : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal Clk : std_ulogic := '0';
	signal Rst : std_ulogic := '0';
	signal En : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_vector (lbl : string;
							x : std_ulogic_vector;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_vector;

	procedure print_signals is
	begin
		print_vector("A", A);
		print_vector("B", B);
		print_vector("M", M);
		print_vector("Y", Y);
		print_vector("Z", Z);
	end print_signals;

	procedure do_error (lbl : string;
						a_x, a_y : std_ulogic_vector) is
	begin
		writestr("WHOA THERE!!!");
		print_signals;
		print_vector(lbl, a_x);
		print_vector(lbl, a_y, " /= ");
	end do_error;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y rem 2**x'length, x'length));
		if x /= tmp then
			do_error(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length
			report "bad args in check_logic" severity failure;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_error(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : ICmp64
		generic map (WIDTH => WIDTH)
		port map (
			A => A,
			B => B,
			Comp => M(0),
			Sort => M(1),
			Msb0 => M(2),
			Msb1 => M(3),
			Sig => M(4),
			U => M(7 downto 5),
			Clk => Clk,
			Rst => Rst,
			En => En,
			Y => Y,
			Z => Z
		);

	-- driver process
	process
		constant std_ulogic_0 : std_ulogic := '0';
		constant std_ulogic_1 : std_ulogic := '1';

		procedure do_clock is
		begin
			wait for 1 ns;
			Clk <= '1';
			wait for 1 ns;
			Clk <= '0';
		end do_clock;

		procedure print_mode (simd : in natural) is
			variable lout : line;
		begin
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
		end print_mode;

		procedure check_cmpg (left, right : natural; res : in std_ulogic) is
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
		begin
			do_clock;
			tmp := (others => '-');
			tmp(left downto right) := (left downto right => res);
			check_logic("Y", Y, tmp);
			tmp(left downto right) := (left downto right => not res);
			check_logic("Z", Z, tmp);
		end check_cmpg;

		procedure test_ucmp is
			variable av, bv : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing unsigned cmp ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX00001";
				for i in 5 to 7 loop
					if i < gran + 5 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					bv := (others => 'X');
					for index in left downto right loop
						-- a > b
						av(index) := '1';
						bv(index) := '0';
						A <= av; B <= bv; check_cmpg(left, right, '1');
						-- a < b
						av(index) := '0';
						bv(index) := '1';
						A <= av; B <= bv; check_cmpg(left, right, '0');
						bv(index) := '0';
					end loop;
					-- a = b
					A <= av; B <= bv; check_cmpg(left, right, '0');
				end loop;
			end loop;
		end test_ucmp;

		procedure test_scmp is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing signed cmp ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX10001";
				for i in 5 to 7 loop
					if i < gran + 5 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					bv := (others => 'X');
					tmp := (others => '-');
					-- a < b
					av(left) := '1';
					bv(left) := '0';
					A <= av; B <= bv; check_cmpg(left, right, '0');
					-- a > b
					av(left) := '0';
					bv(left) := '1';
					A <= av; B <= bv; check_cmpg(left, right, '1');
					av(left) := '1';
					for index in left-1 downto right loop
						-- a > b
						av(index) := '1';
						bv(index) := '0';
						A <= av; B <= bv; check_cmpg(left, right, '1');
						-- a < b
						av(index) := '0';
						bv(index) := '1';
						A <= av; B <= bv; check_cmpg(left, right, '0');
						av(index) := '1';
					end loop;
					-- a = b
					A <= av; B <= bv; check_cmpg(left, right, '0');
				end loop;
			end loop;
		end test_scmp;

		procedure test_usort is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing unsigned sort ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX00010";
				for i in 5 to 7 loop
					if i < gran + 5 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');	-- XXX: should use random data
					bv := (others => 'X');	-- XXX: should use random data
					tmp := (others => '-');
					for index in left downto right loop
						-- min = A, max = B
						av(index) := '0';
						bv(index) := '1';
						A <= av; B <= bv; do_clock;
						tmp(left downto right) := av(left downto right);
						check_logic("Y", Y, tmp);
						tmp(left downto right) := bv(left downto right);
						check_logic("Z", Z, tmp);
						-- min = B, max = A
						av(index) := '1';
						bv(index) := '0';
						A <= av; B <= bv; do_clock;
						tmp(left downto right) := bv(left downto right);
						check_logic("Y", Y, tmp);
						tmp(left downto right) := av(left downto right);
						check_logic("Z", Z, tmp);
						av(index) := '0';
					end loop;
				end loop;
			end loop;
		end test_usort;

		procedure test_ssort is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
		begin
			writestr("*** testing signed sort ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXX10010";
				for i in 5 to 7 loop
					if i < gran + 5 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');	-- XXX: should use random data
					bv := (others => 'X');	-- XXX: should use random data
					tmp := (others => '-');
					-- min = B, max = A
					av(left) := '0';
					bv(left) := '1';
					A <= av; B <= bv; do_clock;
					tmp(left downto right) := bv(left downto right);
					check_logic("Y", Y, tmp);
					tmp(left downto right) := av(left downto right);
					check_logic("Z", Z, tmp);
					-- min = A, max = B
					av(left) := '1';
					bv(left) := '0';
					A <= av; B <= bv; do_clock;
					tmp(left downto right) := av(left downto right);
					check_logic("Y", Y, tmp);
					tmp(left downto right) := bv(left downto right);
					check_logic("Z", Z, tmp);
					bv(left) := '1';
					for index in left-1 downto right loop
						-- min = A, max = B
						av(index) := '0';
						bv(index) := '1';
						A <= av; B <= bv; do_clock;
						tmp(left downto right) := av(left downto right);
						check_logic("Y", Y, tmp);
						tmp(left downto right) := bv(left downto right);
						check_logic("Z", Z, tmp);
						-- min = B, max = A
						av(index) := '1';
						bv(index) := '0';
						A <= av; B <= bv; do_clock;
						tmp(left downto right) := bv(left downto right);
						check_logic("Y", Y, tmp);
						tmp(left downto right) := av(left downto right);
						check_logic("Z", Z, tmp);
						av(index) := '0';
					end loop;
				end loop;
			end loop;
		end test_ssort;

		procedure test_msb (bit : in std_ulogic) is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, left, right : natural;
			variable lout : line;
		begin
			write(lout, string'("*** testing msb"));
			write(lout, bit);
			write(lout, string'(" ***"));
			writeline(output, lout);
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= "XXXXXX00";
				M(2) <= not bit;
				M(3) <= bit;
				for i in 5 to 7 loop
					if i < gran + 5 then
						M(i) <= '1';
					else
						M(i) <= '0';
					end if;
				end loop;
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk*simd;
					left := right + simd - 1;
					av := (others => 'X');
					av(left downto right) := (left downto right => not bit);
					bv := (others => 'X');
					tmp := (others => '-');
					tmp(left downto right) := (left downto right => '0');
					A <= av; B <= bv; do_clock;
					check_logic("Y", Y, tmp);
					check_logic("Z", Z, tmp);
					for index in right to left loop
						av(index) := bit;
						A <= av; B <= bv; do_clock;
						tmp(left downto right) := std_ulogic_vector(
							to_unsigned(index - right + 1, simd));
						check_logic("Y", Y, tmp);
						tmp(left downto right) := (left downto right => '0');
						tmp(index) := '1';
						check_logic("Z", Z, tmp);
						av(index) := 'X';
					end loop;
				end loop;
			end loop;
		end test_msb;
	begin
		test_ucmp;
		test_scmp;
		test_usort;
		test_ssort;
		test_msb('0');
		test_msb('1');

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

--pragma synthesis_on

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
