-- shuffle64_test2.vhdl -- Testbench for 64-Bit F-CPU Bit Shuffling Unit
-- Copyright (C) 2001 - 2003 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- @(#) $Id: shuffle64_test2.vhdl,v 1.5 2003/03/25 14:05:17 michael Exp $

--pragma synthesis_off

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;
use work.Bit_Manipulation.all;

entity Shuffle64_test2 is
end Shuffle64_test2;

architecture Arch_1 of Shuffle64_test2 is
	component Shuffle64
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			CommonShiftCount : in std_ulogic;
			ShiftL : in std_ulogic;
			ShiftR : in std_ulogic;
			ShiftRA : in std_ulogic;
			RotL : in std_ulogic;
			RotR : in std_ulogic;
			Bitrev : in std_ulogic;
			Byterev : in std_ulogic;
			Permute : in std_ulogic;
			Mix : in std_ulogic;
			Expand : in std_ulogic;
			Cshift : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Z : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	constant WIDTH : natural := 64;

	signal M : std_ulogic_vector(13 downto 0) := (others => '0');
	signal A, B, Y, Z : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal CommonShiftCount : std_ulogic := 'X';
	signal Clk : std_ulogic := '0';
	signal Rst : std_ulogic := '0';
	signal En : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure print_vector (lbl : string;
							x : std_ulogic_vector;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_vector;

	procedure print_signals is
	begin
		print_vector("A", A);
		print_vector("B", B);
		print_vector("M", M);
		print_vector("Y", Y);
		print_vector("Z", Z);
	end print_signals;

	procedure do_error (lbl : string;
						a_x, a_y : std_ulogic_vector) is
	begin
		writestr("WHOA THERE!!!");
		print_signals;
		print_vector(lbl, a_x);
		print_vector(lbl, a_y, " /= ");
	end do_error;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y rem 2**x'length, x'length));
		if x /= tmp then
			do_error(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length
			report "bad args in check_logic" severity failure;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_error(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : Shuffle64
		generic map (WIDTH => WIDTH)
		port map (
			A => A, B => B, U => M(2 downto 0),
			CommonShiftCount => CommonShiftCount,
			ShiftL => M(3), ShiftR => M(4), ShiftRA => M(5),
			RotL => M(6), RotR => M(7), Bitrev => M(8),
			Byterev => M(9), Permute => M(10),
			Mix => M(11), Expand => M(12), Cshift => M(13),
			Clk => Clk, Rst => Rst, En => En,
			Y => Y, Z => Z
		);

	-- driver process
	process
		constant std_ulogic_0 : std_ulogic := '0';
		constant std_ulogic_1 : std_ulogic := '1';

		procedure print_mode (simd : in natural) is
			variable lout : line;
		begin
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
		end print_mode;

		type shl_op is (
			FCOP_SHIFTL, FCOP_SHIFTR, FCOP_SHIFTRA,
			FCOP_ROTL, FCOP_ROTR, FCOP_BITREV, FCOP_BYTEREV,
			FCOP_PERMUTE, FCOP_MIX, FCOP_EXPAND, FCOP_CSHIFT
		);

		function bitop (op : in shl_op; alt_output : boolean;
						A : in std_ulogic_vector;
						B : in natural) return std_ulogic_vector is
			constant cs : natural := A'length;
			alias aa : std_ulogic_vector(cs-1 downto 0) is A;
			variable yy : std_ulogic_vector(cs-1 downto 0);
			variable zz : std_ulogic_vector(cs-1 downto 0);
			variable j : natural;
		begin
			assert B < cs;
			case op is
				when FCOP_SHIFTL =>
					yy := lshift(aa, B);
					zz := rshift(aa, cs - B);
				when FCOP_SHIFTR =>
					yy := rshift(aa, B);
					zz := lshift(aa, cs - B);
				when FCOP_SHIFTRA =>
					yy := rshifta(aa, B);
					zz := lshift(aa, cs - B);
				when FCOP_ROTL =>
					yy := lrotate(aa, B);
					zz := (others => '-');
				when FCOP_ROTR =>
					yy := rrotate(aa, B);
					zz := (others => '-');
				when FCOP_BITREV =>
					yy := rshift(bit_reverse(aa), cs - B - 1);
					zz := lshift(bit_reverse(aa), B + 1);
				when FCOP_BYTEREV =>
					assert not alt_output;
					for i in cs-1 downto 0 loop
						j := cs - i - 1;
						j := j - j rem 8 + i rem 8;
						yy(i) := aa(j);
					end loop;
					zz := (others => '-');
				when others =>
					assert false;
			end case;
			if alt_output then
				return zz;
			end if;
			return yy;
		end bitop;

		function shlmode (op : in shl_op;
						  gran : in natural) return std_ulogic_vector is
			variable m : std_ulogic_vector(13 downto 0);
		begin
			case op is
				when FCOP_SHIFTL =>
					m := "00000000001XXX";
				when FCOP_SHIFTR =>
					m := "00000000010XXX";
				when FCOP_SHIFTRA =>
					m := "00000000100XXX";
				when FCOP_ROTL =>
					m := "00000001000XXX";
				when FCOP_ROTR =>
					m := "00000010000XXX";
				when FCOP_BITREV =>
					m := "00000100000XXX";
				when FCOP_BYTEREV =>
					m := "00001000000XXX";
				when FCOP_PERMUTE =>
					m := "00010000000XXX";
				when FCOP_MIX =>
					m := "00100000000XXX";
				when FCOP_EXPAND =>
					m := "01000000000XXX";
				when FCOP_CSHIFT =>
					m := "10000000000XXX";
				when others =>
					assert false;
			end case;
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					m(n) := '1';
				else
					m(n) := '0';
				end if;
			end loop;
			return m;
		end shlmode;

		procedure write (l : inout line; op : in shl_op) is
		begin
			case op is
				when FCOP_SHIFTL =>
					write(l, string'("shiftl"));
				when FCOP_SHIFTR =>
					write(l, string'("shiftr"));
				when FCOP_SHIFTRA =>
					write(l, string'("shiftra"));
				when FCOP_ROTL =>
					write(l, string'("rotl"));
				when FCOP_ROTR =>
					write(l, string'("rotr"));
				when FCOP_BITREV =>
					write(l, string'("bitrev"));
				when others =>
					assert false;
			end case;
		end write;

		procedure print_testing (op : in shl_op) is
			variable lout : line;
		begin
			write(lout, string'("*** testing "));
			write(lout, op);
			write(lout, string'(" instruction ***"));
			writeline(output, lout);
		end print_testing;

		procedure test_byterev (alt : in boolean) is
			constant op : shl_op := FCOP_BYTEREV;
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, chunk, index, dest : natural;
		begin
			if alt then
				writestr("*** testing byterev instruction (alt. output) ***");
			else
				writestr("*** testing byterev instruction ***");
			end if;
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= shlmode(op, gran);
				av := (others => 'X');
				bv := (others => 'X');
				for bit_pos in 0 to WIDTH-1 loop
					index := bit_pos rem simd;
					chunk := bit_pos - index;
					dest := simd - 1 - index;
					dest := dest - dest rem 8 + index rem 8;
					tmp := (others => '-');
					for bit in std_ulogic_1 downto std_ulogic_0 loop
						tmp(chunk + dest) := bit;
						if alt then
							bv(bit_pos) := bit;
							A <= av; B <= bv; wait for 1 ns;
							check_logic("Z", Z, tmp);
							bv(bit_pos) := 'X';
						else
							av(bit_pos) := bit;
							A <= av; B <= bv; wait for 1 ns;
							check_logic("Y", Y, tmp);
							av(bit_pos) := 'X';
						end if;
					end loop;
				end loop;
			end loop;
		end test_byterev;

		procedure test_permute is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right, simd : natural;
		begin
			writestr("*** testing permute instructions ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= shlmode(FCOP_PERMUTE, gran);
				av := (others => 'X');
				tmp := (others => '-');
				for chunk in 0 to WIDTH/simd-1 loop
					right := chunk * simd;
					left := right + simd - 1;
					bv := (others => 'X');
					for sel in 0 to WIDTH/simd-1 loop
						bv(right+2-gran downto right) :=
							std_ulogic_vector(to_unsigned(sel, 3 - gran));
						for bit_pos in 0 to simd-1 loop
							for bit in std_ulogic_1 downto std_ulogic_0 loop
								av(sel * simd + bit_pos) := bit;
								tmp(right + bit_pos) := bit;
								A <= av; B <= bv; wait for 1 ns;
								check_logic("Y", Y, tmp);
							end loop;
							av(sel * simd + bit_pos) := 'X';
							tmp(right + bit_pos) := '-';
						end loop;
					end loop;
				end loop;
			end loop;
		end test_permute;

		procedure test_mix is
			variable ab : std_ulogic_vector(2*WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, index, chunk : natural;
		begin
			writestr("*** testing mix instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= shlmode(FCOP_MIX, gran);
				-- main test loop
				ab := (others => 'X');
				for bit_pos in 0 to 2 * WIDTH-1 loop
					index := bit_pos rem simd;
					chunk := bit_pos rem WIDTH - index;
					if bit_pos < WIDTH then
						index := 2 * chunk + index;
					else
						index := 2 * chunk + simd + index;
					end if;
					tmp := (others => '-');
					for bit in std_ulogic_1 downto std_ulogic_0 loop
						ab(bit_pos) := bit;
						A <= ab(WIDTH-1 downto 0);
						B <= ab(2*WIDTH-1 downto WIDTH);
						wait for 1 ns;
						if index < WIDTH then
							tmp(index) := bit;
							check_logic("Y", Y, tmp);
						else
							tmp(index - WIDTH) := bit;
							check_logic("Z", Z, tmp);
						end if;
					end loop;
					ab(bit_pos) := 'X';
				end loop;
			end loop;
		end test_mix;

		procedure test_expand is
			variable ab : std_ulogic_vector(2*WIDTH-1 downto 0);
			variable tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable simd, index : natural;
		begin
			writestr("*** testing expand instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= shlmode(FCOP_EXPAND, gran);
				-- main test loop
				ab := (others => 'X');
				for bit_pos in 0 to 2*WIDTH-1 loop
					if (bit_pos / simd) rem 2 = 0 then
						if bit_pos < WIDTH then
							index := bit_pos;
						else
							index := bit_pos - WIDTH + simd;
						end if;
					else
						if bit_pos < WIDTH then
							index := bit_pos + WIDTH - simd;
						else
							index := bit_pos;
						end if;
					end if;
					tmp := (others => '-');
					for bit in std_ulogic_1 downto std_ulogic_0 loop
						ab(bit_pos) := bit;
						A <= ab(WIDTH-1 downto 0);
						B <= ab(2*WIDTH-1 downto WIDTH);
						wait for 1 ns;
						if index < WIDTH then
							tmp(index) := bit;
							check_logic("Y", Y, tmp);
						else
							tmp(index - WIDTH) := bit;
							check_logic("Z", Z, tmp);
						end if;
						ab(bit_pos) := 'X';
					end loop;
				end loop;
			end loop;
		end test_expand;

		procedure test_cshift is
			variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
			variable left, right, simd : natural;
		begin
			writestr("*** testing cshift instructions ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				print_mode(simd);
				M <= shlmode(FCOP_CSHIFT, gran);
				tmp := (others => '-');
				for bit_pos in 0 to WIDTH-1 loop
					-- cshiftl
					av := (others => 'X');
					bv := (others => 'X');
					for bit in std_ulogic_1 downto std_ulogic_0 loop
						tmp(bit_pos) := bit;
						if bit_pos < simd then
							bv(bit_pos) := bit;
						else
							av(bit_pos - simd) := bit;
						end if;
						A <= av; B <= bv; wait for 1 ns;
						check_logic("Y", Y, tmp);
					end loop;
					-- cshiftr
					av := (others => 'X');
					bv := (others => 'X');
					for bit in std_ulogic_1 downto std_ulogic_0 loop
						tmp(bit_pos) := bit;
						if bit_pos + simd < WIDTH then
							av(bit_pos + simd) := bit;
						else
							bv(bit_pos + simd - WIDTH) := bit;
						end if;
						A <= av; B <= bv; wait for 1 ns;
						check_logic("Z", Z, tmp);
					end loop;
					tmp(bit_pos) := '-';
				end loop;
			end loop;
		end test_cshift;
	begin
		writestr("+++ testing bytewise instructions +++");
		CommonShiftCount <= 'X';
		test_byterev(false);
		test_byterev(true);
		test_permute;
		test_mix;
		test_expand;
		test_cshift;
		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

--pragma synthesis_on

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
