-- shuffle64_test.vhdl -- Testbench for 64-Bit F-CPU Bit Shuffling Unit
-- Copyright (C) 2001 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-- $Id: shuffle64_test.vhdl,v 1.6 2001/11/02 20:07:45 michael Exp $

--pragma synthesis_off

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

entity Shuffle64_test is
	generic (
		TEST_BITWISE : boolean := true;
		TEST_BITREV : boolean := true;
		TEST_BITREV_SIMD : boolean := true;
		TEST_BYTEWISE : boolean := true
	);
end Shuffle64_test;

architecture Arch_1 of Shuffle64_test is
	component Shuffle64
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			ShiftL : in std_ulogic;
			ShiftR : in std_ulogic;
			ShiftRA : in std_ulogic;
			RotL : in std_ulogic;
			RotR : in std_ulogic;
			Bitrev : in std_ulogic;
			Byterev : in std_ulogic;
			Sdup : in std_ulogic;
			Mix : in std_ulogic;
			Expand : in std_ulogic;
			U : in std_ulogic_vector(2 downto 0);
			Clk : in std_ulogic;
			Rst : in std_ulogic;
			En : in std_ulogic;
		--
			Y : out std_ulogic_vector(WIDTH-1 downto 0);
			Y2 : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	constant WIDTH : natural := 64;

	signal M : std_ulogic_vector(12 downto 0) := (others => '0');
	signal A, B, Y, Z : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');
	signal Clk : std_ulogic := '0';
	signal Rst : std_ulogic := '0';
	signal En : std_ulogic := '1';

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 a_x, a_y : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("Y := ")); write(lout, Y); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, a_x);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, a_y);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length
			report "bad args in check_logic" severity failure;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : Shuffle64
		generic map (WIDTH => WIDTH)
		port map (
			A => A, B => B, U => M(2 downto 0),
			ShiftL => M(3), ShiftR => M(4), ShiftRA => M(5),
			RotL => M(6), RotR => M(7), Bitrev => M(8),
			Byterev => M(9), Sdup => M(10),
			Mix => M(11), Expand => M(12),
			Clk => Clk, Rst => Rst, En => En,
			Y => Y, Y2 => Z
		);

	-- driver process
	process
		variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		variable simd, right, index, chunk : natural;
		variable lout : line;
	begin
		if TEST_BITWISE then
			M <= "0000000001XXX";	-- ShiftL
			writestr("*** testing shiftl instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => 'X');
				for shift_count in 0 to simd-1 loop
					bv(gran+2 downto 0) :=
						std_ulogic_vector(to_unsigned(shift_count, gran+3));
					for bit_pos in 0 to WIDTH-1 loop
						av(bit_pos) := '1';
						tmp := (others => '0');
						if (bit_pos mod simd) + shift_count < simd then
							tmp(bit_pos + shift_count) := '1';
						end if;
						A <= av; B <= bv; wait for 1 ns;
						check_logic("Y", Y, tmp);
						tmp := not tmp;
						for i in 0 to WIDTH-1 loop
							if (i mod simd) < shift_count then
								tmp(i) := '0';
							end if;
						end loop;
						A <= not av; wait for 1 ns;
						check_logic("Y", Y, tmp);
						av(bit_pos) := '0';
					end loop;
				end loop;
			end loop;

			M <= "0000000010XXX";	-- ShiftR
			writestr("*** testing shiftr instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => 'X');
				for shift_count in 0 to simd-1 loop
					bv(gran+2 downto 0) :=
						std_ulogic_vector(to_unsigned(shift_count, gran+3));
					for bit_pos in 0 to WIDTH-1 loop
						av(bit_pos) := '1';
						tmp := (others => '0');
						if (bit_pos mod simd) >= shift_count then
							tmp(bit_pos - shift_count) := '1';
						end if;
						A <= av; B <= bv; wait for 1 ns;
						check_logic("Y", Y, tmp);
						tmp := not tmp;
						for i in 0 to WIDTH-1 loop
							if (i mod simd) + shift_count >= simd then
								tmp(i) := '0';
							end if;
						end loop;
						A <= not av; wait for 1 ns;
						check_logic("Y", Y, tmp);
						av(bit_pos) := '0';
					end loop;
				end loop;
			end loop;

			M <= "0000000100XXX";	-- ShiftRA
			writestr("*** testing shiftra instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => 'X');
				for shift_count in 0 to simd-1 loop
					bv(gran+2 downto 0) :=
						std_ulogic_vector(to_unsigned(shift_count, gran+3));
					for bit_pos in 0 to WIDTH-1 loop
						av(bit_pos) := '1';
						tmp := (others => '0');
						if (bit_pos mod simd) = simd - 1 then
							-- sign bit = 1
							for i in bit_pos downto bit_pos - shift_count loop
								tmp(i) := '1';
							end loop;
						else
							-- other bit = 1
							if (bit_pos mod simd) >= shift_count then
								tmp(bit_pos - shift_count) := '1';
							end if;
						end if;
						A <= av; B <= bv; wait for 1 ns;
						check_logic("Y", Y, tmp);
						tmp := (others => '1');
						if (bit_pos mod simd) = simd - 1 then
							-- sign bit = 0
							for i in bit_pos downto bit_pos - shift_count loop
								tmp(i) := '0';
							end loop;
						else
							-- other bit = 1
							if (bit_pos mod simd) >= shift_count then
								tmp(bit_pos - shift_count) := '0';
							end if;
						end if;
						A <= not av; wait for 1 ns;
						check_logic("Y", Y, tmp);
						av(bit_pos) := '0';
					end loop;
				end loop;
			end loop;

			M <= "0000001000XXX";	-- RotL
			writestr("*** testing rotl instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => 'X');
				for shift_count in 0 to simd-1 loop
					bv(gran+2 downto 0) :=
						std_ulogic_vector(to_unsigned(shift_count, gran+3));
					for bit_pos in 0 to WIDTH-1 loop
						av(bit_pos) := '1';
						right := bit_pos - bit_pos mod simd;
						tmp := (others => '0');
						tmp(right + (bit_pos + shift_count) mod simd) := '1';
						A <= av; B <= bv; wait for 1 ns;
						check_logic("Y", Y, tmp);
						A <= not av; wait for 1 ns;
						check_logic("Y", Y, not tmp);
						av(bit_pos) := '0';
					end loop;
				end loop;
			end loop;

			M <= "0000010000XXX";	-- RotR
			writestr("*** testing rotr instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => 'X');
				for shift_count in 0 to simd-1 loop
					bv(gran+2 downto 0) :=
						std_ulogic_vector(to_unsigned(shift_count, gran+3));
					for bit_pos in 0 to WIDTH-1 loop
						av(bit_pos) := '1';
						right := bit_pos - bit_pos mod simd;
						tmp := (others => '0');
						tmp(right + (bit_pos + simd - shift_count) mod simd) := '1';
						A <= av; B <= bv; wait for 1 ns;
						check_logic("Y", Y, tmp);
						A <= not av; wait for 1 ns;
						check_logic("Y", Y, not tmp);
						av(bit_pos) := '0';
					end loop;
				end loop;
			end loop;
		end if;

		if TEST_BITREV then
			M <= "0000100000XXX";	-- Bitrev
			writestr("*** testing bitrev instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => 'X');
				for shift_count in 0 to simd-1 loop
					bv(gran+2 downto 0) :=
						std_ulogic_vector(to_unsigned(shift_count, gran+3));
					for bit_pos in 0 to WIDTH-1 loop
						av(bit_pos) := '1';
						tmp := (others => '0');
						if TEST_BITREV_SIMD then
							index := bit_pos mod simd;
							chunk := bit_pos - index;
						else
							index := bit_pos;
							chunk := 0;
						end if;
						if index + shift_count < simd then
							tmp(chunk + simd - 1 - index - shift_count) := '1';
						end if;
						A <= av; B <= bv; wait for 1 ns;
						check_logic("Y", Y, tmp);
						av(bit_pos) := '0';
					end loop;
				end loop;
			end loop;
		end if;

		if TEST_BYTEWISE then
			M <= "0001000000XXX";	-- Byterev
			writestr("*** testing byterev instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => 'X');
				for bit_pos in 0 to WIDTH-1 loop
					av(bit_pos) := '1';
					right := bit_pos - bit_pos mod simd;
					index := simd - bit_pos mod simd - 1;
					index := index - index mod 8;
					tmp := (others => '0');
					tmp(right + index + bit_pos mod 8) := '1';
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					A <= not av; wait for 1 ns;
					check_logic("Y", Y, not tmp);
					av(bit_pos) := '0';
				end loop;
			end loop;

			M <= "0010000000XXX";	-- Sdup
			writestr("*** testing sdup instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => 'X');
				for bit_pos in 0 to WIDTH-1 loop
					av(bit_pos) := '1';
					tmp := (others => '0');
					for i in 0 to WIDTH-1 loop
						if i mod simd = bit_pos then
							tmp(i) := '1';
						end if;
					end loop;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					A <= not av; wait for 1 ns;
					check_logic("Y", Y, not tmp);
					av(bit_pos) := '0';
				end loop;
			end loop;

			M <= "0100000000XXX";	-- MixL
			writestr("*** testing mixl instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => '0');
				for bit_pos in 0 to WIDTH-1 loop
					av(bit_pos) := '1';
					index := bit_pos mod simd;
					chunk := bit_pos / simd;
					index := 2 * simd * chunk + index;
					tmp := (others => '0');
					if index < WIDTH then
						tmp(index) := '1';
					end if;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					A <= not av; B <= not bv; wait for 1 ns;
					check_logic("Y", Y, not tmp);
					av(bit_pos) := '0';

					bv(bit_pos) := '1';
					index := index + simd;
					tmp := (others => '0');
					if index < WIDTH then
						tmp(index) := '1';
					end if;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					A <= not av; B <= not bv; wait for 1 ns;
					check_logic("Y", Y, not tmp);
					bv(bit_pos) := '0';
				end loop;
			end loop;

			M <= "0100000000XXX";	-- MixH
			writestr("*** testing mixh instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => '0');
				for bit_pos in 0 to WIDTH-1 loop
					av(bit_pos) := '1';
					index := bit_pos mod simd;
					chunk := bit_pos / simd;
					index := 2 * simd * chunk + index;
					tmp := (others => '0');
					if index >= WIDTH then
						tmp(index - WIDTH) := '1';
					end if;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Z", Z, tmp);
					A <= not av; B <= not bv; wait for 1 ns;
					check_logic("Z", Z, not tmp);
					av(bit_pos) := '0';

					bv(bit_pos) := '1';
					index := index + simd;
					tmp := (others => '0');
					if index >= WIDTH then
						tmp(index - WIDTH) := '1';
					end if;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Z", Z, tmp);
					A <= not av; B <= not bv; wait for 1 ns;
					check_logic("Z", Z, not tmp);
					bv(bit_pos) := '0';
				end loop;
			end loop;

			M <= "1000000000XXX";	-- ExpandL
			writestr("*** testing expandl instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => '0');
				for bit_pos in 0 to 2*WIDTH-1 loop
					if bit_pos < WIDTH then
						av(bit_pos) := '1';
					else
						bv(bit_pos - WIDTH) := '1';
					end if;
					tmp := (others => '0');
					if (bit_pos / simd) mod 2 = 0 then
						if bit_pos < WIDTH then
							tmp(bit_pos) := '1';
						else
							tmp(bit_pos + simd - WIDTH) := '1';
						end if;
					end if;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					A <= not av; B <= not bv; wait for 1 ns;
					check_logic("Y", Y, not tmp);
					if bit_pos < WIDTH then
						av(bit_pos) := '0';
					else
						bv(bit_pos - WIDTH) := '0';
					end if;
				end loop;
			end loop;

			M <= "1000000000XXX";	-- ExpandH
			writestr("*** testing expandh instruction ***");
			for gran in 0 to 3 loop
				simd := 8 * 2**gran;
				write(lout, string'("*** testing "));
				write(lout, simd);
				write(lout, string'("-bit mode ***"));
				writeline(output, lout);
				-- set simd mode bits
				for n in 0 to 2 loop
					if gran > n then
						M(n) <= '1';
					else
						M(n) <= '0';
					end if;
				end loop;
				-- main test loop
				av := (others => '0');
				bv := (others => '0');
				for bit_pos in 0 to 2*WIDTH-1 loop
					if bit_pos < WIDTH then
						av(bit_pos) := '1';
					else
						bv(bit_pos - WIDTH) := '1';
					end if;
					tmp := (others => '0');
					if (bit_pos / simd) mod 2 = 1 then
						if bit_pos < WIDTH then
							tmp(bit_pos - simd) := '1';
						else
							tmp(bit_pos - WIDTH) := '1';
						end if;
					end if;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Z", Z, tmp);
					A <= not av; B <= not bv; wait for 1 ns;
					check_logic("Z", Z, not tmp);
					if bit_pos < WIDTH then
						av(bit_pos) := '0';
					else
						bv(bit_pos - WIDTH) := '0';
					end if;
				end loop;
			end loop;
		end if;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;

--pragma synthesis_on

-- vi: set ts=4 sw=4 equalprg="fmt -72 -p--": please
