-- shltest.vhdl - Testbench for Shuffle unit
-- Copyright (C) 2000, 2001 Michael Riepe <michael@stud.uni-hannover.de>
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

-- $Id$

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_textio.all;
use IEEE.numeric_std.all;
use std.textio.all;

entity Shuffle_test is
end Shuffle_test;

architecture Arch_1 of Shuffle_test is
	component Shuffle
		generic (
			WIDTH : natural := 64
		);
		port (
			A : in std_ulogic_vector(WIDTH-1 downto 0);
			B : in std_ulogic_vector(WIDTH-1 downto 0);
			ShiftL : in std_ulogic := '0';
			ShiftR : in std_ulogic := '0';
			ShiftRA : in std_ulogic := '0';
			RotL : in std_ulogic := '0';
			RotR : in std_ulogic := '0';
			Bitrev : in std_ulogic := '0';
			Byterev : in std_ulogic := '0';
			Sdup : in std_ulogic := '0';
			U08, U16, U32 : in std_ulogic := '0';
		--
			Y : out std_ulogic_vector(WIDTH-1 downto 0)
		);
	end component;

	constant WIDTH : natural := 64;

	signal M : std_ulogic_vector(10 downto 0) := (others => '0');
	signal A, B, Y : std_ulogic_vector(WIDTH-1 downto 0) := (others => '0');

	procedure writestr (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end writestr;

	procedure do_report (lbl : string;
						 a_x, a_y : std_ulogic_vector) is
		variable lout : line;
	begin
		write(lout, string'("WHOA THERE!!!")); writeline(output, lout);
		write(lout, string'("A := ")); write(lout, A); writeline(output, lout);
		write(lout, string'("B := ")); write(lout, B); writeline(output, lout);
		write(lout, string'("M := ")); write(lout, M); writeline(output, lout);
		write(lout, string'("Y := ")); write(lout, Y); writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" := "));
		write(lout, a_x);
		writeline(output, lout);
		write(lout, lbl);
		write(lout, string'(" /= "));
		write(lout, a_y);
		writeline(output, lout);
	end do_report;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_report(lbl, x, tmp);
		end if;
	end check_numeric;

	procedure check_logic (lbl : string;
						   a, b : std_ulogic_vector) is
		alias x : std_ulogic_vector(a'length downto 1) is a;
		alias y : std_ulogic_vector(b'length downto 1) is b;
		variable lout : line;
	begin
		assert a'length = b'length;
		for i in x'range loop
			next when y(i) = '-';
			next when x(i) = y(i);
			do_report(lbl, x, y);
			return;
		end loop;
	end check_logic;
begin
	-- module under test
	mut : Shuffle
		generic map (WIDTH => WIDTH)
		port map (
			A => A, B => B, U08 => M(0), U16 => M(1), U32 => M(2),
			ShiftL => M(3), ShiftR => M(4), ShiftRA => M(5),
			RotL => M(6), RotR => M(7), Bitrev => M(8), Byterev => M(9),
			Sdup => M(10), Y => Y
		);

	-- driver process
	process
		variable av, bv, tmp : std_ulogic_vector(WIDTH-1 downto 0);
		variable simd, right, index : natural;
		variable lout : line;
	begin
		M <= "00000001XXX";	-- ShiftL
		writestr("*** testing shiftl instruction ***");
		for gran in 0 to 3 loop
			simd := 8 * 2**gran;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					M(n) <= '1';
				else
					M(n) <= '0';
				end if;
			end loop;
			-- main test loop
			av := (others => 'X');
			bv := (others => 'X');
			for shift_count in 0 to simd-1 loop
				bv(gran+2 downto 0) :=
					std_ulogic_vector(to_unsigned(shift_count, gran+3));
				for bit_pos in 0 to WIDTH-1 loop
					av(bit_pos) := '1';
					tmp := (others => 'X');
					for i in 0 to WIDTH-1 loop
						if (i mod simd) < shift_count then
							tmp(i) := '0';
						end if;
					end loop;
					if (bit_pos mod simd) + shift_count < simd then
						tmp(bit_pos + shift_count) := '1';
					end if;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					av(bit_pos) := 'X';
				end loop;
			end loop;
		end loop;

		M <= "00000010XXX";	-- ShiftR
		writestr("*** testing shiftr instruction ***");
		for gran in 0 to 3 loop
			simd := 8 * 2**gran;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					M(n) <= '1';
				else
					M(n) <= '0';
				end if;
			end loop;
			-- main test loop
			av := (others => 'X');
			bv := (others => 'X');
			for shift_count in 0 to simd-1 loop
				bv(gran+2 downto 0) :=
					std_ulogic_vector(to_unsigned(shift_count, gran+3));
				for bit_pos in 0 to WIDTH-1 loop
					av(bit_pos) := '1';
					tmp := (others => 'X');
					for i in 0 to WIDTH-1 loop
						if (i mod simd) + shift_count >= simd then
							tmp(i) := '0';
						end if;
					end loop;
					if (bit_pos mod simd) >= shift_count then
						tmp(bit_pos - shift_count) := '1';
					end if;
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					av(bit_pos) := 'X';
				end loop;
			end loop;
		end loop;

		M <= "00001000XXX";	-- RotL
		writestr("*** testing rotl instruction ***");
		for gran in 0 to 3 loop
			simd := 8 * 2**gran;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					M(n) <= '1';
				else
					M(n) <= '0';
				end if;
			end loop;
			-- main test loop
			av := (others => 'X');
			bv := (others => 'X');
			for shift_count in 0 to simd-1 loop
				bv(gran+2 downto 0) :=
					std_ulogic_vector(to_unsigned(shift_count, gran+3));
				for bit_pos in 0 to WIDTH-1 loop
					av(bit_pos) := '1';
					right := bit_pos - bit_pos mod simd;
					tmp := (others => 'X');
					tmp(right + (bit_pos + shift_count) mod simd) := '1';
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					av(bit_pos) := 'X';
				end loop;
			end loop;
		end loop;

		M <= "00010000XXX";	-- RotR
		writestr("*** testing rotr instruction ***");
		for gran in 0 to 3 loop
			simd := 8 * 2**gran;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					M(n) <= '1';
				else
					M(n) <= '0';
				end if;
			end loop;
			-- main test loop
			av := (others => 'X');
			bv := (others => 'X');
			for shift_count in 0 to simd-1 loop
				bv(gran+2 downto 0) :=
					std_ulogic_vector(to_unsigned(shift_count, gran+3));
				for bit_pos in 0 to WIDTH-1 loop
					av(bit_pos) := '1';
					right := bit_pos - bit_pos mod simd;
					tmp := (others => 'X');
					tmp(right + (bit_pos + simd - shift_count) mod simd) := '1';
					A <= av; B <= bv; wait for 1 ns;
					check_logic("Y", Y, tmp);
					av(bit_pos) := 'X';
				end loop;
			end loop;
		end loop;

		M <= "01000000XXX";	-- Byterev
		writestr("*** testing byterev instruction ***");
		for gran in 0 to 3 loop
			simd := 8 * 2**gran;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					M(n) <= '1';
				else
					M(n) <= '0';
				end if;
			end loop;
			-- main test loop
			av := (others => 'X');
			bv := (others => 'X');
			for bit_pos in 0 to WIDTH-1 loop
				av(bit_pos) := '1';
				right := bit_pos - bit_pos mod simd;
				index := simd - bit_pos mod simd - 1;
				index := index - index mod 8;
				tmp := (others => 'X');
				tmp(right + index + bit_pos mod 8) := '1';
				A <= av; B <= bv; wait for 1 ns;
				check_logic("Y", Y, tmp);
				av(bit_pos) := 'X';
			end loop;
		end loop;

		M <= "10000000XXX";	-- Sdup
		writestr("*** testing sdup instruction ***");
		for gran in 0 to 3 loop
			simd := 8 * 2**gran;
			write(lout, string'("*** testing "));
			write(lout, simd);
			write(lout, string'("-bit mode ***"));
			writeline(output, lout);
			-- set simd mode bits
			for n in 0 to 2 loop
				if gran > n then
					M(n) <= '1';
				else
					M(n) <= '0';
				end if;
			end loop;
			-- main test loop
			av := (others => 'X');
			bv := (others => 'X');
			for bit_pos in 0 to WIDTH-1 loop
				av(bit_pos) := '1';
				tmp := (others => 'X');
				for i in 0 to WIDTH-1 loop
					if i mod simd = bit_pos then
						tmp(i) := '1';
					end if;
				end loop;
				A <= av; B <= bv; wait for 1 ns;
				check_logic("Y", Y, tmp);
				av(bit_pos) := 'X';
			end loop;
		end loop;

		-- stop simulation
		writestr("*** simulation complete ***");
		wait;
	end process;
end Arch_1;
