%{
#include <stdio.h>
#include "globals.h"
#include "fcpu_asm.h"

  extern char *yytext;
  
  int line = 1;

  f_cpu_intermediate instr;
  
%}
%start lines

%token SPACE EOL
%token DOT COLON COMMA PLUS MINUS STAR DOLLAR
%token OP_BRACKET CL_BRACKET OP_PAREN CL_PAREN
%token IDENTIFIER REGISTER INTEGER
%token ADD SUB MUL DIV MOD SHIFT ROT AND OR XOR LOGIC
%token MOV LOAD STORE TEST GET PUT PREFETCH SYSCALL TRAP
%token JMPRA JMPIR JMPRIA
%token FADD FSUB FMUL FDIV FINV FSQRT FINVSQRT F2INT INT2F
%%

lines
:
| lines stat
;

stat
: EOL                               { line++; }      
| instruction EOL                   { line++; }
| label EOL                         { line++; }    
;

label
: DOT IDENTIFIER COLON            
;

constant
: DOLLAR INTEGER;

adress
: OP_BRACKET REGISTER CL_BRACKET         { instr.format = 1;
                                           instr.reg1 = $2.ival; }
| OP_BRACKET REGISTER PLUS REGISTER CL_BRACKET
                                         
| OP_BRACKET REGISTER PLUS REGISTER STAR INTEGER CL_BRACKET
| OP_BRACKET REGISTER PLUS constant STAR INTEGER CL_BRACKET
;

operands
: REGISTER COMMA REGISTER COMMA REGISTER { instr.format = 1;
                                           instr.reg1 = $1.ival; 
                                           instr.reg2 = $3.ival; 
                                           instr.reg3 = $5.ival; }
| REGISTER COMMA constant COMMA REGISTER { instr.format = 2;
                                           instr.reg1 = $1.ival; 
                                           instr.reg2 = $5.ival; 
                                           instr.imm  = $3.ival; }
| REGISTER COMMA REGISTER                { instr.format = 1;
                                           instr.reg1 = $1.ival; 
                                           instr.reg2 = $3.ival; } 
| REGISTER COMMA constant                { instr.format = 2;
                                           instr.reg1 = $1.ival; 
                                           instr.imm  = $3.ival; }
| adress COMMA REGISTER                  { instr.reg3  = $3.ival; }
| REGISTER COMMA adress
| DOT IDENTIFIER
| REGISTER
| constant
;

arithm
: ADD                        { instr.opcode = OP_ADD; } 
| SUB                        { instr.opcode = OP_ADD;
                               instr.arithm = ARITHM_SUB; } 
| MUL                        { instr.opcode = OP_MUL; }
| DIV                        { instr.opcode = OP_DIV; }
| MOD                        { instr.opcode = OP_DIV;
                               instr.arithm = ARITHM_MOD; } 
| SHIFT                      { instr.opcode = OP_SHIFT; }
| ROT                        { instr.opcode = OP_ROT; }
| AND                        { instr.opcode = OP_LOGIC; 
                               instr.logic  = LOGIC_AND; }
| OR                         { instr.opcode = OP_LOGIC; 
                               instr.logic  = LOGIC_OR;  }
| XOR                        { instr.opcode = OP_LOGIC; 
                               instr.logic  = LOGIC_XOR; }
| LOGIC                      { instr.opcode = OP_LOGIC; }
;

fpu
: FADD                       { instr.opcode = OP_FADD; }                
| FSUB                       { instr.opcode = OP_FSUB; }
| FMUL                       { instr.opcode = OP_FMUL; }
| FDIV                       { instr.opcode = OP_FDIV; }
| FINV                       { instr.opcode = OP_FINV; }
| FSQRT                      { instr.opcode = OP_FSQRT; } 
| FINVSQRT                   { instr.opcode = OP_FINVSQRT; }
| F2INT                      { instr.opcode = OP_F2INT;
                               instr.direc  = FLOAT_INT2F; }
| INT2F                      { instr.opcode = OP_INT2F; 
                               instr.direc  = FLOAT_F2INT; }
;
loadstore
: LOAD                       { instr.opcode = OP_LOAD; }
| STORE                      { instr.opcode = OP_STORE; }       
| MOV                        { instr.opcode = OP_MOV; }
;
control
: JMPRA                      { instr.opcode = OP_JMPRA; }
| JMPIR                      { instr.opcode = OP_JMPIR; }
| JMPRIA                     { instr.opcode = OP_JMPRIA; }
| TEST                       { instr.opcode = OP_TEST; }
| GET                        { instr.opcode = OP_GET; }
| PUT                        { instr.opcode = OP_PUT; }
| PREFETCH                   { instr.opcode = OP_PREFETCH; }
| SYSCALL                    { instr.opcode = OP_SYSCALL; }
| TRAP;                      { instr.opcode = OP_TRAP; }

opcode
: arithm
| fpu
| loadstore
| control
;

flags
:
|  DOT IDENTIFIER
|  DOT INTEGER
;

instruction 
: opcode flags operands
;

%%

main ()
{
  yyparse ();
}

void yyerror(char *mesg)
{       /* Type is either fSYN or fSYM  */
  if ( !strcmp(mesg,"syntax error") )
    {
      printf("asm: error on line %d: %s\n",line,mesg);
      printf("\tcurrent token = %s\n",yytext);
      printf("%d\n", yylval.token);
      exit(-1);
    }
  else
    {      
      printf("asm: warning on line %d: %s\n", line, mesg);
    }
}

