//
// FCPU Emulator 2.0
//

#include "stdio.h"
#include "fcpudefs.h"

#define MAX_BREAKPOINTS 9

typedef struct _EBPD {
    char    valid;
    UL64    exceptionmask;
} EBPD, *PEBPD;

typedef struct _FED {
    PFCPU   pcpu;
    EBPD    bpd[ MAX_BREAKPOINTS ];
    char    quit;
} FED, *PFED;

void fcpu_emu_dump_ul08( UL08 x )
{
    char buffer[ 128 ];

    sprintf( buffer, "%02X", x );
    printf( buffer );
}

void fcpu_emu_dump_ul08_dec( UL08 x )
{
    char buffer[ 128 ];

    sprintf( buffer, "%02u", x );
    printf( buffer );
}

void fcpu_emu_dump_ul64( UL64 x )
{
    char            buffer[ 128 ];
    unsigned long   ulLow;
    unsigned long   ulHigh;

    ulHigh = (unsigned long)( ( x >> 32 ) & 0xFFFFFFFF );
    ulLow  = (unsigned long)( ( x >>  0 ) & 0xFFFFFFFF );
        
    sprintf( buffer, "%08X%08X", ulHigh, ulLow );
    printf( buffer );
}

void fcpu_emu_dump_float( double x )
{
    char            buffer[ 128 ];

    sprintf( buffer, "%16E", x );
    printf( buffer );
}

void fcpu_emu_dump_cpu_regs_hexint( PFED p, UL64 rb, UL64 re )
{
    UL64 i;
    UL64 j;

    j = 0;
    for( i = rb; i <= re; i++ )
    {
        printf( "R" );
        fcpu_emu_dump_ul08_dec( (UL08)i );
        printf( " " );
        fcpu_emu_dump_ul64( p->pcpu->reg[ i ] );
        printf( "  " );

        if( ( j % 3 ) == 2 )
            printf( "\n" );
        j++;
    }
}

void fcpu_emu_dump_cpu_exception( PFED p )
{
    if( p->pcpu->exception != 0 )
        printf( "active exceptions:\n" );
    if( p->pcpu->exception & FCPU_EXCEPTION_RESET )
        printf( "    reset\n" );
    if( p->pcpu->exception & FCPU_EXCEPTION_HALT )
        printf( "    halt\n" );
    if( p->pcpu->exception & FCPU_EXCEPTION_INVALID_INSTRUCTION )
        printf( "    invalidinstruction\n" );
    if( p->pcpu->exception & FCPU_EXCEPTION_PAGEFAULT )
        printf( "    pagefault\n" );
    if( p->pcpu->exception & FCPU_EXCEPTION_PROTECTIONFAULT )
        printf( "    protectionfault\n" );
    if( p->pcpu->exception & FCPU_EXCEPTION_ALIGNMENTFAULT )
        printf( "    alignmentfault\n" );
    if( p->pcpu->exception & FCPU_EXCEPTION_DIVISION_BY_ZERO )
        printf( "    devisionbyzero\n" );
    if( p->pcpu->exception & FCPU_EXCEPTION_TIMER_COUNT_ZERO )
        printf( "    timercountzero\n" );
}

void fcpu_emu_dump_cpu( PFED p )
{
    fcpu_emu_dump_cpu_regs_hexint( p, 0, 63 );
    fcpu_emu_dump_cpu_exception( p );
}

char fcpu_emu_check_breakpoint_active( PFED p )
{
    unsigned long i;

    for( i = 0; i < MAX_BREAKPOINTS; i++ )
    {
        if( p->bpd[ i ].valid )
        {
            if( p->pcpu->exception & p->bpd[ i ].exceptionmask )
                return 1;
        }
    }

    return 0;
}

char fcpu_emu_check_cpu_halted( PFED p )
{
    if( p->pcpu->exception & FCPU_EXCEPTION_HALT )
        return 1;

    return 0;
}

void fcpu_emu_exec_go_command( PFED p )
{
    while( ( !fcpu_emu_check_breakpoint_active( p ) ) &&
           ( !fcpu_emu_check_cpu_halted( p )        ) )
        fcpu_cpu_clock( p->pcpu );

    fcpu_emu_dump_cpu( p );
}

void fcpu_emu_exec_step_command( PFED p )
{
    fcpu_cpu_clock( p->pcpu );
    
    fcpu_emu_dump_cpu( p );
}

void fcpu_emu_exec_reset_command( PFED p )
{
    fcpu_cpu_reset( p->pcpu );
    
    fcpu_emu_dump_cpu( p );
}

void fcpu_emu_exec_dumpcpu_command( PFED p )
{
    fcpu_emu_dump_cpu( p );
}

void fcpu_emu_exec_load_and_assemble_command( PFED p, char* pszf )
{
    fcpu_asm_assemble_file( pszf );
}

UL64 fcpu_emu_get_register_id( PFED p, char* pszr )
{
    UL64 c0;
    UL64 c1;
    char err;

    if( fcpu_utl_length_of_string( pszr ) < 2 )
        return 0;

    if( ( *pszr != 'R' ) && ( *pszr != 'r' ) )
        return 0;

    err = 0;
    c0 = fcpu_utl_convert_digit_to_constant( *( pszr + 1 ), 10, &err );
    if( err )
        c0 = 0;

    if( *( pszr + 2 ) )
    {
        err = 0;
        c1 = fcpu_utl_convert_digit_to_constant( *( pszr + 2 ), 10, &err );
        if( err )
            c1 = 0;

        return 10 * c0 + c1;
    }
    else
        return c0;
}

void fcpu_emu_get_param_reg_begin_end( PFED p, char* pszp, UL64* prb, UL64* pre )
{
    char    szrb[4];
    char    szre[4];

    fcpu_utl_memset( &szrb[ 0 ], 0, sizeof( szrb ) );
    while( ( *pszp != 0 ) && ( *pszp != '-' ) && ( *pszp != ',' ) )
    {
        fcpu_utl_append_char_to_string( *pszp, &szrb[ 0 ], sizeof( szrb ) );
        pszp++;
    }

    if( ( *pszp == ',' ) || ( *pszp == '-' ) )
        pszp++;

    fcpu_utl_memset( &szre[ 0 ], 0, sizeof( szre ) );
    while( ( *pszp != 0 ) && ( *pszp != '-' ) && ( *pszp != ',' ) )
    {
        fcpu_utl_append_char_to_string( *pszp, &szre[ 0 ], sizeof( szre ) );
        pszp++;
    }

    if( ( szrb[ 0 ] == 0 ) && ( szre[ 0 ] == 0 ) )
    {
        *prb = 0;
        *pre = 63;
    }
    else if( ( szrb[ 0 ] != 0 ) && ( szre[ 0 ] == 0 ) )
    {
        *prb = fcpu_emu_get_register_id( p, &szrb[ 0 ] ) & 0x3F;;
        *pre = *prb;
    }
    else
    {
        *prb = fcpu_emu_get_register_id( p, &szrb[ 0 ] ) & 0x3F;
        *pre = fcpu_emu_get_register_id( p, &szre[ 0 ] ) & 0x3F;
    }
}

void fcpu_emu_exec_dump_regs_integer( PFED p, char* pszp )
{
    UL64    rb;
    UL64    re;

    fcpu_emu_get_param_reg_begin_end( p, pszp, &rb, &re );

    fcpu_emu_dump_cpu_regs_hexint( p, rb, re );
}

void fcpu_emu_exec_dump_regs_float_single( PFED p, char* pszp )
{
    float   fop;
    UL64    op;
    UL64    rb;
    UL64    re;
    UL64    i;

    fcpu_emu_get_param_reg_begin_end( p, pszp, &rb, &re );

    for( i = rb; i <= re; i++ )
    {
        printf( "R" );
        fcpu_emu_dump_ul08_dec( (UL08)i );
        printf( " " );

        op = p->pcpu->reg[ i ];
        fop = *( (float*)( (unsigned char*)&op + 4 ) );
        fcpu_emu_dump_float( (double)fop );

        printf( " : " );
        
        op = p->pcpu->reg[ i ];
        fop = *( (float*)( (unsigned char*)&op + 4 ) );
        fcpu_emu_dump_float( (double)fop );

        printf( "\n" );       
    }
}

void fcpu_emu_exec_dump_regs_float_double( PFED p, char* pszp )
{
    double  fop;
    UL64    op;
    UL64    rb;
    UL64    re;
    UL64    i;

    fcpu_emu_get_param_reg_begin_end( p, pszp, &rb, &re );

    for( i = rb; i <= re; i++ )
    {
        op = p->pcpu->reg[ i ];
        fop = *( (float*)&op );

        printf( "R" );
        fcpu_emu_dump_ul08_dec( (UL08)i );
        printf( " " );
        fcpu_emu_dump_float( fop );
        printf( "\n" );       
    }
}

void fcpu_emu_get_exec_user_command( PFED p )
{
    char command[ 16 ];
    char paramet[ 128 ];
    int  ich;

    printf( ">" );

    p->quit = 0;

    fcpu_utl_memset( &command[ 0 ], 0, sizeof( command ) );
    fcpu_utl_memset( &paramet[ 0 ], 0, sizeof( paramet ) );

    ich = getchar();
    while( ( ich != EOF ) && ( ich != '\n' ) && ( ich != ' ' ) )
    {
        fcpu_utl_append_char_to_string( (char)ich, &command[ 0 ], sizeof( command ) ); 
        ich = getchar();
    }
    while( ( ich != EOF ) && ( ich != '\n' ) && ( ich == ' ' )  )
    {
        ich = getchar();
    }
    while( (ich != EOF ) && ( ich != '\n' ) )
    {
        fcpu_utl_append_char_to_string( (char)ich, &paramet[ 0 ], sizeof( paramet ) ); 
        ich = getchar();
    }

    if( ( fcpu_utl_compare_strings_i( &command[ 0 ], "help" ) ) ||
        ( fcpu_utl_compare_strings_i( &command[ 0 ], "h"    ) ) ||
        ( fcpu_utl_compare_strings_i( &command[ 0 ], "?"    ) ) ||
        ( fcpu_utl_compare_strings_i( &command[ 0 ], ""     ) ) )
    {
        printf( "available commands:\n" );
        printf( "\n" );
        printf( "help, h, ?    ... print help\n" );
        printf( "go, g         ... run program until halt instruction\n" );
        printf( "step, s       ... single step one instruction\n" );
        printf( "reset, r      ... reset cpu\n" );
        printf( "quit, q       ... quit emulator\n" );
        printf( "dumpcpu, dc   ... dump cpu state\n" );
        printf( "loadasm, la   ... load and assemble file. example: la <filename>\n" );
        printf( "drui          ... dump registers as unsigned hex. integer: dr r5-r15\n" );
        printf( "drfs          ... dump registers as single precision float: drfs r0-r20\n" );
        printf( "drfd          ... dump registers as double precision float: drfd r12-r16\n" );
    }
    else if( ( fcpu_utl_compare_strings_i( &command[ 0 ], "go" ) ) ||
             ( fcpu_utl_compare_strings_i( &command[ 0 ], "g"  ) ) )
    {
        fcpu_emu_exec_go_command( p );
    }
    else if( ( fcpu_utl_compare_strings_i( &command[ 0 ], "step" ) ) ||
             ( fcpu_utl_compare_strings_i( &command[ 0 ], "s"    ) ) )
    {
        fcpu_emu_exec_step_command( p );
    }
    else if( ( fcpu_utl_compare_strings_i( &command[ 0 ], "reset" ) ) ||
             ( fcpu_utl_compare_strings_i( &command[ 0 ], "r"     ) ) )
    {
        fcpu_emu_exec_reset_command( p );
    }
    else if( ( fcpu_utl_compare_strings_i( &command[ 0 ], "quit" ) ) ||
             ( fcpu_utl_compare_strings_i( &command[ 0 ], "q"    ) ) )
    {
        p->quit = 1;
    }
    else if( ( fcpu_utl_compare_strings_i( &command[ 0 ], "dumpcpu" ) ) ||
             ( fcpu_utl_compare_strings_i( &command[ 0 ], "dc"      ) ) )
    {
        fcpu_emu_exec_dumpcpu_command( p );
    }
    else if( ( fcpu_utl_compare_strings_i( &command[ 0 ], "loadasm" ) ) ||
             ( fcpu_utl_compare_strings_i( &command[ 0 ], "la"      ) ) )
    {
        fcpu_emu_exec_load_and_assemble_command( p, &paramet[ 0 ] );
    }
    else if( fcpu_utl_compare_strings_i( &command[ 0 ], "drui" ) )
    {
        fcpu_emu_exec_dump_regs_integer( p, &paramet[ 0 ] );
    }
    else if( fcpu_utl_compare_strings_i( &command[ 0 ], "drfs" ) )
    {
        fcpu_emu_exec_dump_regs_float_single( p, &paramet[ 0 ] );
    }
    else if( fcpu_utl_compare_strings_i( &command[ 0 ], "drfd" ) )
    {
        fcpu_emu_exec_dump_regs_float_double( p, &paramet[ 0 ] );
    }

    printf( "\n" );

}

void fcpu_emu_exec_user_commands( PFED p )
{
    p->quit = 0;
    while( !p->quit )
        fcpu_emu_get_exec_user_command( p );
}

void fcpu_emu_hello( void )
{
    printf( "******************************\n" );
    printf( "*                            *\n" );
    printf( "* F-CPU Virtual Machine V2.0 *\n" );
    printf( "*                            *\n" );
    printf( "******************************\n" );
    printf( "\n" );
}

void fcpu_emu_init( PFED ped, PFCPU pcpu )
{
    unsigned long i;

    ped->pcpu = pcpu;

    for( i = 0; i < MAX_BREAKPOINTS; i++ )
    {
        ped->bpd[ i ].valid = 0;
    }

    ped->quit = 0;
}

void main( int argc, char* argv[] )
{
    FCPU    cpu;
    FED     ed;

    fcpu_emu_hello();

    fcpu_emu_init( &ed, &cpu );
    
    fcpu_cpu_reset( &cpu );

    if( argc > 1 )
    {
        printf( "running " );
        printf( argv[1] );
        printf( "\n\n" );

        if( fcpu_asm_assemble_file( argv[1] ) )
            fcpu_emu_exec_go_command( &ed );
    }
    else
        fcpu_emu_exec_user_commands( &ed );
}