#include "stdio.h"

#include "fcpudefs.h"

#define FCPU_ASM_IDBUFSIZE  128L

typedef struct _ASMSYMDEF {
    void* pnext;
    char* pszs;
    char* pszc;
} ASMSYMDEF, *PASMSYMDEF;

typedef struct _ASMLBLDEF {
    void*   pnext;
    char*   pszl;
    UL64    addr;
} ASMLBLDEF, *PASMLBLDEF;

typedef struct _ASMINSDEF {
    void*   pnext;
    char*   pszop;
    char*   pszp1;
    char*   pszp2;
    char*   pszp3;
    UL64    addr;
} ASMINSDEF, *PASMINSDEF;

typedef struct _ASMDATA {   
    unsigned char*  mp;     // pointer to memory to receive the assembled code
    UL64            ms;     // max size of this memory (ranging from 0 to ms )
    UL64            err;
    UL64            addr;
    PASMSYMDEF      psym;
    PASMLBLDEF      plbl;
    PASMINSDEF      pins;
} ASMDATA, *PASMDATA;

void fcpu_asm_error( PASMDATA p, UL64 err )
{
    if( p )
        p->err = err;

    printf( "assemble error\n" );
}

char fcpu_asm_get_char( FILE* f )
{
    int ch;

    ch = fgetc( f );

    if( ch == EOF )
        return (char)0;

    if( ch == 13 )
        return (char)13;

    if( ch < 32 )
        return ' ';

    if( ch > 127 )
        return ' ';

    return (char)ch;
}

char fcpu_asm_is_whitespace( char ch )
{
    if( ch == ' ' )
        return 1L;

    return 0L;
}

char fcpu_asm_is_identifier_char( char ch )
{
    if( ( ch >= 'A' ) && ( ch <= 'Z' ) )
        return 1L;

    if( ( ch >= 'a' ) && ( ch <= 'z' ) )
        return 1L;

    if( ( ch >= '0' ) && ( ch <= '9' ) )
        return 1L;

    if( ch == '_' ) 
        return 1L;

    if( ch == '.' )
        return 1L;

    if( ch == '@' )
        return 1L;

    return 0L;
}

char fcpu_asm_is_eol( char ch )
{
    if( 13 == ch )
        return 1L;

    return 0L;
}

char fcpu_asm_is_eof( char ch )
{
    if( 0 == ch )
        return 1L;

    return 0L;
}

char fcpu_asm_is_eol_or_eof( char ch )
{
    if( fcpu_asm_is_eol( ch ) )
        return 1L;

    if( fcpu_asm_is_eof( ch ) )
        return 1L;

    return 0L;
}

void fcpu_asm_skip_eol( FILE* f, char* pch )
{
    // Whitespace
    while( !fcpu_asm_is_eol_or_eof( *pch ) )
        *pch = fcpu_asm_get_char( f );

    *pch = fcpu_asm_get_char( f );
}

void fcpu_asm_skip_whitespace( FILE* f, char* pch )
{
    // Whitespace
    while( ( !fcpu_asm_is_eol_or_eof( *pch ) ) &&
           (  fcpu_asm_is_whitespace( *pch ) ) )
        *pch = fcpu_asm_get_char( f );
}

void fcpu_asm_get_identifier( FILE* f, char* pch, char* psz, UL64 m )
{
    while( ( !fcpu_asm_is_eol_or_eof( *pch )      ) &&
           (  fcpu_asm_is_identifier_char( *pch ) ) )
    {        
        fcpu_utl_append_char_to_string( *pch, psz, m );
        *pch = fcpu_asm_get_char( f );
    }
}

void fcpu_asm_get_parameter( FILE* f, char* pch, char* psz, UL64 m )
{
    char s;

    s = 0;
    while( 1 )
    {        
        if( fcpu_asm_is_eol_or_eof( *pch ) )
            break;

        if( !s )
        {
            if( ',' == *pch )
                break;

            if( fcpu_asm_is_whitespace( *pch ) )
                break;

            if( ';' == *pch )
                break;
        }

        if( *pch == '"' )
        {
            if( s )
                s = 0L;
            else
                s = 1L;
        }

        fcpu_utl_append_char_to_string( *pch, psz, m );

        *pch = fcpu_asm_get_char( f );
    }
}

void fcpu_asm_get_line( FILE*   f,
                        char*   pch,
                        char*   label,
                        UL64    maxsize_label,
                        char*   opcode,
                        UL64    maxsize_opcode,
                        char*   parameter1,
                        UL64    maxsize_parameter1,
                        char*   parameter2,
                        UL64    maxsize_parameter2,
                        char*   parameter3,
                        UL64    maxsize_parameter3 )
{
    fcpu_utl_clear_string( label,      maxsize_label      ); 
    fcpu_utl_clear_string( opcode,     maxsize_opcode     );
    fcpu_utl_clear_string( parameter1, maxsize_parameter1 );
    fcpu_utl_clear_string( parameter2, maxsize_parameter2 );
    fcpu_utl_clear_string( parameter3, maxsize_parameter3 );

    // label or opcode
    fcpu_asm_skip_whitespace( f, pch );

    fcpu_asm_get_identifier( f, pch, opcode, maxsize_opcode );

    fcpu_asm_skip_whitespace( f, pch );

    if( *pch == ':' )
    {
        // ':' marks label => opcode follows / predecessor was label

        fcpu_utl_copy_string( label, maxsize_label, opcode );

        fcpu_utl_clear_string( opcode, maxsize_opcode );

        *pch = fcpu_asm_get_char( f );

        fcpu_asm_skip_whitespace( f, pch );

        fcpu_asm_get_identifier( f, pch, opcode, maxsize_opcode );

        fcpu_asm_skip_whitespace( f, pch );
    }

    fcpu_asm_skip_whitespace( f, pch );

    fcpu_asm_get_parameter( f, pch, parameter3, maxsize_parameter3 );

    fcpu_asm_skip_whitespace( f, pch );

    if( *pch == ',' )
        *pch = fcpu_asm_get_char( f );

    fcpu_asm_skip_whitespace( f, pch );

    fcpu_asm_get_parameter( f, pch, parameter2, maxsize_parameter2 );

    fcpu_asm_skip_whitespace( f, pch );

    if( *pch == ',' )
        *pch = fcpu_asm_get_char( f );

    fcpu_asm_skip_whitespace( f, pch );

    fcpu_asm_get_parameter( f, pch, parameter1, maxsize_parameter1 );

    fcpu_asm_skip_whitespace( f, pch );

    fcpu_asm_skip_eol( f, pch );
}

//
//
//

char* fcpu_asm_resolve_symbol( PASMDATA p, char* pszs )
{
    char*       pszc;
    char*       pszcn;
    PASMSYMDEF  ps;

    pszc = NULL;

    // Check for userdefined symbols

    ps   = p->psym;
    while( ( ps ) && ( !pszc ) )
    {
        if( fcpu_utl_compare_strings_i( ps->pszs, pszs ) )
            pszc = ps->pszc;
        else
            ps = (PASMSYMDEF)fcpu_utl_get_next_element_of_linked_list( ps );
    }

    if( pszc )
    {
        pszcn = fcpu_asm_resolve_symbol( p, pszc );

        if( pszcn )
            pszc = pszcn;
    }

    return pszc;
}

void fcpu_asm_define_symbol( PASMDATA p, char* pszs, char* pszc )
{
    PASMSYMDEF ps;

    // link symbol and meaning number into label list of p

    ps = (PASMSYMDEF)fcpu_utl_malloc( sizeof( ASMSYMDEF ) );
    if( !ps )
    {
        fcpu_asm_error( p, 1 );
        return;
    }
    fcpu_utl_memset( ps, 0, sizeof( ASMSYMDEF ) );

    ps->pszs = (char*)fcpu_utl_malloc( fcpu_utl_length_of_string_buffer( pszs ) );
    if( !ps->pszs )
    {
        fcpu_utl_free( (void**)&ps );

        fcpu_asm_error( p, 1 );
        return;
    }
    fcpu_utl_memset( ps->pszs, 0, fcpu_utl_length_of_string_buffer( pszs ) );

    ps->pszc = (char*)fcpu_utl_malloc( fcpu_utl_length_of_string_buffer( pszc ) );
    if( !ps->pszc )
    {
        fcpu_utl_free( (void**)&ps->pszs );
        fcpu_utl_free( (void**)&ps );

        fcpu_asm_error( p, 1 );
        return;
    }
    fcpu_utl_memset( ps->pszc, 0, fcpu_utl_length_of_string_buffer( pszc ) );

    fcpu_utl_copy_string( ps->pszs, fcpu_utl_length_of_string_buffer( pszs ), pszs );
    fcpu_utl_copy_string( ps->pszc, fcpu_utl_length_of_string_buffer( pszc ), pszc );

    fcpu_utl_append_to_linked_list( (void**)&p->psym, ps );
}

void fcpu_asm_clean_symbol_list( PASMDATA p )
{
    PASMSYMDEF ps;
    PASMSYMDEF psn;

    ps = p->psym;
    while( ps )
    {
        fcpu_utl_free( (void**)&ps->pszc );
        fcpu_utl_free( (void**)&ps->pszs );

        psn = (PASMSYMDEF)fcpu_utl_get_next_element_of_linked_list( ps );

        fcpu_utl_free( (void**)&ps );

        ps = psn;
    }

    p->psym = NULL;
}

//
//
//

UL64 fcpu_asm_resolve_label( PASMDATA p, char* pszl, char* perr )
{
    PASMLBLDEF  pl;
    PASMLBLDEF  plf;
    char        err;
    UL64        addr;

    err     = 0L;
    addr    = 0L;

    pl  = p->plbl;
    plf = NULL;
    while( ( pl ) && ( !plf ) )
    {
        if( fcpu_utl_compare_strings_i( pl->pszl, pszl ) )
            plf = pl;
        else
            pl = (PASMLBLDEF)fcpu_utl_get_next_element_of_linked_list( pl );
    }

    if( plf )
        addr = plf->addr;
    else
        err = 1L;

    if( perr )
    {
        if( err )
            *perr = 1L;
        else
            *perr = 0L;
    }

    return addr;
}

char fcpu_asm_check_is_label( PASMDATA p, char* pszl )
{
    char err;

    fcpu_asm_resolve_label( p, pszl, &err );

    if( err )
        return 0L;

    return 1L;
}

void fcpu_asm_define_label( PASMDATA p, char* pszl, UL64 addr )
{
    PASMLBLDEF pl;

    // link label and line number into label list of p

    pl = (PASMLBLDEF)fcpu_utl_malloc( sizeof( ASMLBLDEF ) );
    if( !pl )
    {
        fcpu_asm_error( p, 1 );
        return;
    }
    fcpu_utl_memset( pl, 0, sizeof( ASMLBLDEF ) );

    pl->pszl = (char*)fcpu_utl_malloc( fcpu_utl_length_of_string_buffer( pszl ) );
    if( !pl->pszl )
    {
        fcpu_utl_free( (void**)&pl );
        fcpu_asm_error( p, 1 );
        return;
    }
    fcpu_utl_memset( pl->pszl, 0, fcpu_utl_length_of_string_buffer( pszl ) );

    //
    fcpu_utl_copy_string( pl->pszl, fcpu_utl_length_of_string_buffer( pszl ), pszl );

    pl->addr = addr;

    //
    fcpu_utl_append_to_linked_list( (void**)&p->plbl, pl );
}

void fcpu_asm_clean_label_list( PASMDATA p )
{
    PASMLBLDEF  pl;
    PASMLBLDEF  pln;

    pl = p->plbl;
    while( pl )
    {
        fcpu_utl_free( (void**)&pl->pszl );

        pln = (PASMLBLDEF)fcpu_utl_get_next_element_of_linked_list( pl );

        fcpu_utl_free( (void**)&pl );

        pl = pln;
    }

    p->plbl = NULL;
}

//
//
//

void fcpu_asm_define_instruction_to_assemble_later( PASMDATA p, UL64 addr, char* pszop, char* pszp1, char* pszp2, char* pszp3 )
{
    PASMINSDEF  pi;
    UL64        lop;
    UL64        lp1;
    UL64        lp2;
    UL64        lp3;

    pi = (PASMINSDEF)fcpu_utl_malloc( sizeof( ASMINSDEF ) );
    if( !pi )
    {
        fcpu_asm_error( p, 1 );
        return;
    }
    fcpu_utl_memset( pi, 0, sizeof( ASMINSDEF ) );

    //
    lop = fcpu_utl_length_of_string_buffer( pszop );
    {
        pi->pszop = (char*)fcpu_utl_malloc( lop );
        if( !pi->pszop )
        {
            fcpu_utl_free( (void**)&pi );
            fcpu_asm_error( p, 1 );
            return;
        }
        fcpu_utl_memset( pi->pszop, 0, lop );
    }

    //
    lp1 = fcpu_utl_length_of_string_buffer( pszp1 );
    {
        pi->pszp1 = (char*)fcpu_utl_malloc( lp1 );
        if( !pi->pszp1 )
        {
            fcpu_utl_free( (void**)&pi->pszop );
            fcpu_utl_free( (void**)&pi );
            fcpu_asm_error( p, 1 );
            return;
        }
        fcpu_utl_memset( pi->pszp1, 0, lp1 );
    }

    //
    lp2 = fcpu_utl_length_of_string_buffer( pszp2 );
    {
        pi->pszp2 = (char*)fcpu_utl_malloc( lp2 );
        if( !pi->pszp2 )
        {
            fcpu_utl_free( (void**)&pi->pszop );
            fcpu_utl_free( (void**)&pi->pszp1 );
            fcpu_utl_free( (void**)&pi );
            fcpu_asm_error( p, 1 );
            return;
        }
        fcpu_utl_memset( pi->pszp2, 0, lp2 );
    }

    //
    lp3 = fcpu_utl_length_of_string_buffer( pszp3 );
    {
        pi->pszp3 = (char*)fcpu_utl_malloc( lp3 );
        if( !pi->pszp3 )
        {
            fcpu_utl_free( (void**)&pi->pszop );
            fcpu_utl_free( (void**)&pi->pszp1 );
            fcpu_utl_free( (void**)&pi->pszp2 );
            fcpu_utl_free( (void**)&pi );
            fcpu_asm_error( p, 1 );
            return;
        }
        fcpu_utl_memset( pi->pszp3, 0, lp3 );
    }

    pi->addr = addr;

    fcpu_utl_copy_string( pi->pszop, lop, pszop );

    fcpu_utl_copy_string( pi->pszp1, lp1, pszp1 );

    fcpu_utl_copy_string( pi->pszp2, lp2, pszp2 );

    fcpu_utl_copy_string( pi->pszp3, lp3, pszp3 );

    fcpu_utl_append_to_linked_list( (void**)&p->pins, pi );
}

void fcpu_asm_clean_instruction_to_assemble_later_list( PASMDATA p )
{
    PASMINSDEF  pi;
    PASMINSDEF  pin;

    pi = p->pins;
    while( pi )
    {
        fcpu_utl_free( (void**)&pi->pszop );
        fcpu_utl_free( (void**)&pi->pszp1 );
        fcpu_utl_free( (void**)&pi->pszp2 );
        fcpu_utl_free( (void**)&pi->pszp3 );

        pin = (PASMINSDEF)fcpu_utl_get_next_element_of_linked_list( pi );

        fcpu_utl_free( (void**)&pi );

        pi = pin;
    }

    p->pins = NULL;
}

//
//
//

UL64 fcpu_asm_evaluate_expression( PASMDATA p, char* psze, char* perr )
{
    UL64    x;
    UL64    getdb;
    char*   pszc;
    char    err;

    err   = 0L;
    getdb = (UL64)-1;

    if( fcpu_utl_check_string_inside_string_at_pos_i( psze, "getdd0_", 0 ) )
        getdb = 0;
    else if( fcpu_utl_check_string_inside_string_at_pos_i( psze, "getdd1_", 0 ) )
        getdb = 1;
    else if( fcpu_utl_check_string_inside_string_at_pos_i( psze, "getdd2_", 0 ) )
        getdb = 2;
    else if( fcpu_utl_check_string_inside_string_at_pos_i( psze, "getdd3_", 0 ) )
        getdb = 3;
    if( getdb != (UL64)-1 )
    {
        psze += 7;
        while( ( fcpu_asm_is_whitespace( *psze ) ) && ( 0 != *psze ) )
            psze++;
    }

    pszc = fcpu_asm_resolve_symbol( p, psze );
    if( pszc )
    {
        x = fcpu_utl_convert_string_to_constant( pszc, &err );
    }
    else
    {
        if( fcpu_asm_check_is_label( p, psze ) )
            x = fcpu_asm_resolve_label( p, psze, &err );
        else if( '"' == *psze )
            x = *( psze + 1 );
        else
            x = fcpu_utl_convert_string_to_constant( psze, &err );
    }

    if( getdb <= 3 )
        x = ( x >> ( getdb * 16 ) ) & 0xFFFF;

    if( perr )
    {
        if( err )
            *perr = 1L;
        else
            *perr = 0L;
    }

    return x;
}

UL64 fcpu_asm_evaluate_register_id( PASMDATA p, char* psze, char* perr )
{
    UL64    r;
    char*   pszc;
    char    err;

    pszc = fcpu_asm_resolve_symbol( p, psze );
    if( !pszc )
        pszc = psze;

    if( pszc )
    {
        if( ( *pszc == 'R' ) || ( *pszc == 'r' ) )
        {
            r = fcpu_utl_convert_string_to_constant( pszc + 1, &err );
        }
        else
            err = 1L;
    }
    else
        err = 1L;

    if( perr )
    {
        if( err )
            *perr = 1L;
        else
            *perr = 0L;
    }

    return r;
}

//
//
//
void fcpu_asm_assemble_2_regs( UL64* po, UL64 r2, UL64 r1 )
{
    *po |= ( ( r2 & 0x3F ) << 20 );
    
    *po |= ( ( r1 & 0x3F ) << 26 );
}

void fcpu_asm_assemble_3_regs( UL64* po, UL64 r3, UL64 r2, UL64 r1 )
{
    *po |= ( ( r3 & 0x3F ) << 14 );

    fcpu_asm_assemble_2_regs( po, r2, r1 );
}

void fcpu_asm_assemble_2r1w_instruction( UL64* po, UL64 oc, UL64 flsize, UL64 flsimd, UL64 flsat, UL64 flcy, UL64 r3, UL64 r2, UL64 r1 )
{
    *po = 0;

    *po |= ( ( oc     & 0xFF ) <<  0 );

    *po |= ( ( flsize & 0x03 ) <<  8 );

    *po |= ( ( flsimd & 0x01 ) << 10 );
    
    *po |= ( ( flsat  & 0x01 ) << 12 );

    *po |= ( ( flcy   & 0x01 ) << 13 );

    fcpu_asm_assemble_3_regs( po, r3, r2, r1 );
}

void fcpu_asm_assemble_2r1w_flt_instruction( UL64* po, UL64 oc, UL64 flsize, UL64 flsimd, UL64 flexc, UL64 r3, UL64 r2, UL64 r1 )
{
    fcpu_asm_assemble_2r1w_instruction( po, oc, flsize & 1, flsimd, 0L, 0L, r3, r2, r1 );

    *po |= ( ( flexc & 0x01 ) << 11 );
}

void fcpu_asm_assemble_1r1w_imm_instruction( UL64* po, UL64 oc, UL64 flsize, UL64 flsimd, UL64 imm, UL64 r2, UL64 r1 )
{
    *po = 0;

    *po |= ( ( oc     & 0xFF ) <<  0 );

    *po |= ( ( flsize & 0x03 ) <<  8 );

    *po |= ( ( flsimd & 0x01 ) << 10 );
    
    *po |= ( ( imm    & 0xFF ) << 12 );
    
    fcpu_asm_assemble_2_regs( po, r2, r1 );
}

void fcpu_asm_evaluate_operands( PASMDATA p, char* pszp1, char* pszp2, char* pszp3, UL64* pr3, UL64* pr2, UL64* pr1, UL64* pimm )
{
    UL64 r3;
    UL64 r2;
    UL64 r1;
    UL64 imm;
    char err;

    r3  = fcpu_asm_evaluate_register_id( p, pszp3, &err );
    imm = 0;
    if( err )
    {
        imm = fcpu_asm_evaluate_expression( p, pszp3, &err );
        if( err )
        {
            r3  = 0;
            imm = 0;
        }
    }

    r2 = fcpu_asm_evaluate_register_id( p, pszp2, &err );
    if( err )
        r2 = 0;

    r1 = fcpu_asm_evaluate_register_id( p, pszp1, &err );
    if( err )
        r1 = 0;

    if( pr3 )
        *pr3 = r3;

    if( pr2 )
        *pr2 = r2;

    if( pr1 )
        *pr1 = r1;

    if( pimm )
        *pimm = imm;
}

UL64 fcpu_asm_get_opcode( PASMDATA p, char* pszop, char* pszp1, char* pszp2, char* pszp3, char* perr )
{
    UL64    instrcode;
    UL64    size;
    UL64    rnd;
    UL64    r1;
    UL64    r2;
    UL64    r3;
    UL64    imm;
    char    err;
    char    smf;
    char*   pszop0;

    instrcode   = 0;
    size        = 3;
    rnd         = 0;
    r1          = 0;
    r2          = 0;
    r3          = 0;
    imm         = 0;
    err         = 1;
    smf         = 0;

    pszop0 = (char*)fcpu_utl_malloc( fcpu_utl_length_of_string_buffer( pszop ) );
    if( !pszop0 )
    {
        if( perr )
            *perr = 1;
        else
            *perr = 0;

        return 0;
    }
    fcpu_utl_copy_string( pszop0, fcpu_utl_length_of_string_buffer( pszop ), pszop );

    if( fcpu_utl_check_string_inside_string_i( pszop0, ".b"  ) ||
        fcpu_utl_check_string_inside_string_i( pszop0, ".fs" ) ||
        fcpu_utl_check_string_inside_string_i( pszop0, ".s0" ) )
    {
        size = 0;
        smf++;
    }
    else if( fcpu_utl_check_string_inside_string_i( pszop0, ".d"  ) ||
             fcpu_utl_check_string_inside_string_i( pszop0, ".fd" ) ||
             fcpu_utl_check_string_inside_string_i( pszop0, ".s1" ) )
    {
        size = 1;
        smf++;
    }
    else if( fcpu_utl_check_string_inside_string_i( pszop0, ".q"  ) ||
             fcpu_utl_check_string_inside_string_i( pszop0, ".s2" ) )
    {
        size = 2;
        smf++;
    }
    else if( fcpu_utl_check_string_inside_string_i( pszop0, ".o"  ) ||
             fcpu_utl_check_string_inside_string_i( pszop0, ".s3" ) )
    {
        size = 3;
        smf++;
    }

    if( fcpu_utl_check_string_inside_string_i( pszop0, ".r0" ) )
    {
        rnd = 0;
        smf++;
    }
    else if( fcpu_utl_check_string_inside_string_i( pszop0, ".r1" ) )
    {
        rnd = 1;
        smf++;
    }
    else if( fcpu_utl_check_string_inside_string_i( pszop0, ".r2" ) )
    {
        rnd = 2;
        smf++;
    }
    else if( fcpu_utl_check_string_inside_string_i( pszop0, ".r3" ) )
    {
        rnd = 3;
        smf++;
    }

    if( smf )
    {
        char* pszop0e;

        pszop0e = pszop0 + fcpu_utl_length_of_string( pszop0 );
        while( smf > 0 )
        {
            if( *pszop0e == '.' )
                smf--;

            *pszop0e = 0;
            pszop0e--;
        }
    }

    // UNKNOWN - assembles an unknown instruction op code,
    //           allowing EXCEPTION_UNKNOWNINSTRUCTION to
    //           be checked.
    if( fcpu_utl_compare_strings_i( pszop0, "unknown" ) )
    {
        instrcode = FCPU_OP_UNKNOWN;
        err = 0;
    }

    // HLT
    if( fcpu_utl_compare_strings_i( pszop0, "halt" ) )
    {
        instrcode = FCPU_OP_HALT;
        err = 0;
    }

    // JMPA

    if( fcpu_utl_compare_strings_i( pszop0, "jmpa" ) )
    {
        // alias for move.nul.n r0,rx,ry
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_JMPA;
        instrcode |= ( 0x0 << 11 );
        instrcode |= BIT( 10 );
        fcpu_asm_assemble_3_regs( &instrcode, 0, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "jmpa.ifnul" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_JMPA;
        instrcode |= ( 0x0 << 11 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "jmpa.iflsb" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_JMPA;
        instrcode |= ( 0x3 << 11 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "jmpa.ifmsb" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_JMPA;
        instrcode |= ( 0x2 << 11 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "jmpa.ifnnul" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_JMPA;
        instrcode |= ( 0x0 << 11 );
        instrcode |= BIT( 10 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "jmpa.ifnlsb" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_JMPA;
        instrcode |= ( 0x3 << 11 );
        instrcode |= BIT( 10 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "jmpa.ifnmsb" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_JMPA;
        instrcode |= ( 0x2 << 11 );
        instrcode |= BIT( 10 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }

    // LOADADDR
    
    if( fcpu_utl_compare_strings_i( pszop0, "loadaddr" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADADDR;
        fcpu_asm_assemble_2_regs( &instrcode, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadaddrd" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADADDR;
        instrcode |= BIT( 8 );
        fcpu_asm_assemble_2_regs( &instrcode, r3, r2 );
        err = 0;
    }

    // LOOPENTRY
    
    if( fcpu_utl_compare_strings_i( pszop0, "loopentry" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADADDR;
        fcpu_asm_assemble_3_regs( &instrcode, 0, 0, r3 );
        err = 0;
    }

    // LOADADDRI
    if( fcpu_utl_compare_strings_i( pszop0, "loadaddri" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADADDRI;
        instrcode |= ( imm & 0xFFFF ) << 10;
        fcpu_asm_assemble_2_regs( &instrcode, 0, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadaddrid" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADADDRI;
        instrcode |= ( imm & 0xFFFF ) << 10;
        instrcode |= BIT(8);
        fcpu_asm_assemble_2_regs( &instrcode, 0, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadaddris" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADADDRI;
        instrcode |= ( imm & 0xFFFF ) << 10;
        instrcode |= BIT(9);
        fcpu_asm_assemble_2_regs( &instrcode, 0, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadaddrids" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADADDRI;
        instrcode |= ( imm & 0xFFFF ) << 10;
        instrcode |= BIT(8);
        instrcode |= BIT(9);
        fcpu_asm_assemble_2_regs( &instrcode, 0, r2 );
        err = 0;
    }

    // LOOP
    
    if( fcpu_utl_compare_strings_i( pszop0, "loop" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOOP;
        fcpu_asm_assemble_3_regs( &instrcode, 0, 0, r3 );
        err = 0;
    }

    // MOVE

    if( fcpu_utl_compare_strings_i( pszop0, "move" ) )
    {
        // alias for move.nul.n r0,rx,ry
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_MOVE;
        instrcode |= ( 0x0 << 11 );
        instrcode |= BIT( 10 );
        fcpu_asm_assemble_3_regs( &instrcode, 0, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "move.ifnul" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_MOVE;
        instrcode |= ( 0x0 << 11 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "move.iflsb" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_MOVE;
        instrcode |= ( 0x3 << 11 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "move.iflsb" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_MOVE;
        instrcode |= ( 0x2 << 11 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "move.ifnnul" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_MOVE;
        instrcode |= ( 0x0 << 11 );
        instrcode |= BIT( 10 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "move.ifnlsb" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_MOVE;
        instrcode |= ( 0x3 << 11 );
        instrcode |= BIT( 10 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "move.ifnmsb" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_MOVE;
        instrcode |= ( 0x2 << 11 );
        instrcode |= BIT( 10 );
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }

    // LOADCONS
    
    if( fcpu_utl_compare_strings_i( pszop0, "loadcons.0" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADCONS;
        instrcode |= (   0 << 8  );
        instrcode |= ( imm << 10 );
        instrcode |= (  r2 << 26 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadcons.1" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADCONS;
        instrcode |= (   1 << 8  );
        instrcode |= ( imm << 10 );
        instrcode |= (  r2 << 26 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadcons.2" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADCONS;
        instrcode |= (   2 << 8  );
        instrcode |= ( imm << 10 );
        instrcode |= (  r2 << 26 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadcons.3" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADCONS;
        instrcode |= (   3 << 8  );
        instrcode |= ( imm << 10 );
        instrcode |= (  r2 << 26 );
        err = 0;
    }

    // LOADCONSX
    
    if( fcpu_utl_compare_strings_i( pszop0, "loadconsx.0" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADCONSX;
        instrcode |= (   0 << 8  );
        instrcode |= ( imm << 10 );
        instrcode |= (  r2 << 26 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadconsx.1" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADCONSX;
        instrcode |= (   1 << 8  );
        instrcode |= ( imm << 10 );
        instrcode |= (  r2 << 26 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadconsx.2" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADCONSX;
        instrcode |= (   2 << 8  );
        instrcode |= ( imm << 10 );
        instrcode |= (  r2 << 26 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadconsx.3" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADCONSX;
        instrcode |= (   3 << 8  );
        instrcode |= ( imm << 10 );
        instrcode |= (  r2 << 26 );
        err = 0;
    }

    // GET, GETI, PUT, PUTI

    if( fcpu_utl_compare_strings_i( pszop0, "get" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_GET;
        fcpu_asm_assemble_2_regs( &instrcode, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "geti" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_GETI;
        instrcode |= ( imm & 0xFFFF ) << 10L;
        instrcode |= (  r1 &   0x3F ) << 26L;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "put" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_PUT;
        fcpu_asm_assemble_2_regs( &instrcode, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "puti" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_PUTI;
        instrcode |= ( imm & 0xFFFF ) << 10L;
        instrcode |= (  r1 &   0x3F ) << 26L;
        err = 0;
    }

    // LOADM, STOREM
    if( fcpu_utl_compare_strings_i( pszop0, "loadm" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_LOADM;
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "storem" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_STOREM;
        fcpu_asm_assemble_3_regs( &instrcode, r3, r2, r1 );
        err = 0;
    }

    // LOAD
    if( fcpu_utl_compare_strings_i( pszop0, "load" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOAD, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loade" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOAD, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }

    // LOADI
    if( fcpu_utl_compare_strings_i( pszop0, "loadi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_LOADI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadie" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_LOADI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // STORE
    if( fcpu_utl_compare_strings_i( pszop0, "store" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_STORE, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "storee" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_STORE, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }

    // STOREI
    if( fcpu_utl_compare_strings_i( pszop0, "storei" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_STOREI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "storeie" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_STOREI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // LOADF
    if( fcpu_utl_compare_strings_i( pszop0, "loadf" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOADF, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadfe" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOADF, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }

    // LOADIF
    if( fcpu_utl_compare_strings_i( pszop0, "loadif" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_LOADIF, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "loadife" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_LOADIF, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // STOREF
    if( fcpu_utl_compare_strings_i( pszop0, "storef" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_STOREF, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "storefe" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_STOREF, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }

    // STOREIF
    if( fcpu_utl_compare_strings_i( pszop0, "storeif" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_STOREIF, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "storeife" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_STOREIF, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // CACHEMM
    if( fcpu_utl_compare_strings_i( pszop0, "cachemm" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_CACHEMM;
        fcpu_asm_assemble_2_regs( &instrcode, r2, r1 );
        err = 0;
    }

    //
    // 6.1.1 Core Arithmetic Instructions
    //

    // ADD

    if( fcpu_utl_compare_strings_i( pszop0, "add" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ADD, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "adds" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ADD, size, 0L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "addc" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ADD, size, 0L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sadd" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ADD, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sadds" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ADD, size, 1L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "saddc" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ADD, size, 1L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }

    // SUB BYTE

    if( fcpu_utl_compare_strings_i( pszop0, "sub" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SUB, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "subs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SUB, size, 0L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "subc" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SUB, size, 0L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "ssub" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SUB, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "ssubs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SUB, size, 1L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "ssubc" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SUB, size, 1L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }

    // MUL

    if( fcpu_utl_compare_strings_i( pszop0, "mul" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MUL, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "muls" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MUL, size, 0L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "mulh" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MUL, size, 0L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "mulsh" ) ||
        fcpu_utl_compare_strings_i( pszop0, "mulhs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MUL, size, 0L, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smul" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MUL, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smuls" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MUL, size, 1L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smulh" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MUL, size, 1L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smulsh" ) ||
        fcpu_utl_compare_strings_i( pszop0, "smulhs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MUL, size, 1L, 1L, 1L, r3, r2, r1 );
        err = 0;
    }

    // DIV

    if( fcpu_utl_compare_strings_i( pszop0, "div" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DIV, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "divs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DIV, size, 0L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "divm" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DIV, size, 0L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "divsm" ) ||
        fcpu_utl_compare_strings_i( pszop0, "divms" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DIV, size, 0L, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sdiv" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DIV, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sdivs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DIV, size, 1L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sdivm" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DIV, size, 1L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sdivsm" ) ||
        fcpu_utl_compare_strings_i( pszop0, "sdivms" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DIV, size, 1L, 1L, 1L, r3, r2, r1 );
        err = 0;
    }

    //
    // 6.1.2 Optional Arithmetic Instructions
    //

    // ADDI

    if( fcpu_utl_compare_strings_i( pszop0, "addi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_ADDI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "saddi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_ADDI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // SUBI

    if( fcpu_utl_compare_strings_i( pszop0, "subi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SUBI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "ssubi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SUBI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // MULI

    if( fcpu_utl_compare_strings_i( pszop0, "muli" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_MULI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smuli" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_MULI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // DIVI

    if( fcpu_utl_compare_strings_i( pszop0, "divi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_DIVI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sdivi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_DIVI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // MOD

    if( fcpu_utl_compare_strings_i( pszop0, "mod" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MOD, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "mods" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MOD, size, 0L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smod" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MOD, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smods" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MOD, size, 1L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }

    // MAC
    if( fcpu_utl_compare_strings_i( pszop0, "mac" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAC, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "macs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAC, size, 0L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "mach" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAC, size, 0L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "machs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAC, size, 0L, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smac" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAC, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smacs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAC, size, 1L, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smach" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAC, size, 1L, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smachs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAC, size, 1L, 1L, 1L, r3, r2, r1 );
        err = 0;
    }

    // MODI

    if( fcpu_utl_compare_strings_i( pszop0, "modi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_MODI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smodi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_MODI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // ADDSUB

    if( fcpu_utl_compare_strings_i( pszop0, "addsub" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ADDSUB, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "saddsub" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ADDSUB, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // POPCOUNT

    if( fcpu_utl_compare_strings_i( pszop0, "popcount" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_POPCOUNT, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "spopcount" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_POPCOUNT, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // INC

    if( fcpu_utl_compare_strings_i( pszop0, "inc" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_INC, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sinc" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_INC, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }

    // DEC

    if( fcpu_utl_compare_strings_i( pszop0, "dec" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DEC, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sdec" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_DEC, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }

    // NEG

    if( fcpu_utl_compare_strings_i( pszop0, "neg" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_NEG, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sneg" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_NEG, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }

    // SCAN
    if( ( fcpu_utl_compare_strings_i( pszop0, "scan" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "lsb1" ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SCAN, size, 0L, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( 0 << 12 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "scann" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "lsb0" ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SCAN, size, 0L, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( 1 << 12 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "scanr" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "msb1" ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SCAN, size, 0L, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( 2 << 12 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "scannr" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "msb0" ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SCAN, size, 0L, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( 3 << 12 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "sscan" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "slsb1" ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SCAN, size, 1L, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( 0 << 12 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "sscann" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "slsb0" ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SCAN, size, 1L, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( 1 << 12 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "sscanr" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "smsb1" ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SCAN, size, 1L, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( 2 << 12 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "sscannr" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "smsb0" ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SCAN, size, 1L, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( 3 << 12 );
        err = 0;
    }

    // CMPL
    if( fcpu_utl_compare_strings_i( pszop0, "cmpl" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_CMPL, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "scmpl" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_CMPL, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // CMPLE
    if( fcpu_utl_compare_strings_i( pszop0, "cmple" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_CMPLE, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "scmple" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_CMPLE, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // CMPLI

    if( fcpu_utl_compare_strings_i( pszop0, "cmpli" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_CMPLI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "scmpli" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_CMPLI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // CMPLEI

    if( fcpu_utl_compare_strings_i( pszop0, "cmplei" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_CMPLEI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "scmplei" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_CMPLEI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // ABS

    if( fcpu_utl_compare_strings_i( pszop0, "abs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ABS, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sabs" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ABS, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }

    // MAX

    if( fcpu_utl_compare_strings_i( pszop0, "max" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAX, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smax" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MAX, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // MIN

    if( fcpu_utl_compare_strings_i( pszop0, "min" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MIN, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smin" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MIN, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // MAXI

    if( fcpu_utl_compare_strings_i( pszop0, "maxi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_MAXI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smaxi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_MAXI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // MINI

    if( fcpu_utl_compare_strings_i( pszop0, "mini" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_MINI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "smini" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_MINI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // SORT

    if( fcpu_utl_compare_strings_i( pszop0, "sort" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SORT, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "ssort" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SORT, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // SHIFTL

    if( fcpu_utl_compare_strings_i( pszop0, "shiftl" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SHIFTL, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sshiftl" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SHIFTL, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // SHIFTR

    if( fcpu_utl_compare_strings_i( pszop0, "shiftr" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SHIFTR, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sshiftr" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SHIFTR, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // SHIFTRA

    if( fcpu_utl_compare_strings_i( pszop0, "shiftra" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SHIFTRA, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sshiftra" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_SHIFTRA, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // ROTL

    if( fcpu_utl_compare_strings_i( pszop0, "rotl" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ROTL, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "srotl" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ROTL, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // ROTR

    if( fcpu_utl_compare_strings_i( pszop0, "rotr" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ROTR, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "srotr" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_ROTR, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // SHIFTLI

    if( fcpu_utl_compare_strings_i( pszop0, "shiftli" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SHIFTLI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sshiftli" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SHIFTLI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // SHIFTRI

    if( fcpu_utl_compare_strings_i( pszop0, "shiftri" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SHIFTRI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sshiftri" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SHIFTRI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // SHIFTRAI

    if( fcpu_utl_compare_strings_i( pszop0, "shiftrai" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SHIFTRAI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sshiftrai" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SHIFTRAI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // ROTLI

    if( fcpu_utl_compare_strings_i( pszop0, "rotli" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_ROTLI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "srotli" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_ROTLI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // ROTRI

    if( fcpu_utl_compare_strings_i( pszop0, "rotri" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_ROTRI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "srotri" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_ROTRI, size, 1L, imm, r2, r1 );
        err = 0;
    }

    // BSET

    if( fcpu_utl_compare_strings_i( pszop0, "bset" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITOP, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x00 >> 12 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sbset" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITOP, size, 1L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x00 >> 12 );
        err = 0;
    }

    // BCLR

    if( fcpu_utl_compare_strings_i( pszop0, "bclr" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITOP, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x01 >> 12 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sbclr" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITOP, size, 1L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x01 >> 12 );
        err = 0;
    }

    // BCHG

    if( fcpu_utl_compare_strings_i( pszop0, "bchg" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITOP, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x02 >> 12 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sbchg" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITOP, size, 1L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x02 >> 12 );
        err = 0;
    }

    // BTST

    if( fcpu_utl_compare_strings_i( pszop0, "btst" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITOP, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x03 >> 12 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sbtst" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITOP, size, 1L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x03 >> 12 );
        err = 0;
    }

    // BSETI

    if( fcpu_utl_compare_strings_i( pszop0, "bseti" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITOPI, size, 0L, imm, r2, r1 );
        instrcode |= ( 0x00 >> 12 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sbseti" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITOPI, size, 1L, imm, r2, r1 );
        instrcode |= ( 0x00 >> 12 );
        err = 0;
    }

    // BCLRI

    if( fcpu_utl_compare_strings_i( pszop0, "bclri" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITOPI, size, 0L, imm, r2, r1 );
        instrcode |= ( 0x01 >> 12 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sbclri" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITOPI, size, 1L, imm, r2, r1 );
        instrcode |= ( 0x01 >> 12 );
        err = 0;
    }

    // BCHGI

    if( fcpu_utl_compare_strings_i( pszop0, "bchgi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITOPI, size, 0L, imm, r2, r1 );
        instrcode |= ( 0x02 >> 12 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sbchgi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITOPI, size, 1L, imm, r2, r1 );
        instrcode |= ( 0x02 >> 12 );
        err = 0;
    }

    // BTSTI

    if( fcpu_utl_compare_strings_i( pszop0, "btsti" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITOPI, size, 0L, imm, r2, r1 );
        instrcode |= ( 0x03 >> 12 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sbtsti" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITOPI, size, 1L, imm, r2, r1 );
        instrcode |= ( 0x03 >> 12 );
        err = 0;
    }

    // BETREV
    if( fcpu_utl_compare_strings_i( pszop0, "bitrev" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITREV, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "bitrevo" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BITREV, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }

    // BETREVI
    if( fcpu_utl_compare_strings_i( pszop0, "bitrevi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITREVI, size, 0L, imm, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "bitrevio" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_BITREVI, size, 1L, imm, r2, r1 );
        err = 0;
    }
    
    // BYTEREV
    if( fcpu_utl_compare_strings_i( pszop0, "byterev" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BYTEREV, size, 0L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }    
    if( fcpu_utl_compare_strings_i( pszop0, "sbyterev" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_BYTEREV, size, 1L, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }    

    // MIX
    if( fcpu_utl_compare_strings_i( pszop0, "mixl" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MIX, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }    
    if( fcpu_utl_compare_strings_i( pszop0, "mixh" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_MIX, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }    

    // EXPAND
    if( fcpu_utl_compare_strings_i( pszop0, "expandl" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_EXPAND, size, 0L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }    
    if( fcpu_utl_compare_strings_i( pszop0, "expandh" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_EXPAND, size, 1L, 0L, 0L, r3, r2, r1 );
        err = 0;
    }    

    // SDUP

    if( fcpu_utl_compare_strings_i( pszop0, "sdup" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_SDUP, size, 0L, 0L, r3, r2 );
        err = 0;
    }

    // LOGIC

    if( fcpu_utl_compare_strings_i( pszop0, "logic.0000" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x00 >> 10 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "logic.0001" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "and"        ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x01 >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.0010" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x02 >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.0011" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x03 >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.0100" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x04 >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.0101" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x05 >> 10 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "logic.0110" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "xor"        ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x06 >> 10 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "logic.0111" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "or"         ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x07 >> 10 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "logic.1000" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "nor"        ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x08 >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.1001" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x09 >> 10 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "logic.1010" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "not"        ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x0A >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.1011" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x0B >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.1100" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x0C >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.1101" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x0D >> 10 );
        err = 0;
    }
    if( ( fcpu_utl_compare_strings_i( pszop0, "logic.1110" ) ) ||
        ( fcpu_utl_compare_strings_i( pszop0, "nand"       ) ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x0E >> 10 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "logic.1111" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_instruction( &instrcode, FCPU_OP_LOGIC, size, 0L, 0L, 0L, r3, r2, r1 );
        instrcode |= ( 0x0F >> 10 );
        err = 0;
    }

    // ORI
    if( fcpu_utl_compare_strings_i( pszop0, "ori" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_LOGICI, size, 0L, imm, r2, r1 );
        instrcode |= ( 0x00 >> 10 ); // different position as in bitopi
        err = 0;
    }

    // ANDI
    if( fcpu_utl_compare_strings_i( pszop0, "andi" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_LOGICI, size, 0L, imm, r2, r1 );
        instrcode |= ( 0x01 >> 10 ); // different position as in bitopi
        err = 0;
    }

    // XORI
    if( fcpu_utl_compare_strings_i( pszop0, "xori" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_LOGICI, size, 0L, imm, r2, r1 );
        instrcode |= ( 0x02 >> 10 ); // different position as in bitopi
        err = 0;
    }

    // ANDNI
    if( fcpu_utl_compare_strings_i( pszop0, "andni" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_1r1w_imm_instruction( &instrcode, FCPU_OP_LOGICI, size, 0L, imm, r2, r1 );
        instrcode |= ( 0x03 >> 10 ); // different position as in bitopi
        err = 0;
    }

    // FADD
    if( fcpu_utl_compare_strings_i( pszop0, "fadd" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FADD, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfadd" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FADD, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "faddx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FADD, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfaddx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FADD, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FSUB
    if( fcpu_utl_compare_strings_i( pszop0, "fsub" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSUB, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfsub" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSUB, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "fsubx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSUB, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfsubx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSUB, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FMUL
    if( fcpu_utl_compare_strings_i( pszop0, "fmul" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FMUL, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfmul" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FMUL, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "fmulx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FMUL, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfmulx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FMUL, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // F2INT
    if( fcpu_utl_compare_strings_i( pszop0, "f2int" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_F2INT, size, 0L, 0L, 0L, r3, r2 );
        instrcode |= ( rnd & 3 ) << 12;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sf2int" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_F2INT, size, 1L, 0L, 0L, r3, r2 );
        instrcode |= ( rnd & 3 ) << 12;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "f2intx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_F2INT, size, 0L, 1L, 0L, r3, r2 );
        instrcode |= ( rnd & 3 ) << 12;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sf2intx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_F2INT, size, 1L, 1L, 0L, r3, r2 );
        instrcode |= ( rnd & 3 ) << 12;
        err = 0;
    }
    
    // INT2F
    if( fcpu_utl_compare_strings_i( pszop0, "int2f" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_INT2F, size, 0L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sint2f" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_INT2F, size, 1L, 0L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "int2fx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_INT2F, size, 0L, 1L, 0L, r3, r2 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sint2fx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_INT2F, size, 1L, 1L, 0L, r3, r2 );
        err = 0;
    }
    
    // FIAPRX
    if( fcpu_utl_compare_strings_i( pszop0, "fiaprx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FIAPRX, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfiaprx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FIAPRX, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "fiaprxx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FIAPRX, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfiaprxx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FIAPRX, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FSQRTIAPRX
    if( fcpu_utl_compare_strings_i( pszop0, "fsqrtiaprx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSQRTIAPRX, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfsqrtiaprx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSQRTIAPRX, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "fsqrtiaprxx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSQRTIAPRX, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfsqrtiaprxx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSQRTIAPRX, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FDIV
    if( fcpu_utl_compare_strings_i( pszop0, "fdiv" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FDIV, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfdiv" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FDIV, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "fdivx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FDIV, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfdivx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FDIV, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FSQRT
    if( fcpu_utl_compare_strings_i( pszop0, "fsqrt" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSQRT, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfsqrt" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSQRT, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "fsqrtx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSQRT, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfsqrtx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FSQRT, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FLOG
    if( fcpu_utl_compare_strings_i( pszop0, "flog" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FLOG, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sflog" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FLOG, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "flogx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FLOG, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sflogx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FLOG, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FEXP
    if( fcpu_utl_compare_strings_i( pszop0, "fexp" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FEXP, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfexp" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FEXP, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "fexpx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FEXP, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfexpx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FEXP, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FMAC
    if( fcpu_utl_compare_strings_i( pszop0, "fmac" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FMAC, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfmac" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FMAC, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "fmacx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FMAC, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfmacx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FMAC, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }
    
    // FADDSUB
    if( fcpu_utl_compare_strings_i( pszop0, "faddsub" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FADDSUB, size, 0L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfaddsub" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FADDSUB, size, 1L, 0L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "faddsubx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FADDSUB, size, 0L, 1L, r3, r2, r1 );
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "sfaddsubx" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        fcpu_asm_assemble_2r1w_flt_instruction( &instrcode, FCPU_OP_FADDSUB, size, 1L, 1L, r3, r2, r1 );
        err = 0;
    }

    if( fcpu_utl_compare_strings_i( pszop0, "syscall" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_SYSCALL | ( ( imm & 0xFFFF ) << 10 ) | ( ( r1 & 0x3F ) << 26 ) | 0x0000;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "trap" ) )
    {
        fcpu_asm_evaluate_operands( p, pszp1, pszp2, pszp3, &r3, &r2, &r1, &imm );
        instrcode = FCPU_OP_SYSCALL | ( ( imm & 0xFFFF ) << 10 ) | ( ( r1 & 0x3F ) << 26 ) | 0x0100;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "rfe" ) )
    {
        instrcode = FCPU_OP_RFE;
        err = 0;
    }

    if( fcpu_utl_compare_strings_i( pszop0, "cachemm" ) )
    {
        instrcode = FCPU_OP_CACHEMM;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "srb_save" ) )
    {
        instrcode = FCPU_OP_SRB_SAVE;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "srb_restore" ) )
    {
        instrcode = FCPU_OP_SRB_RESTORE;
        err = 0;
    }
    if( fcpu_utl_compare_strings_i( pszop0, "serialize" ) )
    {
        instrcode = FCPU_OP_SERIALIZE;
        err = 0;
    }

    if( perr )
    {
        if( err )
            *perr = 1L;
        else
            *perr = 0L;
    }

    fcpu_utl_free( (void**)&pszop0 );

    return instrcode;
}

char fcpu_asm_is_opcode( PASMDATA p, char* pszop )
{
    char err;

    fcpu_asm_get_opcode( p, pszop, "", "", "", &err );

    if( err )
        return 0L;

    return 1L;
}

//
//
//

void fcpu_asm_parse_end( PASMDATA p )
{
    fcpu_asm_clean_symbol_list( p );
    fcpu_asm_clean_label_list( p );
    fcpu_asm_clean_instruction_to_assemble_later_list( p );
}

void fcpu_asm_parse_begin( PASMDATA p )
{  
    p->mp   = NULL;
    p->ms   = 0;
    
    p->err  = 0;

    p->addr = 0;

    p->psym = NULL;
    p->plbl = NULL;
    p->pins = NULL;
}

void fcpu_asm_parse_file( char* filename, PASMDATA p )
{
    FILE*   f;
    char    ch;
    char    label[ FCPU_ASM_IDBUFSIZE ];
    char    opcode[ FCPU_ASM_IDBUFSIZE ];
    char    parameter1[ FCPU_ASM_IDBUFSIZE ];
    char    parameter2[ FCPU_ASM_IDBUFSIZE ];
    char    parameter3[ FCPU_ASM_IDBUFSIZE ];

    f = fopen( filename, "rb" );
    if( f )
    {
        ch = fcpu_asm_get_char( f );
        while( ch != 0 )
        {
            fcpu_asm_get_line( f,
                               &ch,
                               &label[ 0 ],
                               sizeof( label ),
                               &opcode[ 0 ],
                               sizeof( opcode ),
                               &parameter1[ 0 ],
                               sizeof( parameter1 ),
                               &parameter2[ 0 ],
                               sizeof( parameter2 ),
                               &parameter3[ 0 ],
                               sizeof( parameter3 ) );

            //
            if( !fcpu_utl_is_empty_string( &label[ 0 ] ) )
            {
                fcpu_asm_define_label( p, &label[ 0 ], p->addr );
            }

            //
            if( fcpu_utl_compare_strings_i( &parameter3[ 0 ], "equ" ) )
            {                
                fcpu_asm_define_symbol( p, &opcode[ 0 ], &parameter2[ 0 ] );
            }
            else if( fcpu_utl_compare_strings_i( &opcode[ 0 ], "include" ) )
            {
                fcpu_asm_parse_file( &parameter1[ 0 ], p );
            }
            else if( fcpu_utl_compare_strings_i( &opcode[ 0 ], "org" ) )
            {
                UL64 org;
                char err;

                org = fcpu_asm_evaluate_expression( p, &parameter3[ 0 ], &err );
                if( !err )
                {
                    p->addr = org;
                }
                else
                    fcpu_asm_error( p, 1 );
            }
            else if( fcpu_utl_compare_strings_i( &opcode[ 0 ], "align" ) )
            {
                UL64 align;
                char err;

                align = fcpu_asm_evaluate_expression( p, &parameter3[ 0 ], &err );
                if( !err )
                {
                    if( align > 0 )
                        p->addr = ( ( p->addr + align - 1 ) / align ) * align;
                    else
                        fcpu_asm_error( p, 1 );
                }
                else
                    fcpu_asm_error( p, 1 );
            }
            else if( fcpu_utl_compare_strings_i( &opcode[ 0 ], "db" ) )
            {                
                if( parameter3[ 0 ] == '"' )
                {
                    UL64 i;

                    i = 1;
                    while( ( parameter3[ i ] != 0 ) && ( parameter3[ i ] != '"' ) )
                    {
                        fcpu_mem_put_n_bytes_le( p->addr, 1, (char)parameter3[ i ] );
                        p->addr++;
                        i++;
                    }
                }
                else
                {
                    UL64 b;
                    char err;

                    b = fcpu_asm_evaluate_expression( p, &parameter3[ 0 ], &err );
                    if( !err )
                    {
                        fcpu_mem_put_n_bytes_le( p->addr, 1, b );
                        p->addr++;
                    }
                    else
                        fcpu_asm_error( p, 1 );
                }
            }
            else if( ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "dd"   ) ) ||
                     ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "ddle" ) ) ||
                     ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "ddbe" ) ) )
            {
                UL64 d;
                char err;

                d = fcpu_asm_evaluate_expression( p, &parameter3[ 0 ], &err );
                if( !err )
                {
                    if( fcpu_utl_check_string_inside_string_i( &opcode[ 0 ], "be" ) )
                        fcpu_mem_put_n_bytes_be( p->addr, 2, d );
                    else
                        fcpu_mem_put_n_bytes_le( p->addr, 2, d );

                    p->addr += 2;
                }
                else
                    fcpu_asm_error( p, 1 );
            }
            else if( ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "dq"   ) ) ||
                     ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "dqle" ) ) ||
                     ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "dqbe" ) ) )
            {
                UL64 d;
                char err;

                d = fcpu_asm_evaluate_expression( p, &parameter3[ 0 ], &err );
                if( !err )
                {
                    if( fcpu_utl_check_string_inside_string_i( &opcode[ 0 ], "be" ) )
                        fcpu_mem_put_n_bytes_be( p->addr, 4, d );
                    else
                        fcpu_mem_put_n_bytes_le( p->addr, 4, d );

                    p->addr += 4;
                }
                else
                    fcpu_asm_error( p, 1 );
            }
            else if( ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "do"   ) ) ||
                     ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "dole" ) ) ||
                     ( fcpu_utl_compare_strings_i( &opcode[ 0 ], "dobe" ) ) )
            {
                UL64 d;
                char err;

                d = fcpu_asm_evaluate_expression( p, &parameter3[ 0 ], &err );
                if( !err )
                {
                    if( fcpu_utl_check_string_inside_string_i( &opcode[ 0 ], "be" ) )
                        fcpu_mem_put_n_bytes_be( p->addr, 8, d );
                    else
                        fcpu_mem_put_n_bytes_le( p->addr, 8, d );

                    p->addr += 8;
                } 
                else
                    fcpu_asm_error( p, 1 );
            }
            else if( fcpu_asm_is_opcode( p, &opcode[ 0 ] ) )
            {               
                if( ( p->addr & 3 ) == 0 )
                {
                    fcpu_asm_define_instruction_to_assemble_later( p,
                                                                   p->addr,
                                                                   &opcode[ 0 ],
                                                                   &parameter1[ 0 ],
                                                                   &parameter2[ 0 ],
                                                                   &parameter3[ 0 ] );
                    p->addr += 4;
                }
                else
                    fcpu_asm_error( p, 1 );
            }
            else if( fcpu_utl_compare_strings_i( &opcode[ 0 ],     "" ) &&
                     fcpu_utl_compare_strings_i( &parameter1[ 0 ], "" ) &&
                     fcpu_utl_compare_strings_i( &parameter2[ 0 ], "" ) &&
                     fcpu_utl_compare_strings_i( &parameter3[ 0 ], "" ) )
            {
            }
            else
            {
                fcpu_asm_error( p, 1 );
                printf( "  " );
                printf( &label[ 0 ] );
                printf( " " );
                printf( &opcode[ 0 ] );
                printf( " " );
                printf( &parameter1[ 0 ] );
                printf( " " );
                printf( &parameter2[ 0 ] );
                printf( " " );
                printf( &parameter3[ 0 ] );
                printf( "\n\n" );
            }
        }

        fclose( f );
    }
    else
        fcpu_asm_error( p, 1 );
}

void fcpu_asm_assemble_instructions( PASMDATA p )
{
    PASMINSDEF  pi;
    UL64        opcode;
    char        err;

    pi = p->pins;
    while( pi )
    {       
        opcode = fcpu_asm_get_opcode( p, pi->pszop, pi->pszp1, pi->pszp2, pi->pszp3, &err );
        if( err )
        {
            fcpu_asm_error( p, 1 );
            pi = 0;
        }
        else
        {
            fcpu_mem_put_n_bytes_le( pi->addr, 4, opcode );

            pi = (PASMINSDEF)fcpu_utl_get_next_element_of_linked_list( (void*)pi );
        }
    }
}

char fcpu_asm_assemble_file( char* pszf )
{
    ASMDATA asmdata;

    fcpu_asm_parse_begin( &asmdata );
    
    if( !asmdata.err )
        fcpu_asm_parse_file( pszf, &asmdata );

    if( !asmdata.err )  
        fcpu_asm_assemble_instructions( &asmdata );

    fcpu_asm_parse_end( &asmdata );

    if( !asmdata.err )
        return 1;

    return 0;
}