--------------------------------------------------------------------------
-- ROP2.vhdl - ROP2 Execution Unit for the F-CPU
-- Copyright (C) 2000 Yann GUIDON (whygee@f...)
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--
--------------------------------------------------------------------------
-- This is the first version ever for this unit.
-- It should be easily synthetizable but there is no proof yet.
-- What matters most today is that it compiles and behaves correctly.
-- Warning : this code is and should remain purely combinatorial,
-- there is no latching here, it must be done at another level.
-- Furthermore, the function lookup table should be moved earlier
-- in the pipeline, in parallel with the Xbar cycle.
-- Finally, only byte combines are possible yet.
--------------------------------------------------------------------------

LIBRARY ieee;
    USE ieee.std_logic_1164.ALL;
    USE ieee.numeric_std.all;
LIBRARY work;
    USE work.FCPU_config.ALL;

Entity EU_ROP2 is
  port(
    Din_A, Din_B  : in F_VECTOR;     -- the 2 operands
    ROP_function  : in Std_ulogic_vector(2 downto 0);  -- 3 function bits
    Combine_AND, Combine_OR : in Std_ulogic;           -- additional funtion
    Combine_size : in Std_ulogic_vector(1 downto 0);   -- unused ATM. Byte chuncks only.
    Dout         : out F_VECTOR     -- the result
  );
end EU_ROP2;

Architecture arch1 of EU_ROP2 is
  signal partial_result, partial_OR, partial_AND : F_VECTOR;  -- the partial results.
  signal local_function : Std_ulogic_vector(3 downto 0);  -- the local, decoded, function bits.
begin

-- 1 : lookup table that decodes the function bits
--        (should be done in an earlier stage)
  with ROP_function select
    local_function <=
      "0001" when "000",  -- AND
      "0010" when "001",  -- ANDN
      "0110" when "010",  -- XOR
      "0111" when "011",  -- OR
      "1000" when "100",  -- NOR
      "1001" when "101",  -- XNOR
      "1011" when "110",  -- ORN
      "1110" when others; -- NAND

-- 2 : the ROP2 operator itself.
  partial_result <=
       ((not Din_A) and (not Din_B) and (F_RANGE => local_function(3)))
    or ((not Din_A) and (    Din_B) and (F_RANGE => local_function(2)))
    or ((    Din_A) and (not Din_B) and (F_RANGE => local_function(1)))
    or ((    Din_A) and (    Din_B) and (F_RANGE => local_function(0)));

-- 3 : partial ORs and ANDs on the byte chuncks :
  BYTE_COMBINE : for i in UMAX/8-1 downto 0 generate
    partial_OR(8*i+7 downto 8*i) <= "11111111" when
      partial_result(8*i+7 downto 8*i) /= "00000000"
      else "00000000";
    partial_AND(8*i+7 downto 8*i) <= "11111111" when
      partial_result(8*i+7 downto 8*i) = "11111111"
      else "00000000";
-- i'm still uncertain about the best way to write a multi-size version.
  end generate BYTE_COMBINE;

-- 4 : final selection stage :
  Dout <= partial_OR  when (Combine_OR  = '1')
     else partial_AND when (Combine_AND = '1')
     else partial_result;

end;
