/*
 *
 *    ASSEMBLE.C : miscelaneous C definitions and functions
 *
 */

/*
 *   Whygee   Sept 15 1999  
 * Revamped from the Gamma for the F-CPU project.
 * Not much F-CPU stuff here to change, except some
 * "constant values" used for the forward label resolution,
 * nothing severe. This file mainly deals with label, define
 * ifdefs. Fortunately, the Gamma is a kind of cousin of the
 * F-CPU so several specific structures remain unchanged ;-)
 *
 *  Please look at lines 51, 128, 304
 *  for more precise remarks.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*
 *   global variables:
 */

unsigned int
/* for the IFDEF stuffs */
 skip_count=0,    /* depth of the skipped #IFDEFs */
 ifdef_count=0,   /* flag for ifdef->endif (nesting count) */
#define MAX_IFDEF (sizeof(int)*8) /* it's implementation dependent ! */
 else_count=0,    /* flag for only one #ELSE  -> ACTUALLY this is a bit stack ! */

 warning_count=0,          /* is it THAT useful ? */
 count_nested_comments=0,  /* nested comments counter */
 enable_nested_comments=1000, /* can be modified within the parser */
 Line_Number=1,            /* the global one */
 parse_line;               /* line_number of the beginning of the instruction */


unsigned char c0,c1,c2,c3, /* hold the four bytes of an instruction */
  r1,r2;
FILE * list_output=stdout; /* will be redefined later (one day...) */


/*
 *  This part deals with individual data and/or instruction
 *  "spaces". i don't know how to integrate into the F-CPU project,
 *  who knows, it can be useful later, so i don't erase it.
 *  consider it as "dead code", but it can come back to life
 *  if someone creates a compiler/linker/loader that supports
 *  this kind of features.
 *  Now, only space #0 is used in practice, if you don't
 *  explicitely use "spaces" declarations in the source.
 *  it can be considered as "harmless".
 *
 *  What could be considered as more important is "MAX_INSTR"
 *  because it limits the number of instructions into
 *  a "space". increase it at will, or even make a dynamic
 *  memory allocation.
 */


/* definition of the memory spaces */

#define MAXSPACE 5
#define MAX_INSTR 200

unsigned int Instruction_Pointer=0, /* relocated pointer (IP for the user) */
 instr_tab_offset[MAXSPACE], /* the pointer of instr_tab */
 original_index[MAXSPACE],
 current_space=0;
unsigned char property[MAXSPACE];
unsigned char * instr_tab[MAXSPACE];

char *properties[]={"X","R","W","RW","RX","WX","RWX"};
#define EXECUTABLE          0
#define WRITEONLY           2
#define READONLY            1
#define READWRITE           3
#define EXECUTABLEWRITE     5
#define EXECUTABLEREAD      4
#define EXECUTABLEREADWRITE 6

unsigned int sizes[4]={8,16,32,64}; /* by default at reset */

/*
 *  The following structures share the same internal memory allocation
 */

/* FILE DESCRIPTOR used as a (linked) "include" stack */
struct include_node {
  void * next;          /* linked list, terminated by p->next==p */
  FILE *input_file;     /* backup of the global variable */
  int node_line_number; /* backup */
  char filename[1];     /* first byte of a normal string */
} *include_prec;        /* points to the last valid node, that is: the current file */

/* LABEL DESCRIPTOR */
struct label_struct
{
  void * next;         /* linked list terminated by p->next==NULL */
  struct include_node *include_file; /* points to the file desriptor (for the error messages) */
  int  line;           /* where the label is defined (sometimes, declared as global, then overwritten) */
  int value;           /* value of the label/ID */
  int space;           /* in which memory space */
  struct undefined_node *undefined_list; /* NULL or linked list to the relocation items */
  int  properties;     /* bits as defined below: */
#define LABEL_DECLARED  0x10 /* if the label is already declared in a label or with an equal statement */
  char string[1];      /* this byte only contains the size (like with PASCAL strings) */
} *label_prec=NULL,    /* tail of the linked list */
  *referenced_label,   /* for the interface with BISON */
  *referenced_label2;  /* for the interface with BISON (solves a bug when two labels are used in an expression */

/* #DEFINEs and other preprocessor commands */
struct define_node
{
  void * next;         /* linked list, finished when p->next==NULL */
  int line;            /* where it is DEFINed */   /* is it THAT useful ? */
  struct define_node * last_scan; /* is an include_node when NULL, a *define_node otherwise */
  char *buffer;        /* where the definition is stored; uses realloc */
  int buffer_size;     /* number of char. in the buffer */
  signed int char_counter; /* points to the next char to be read, -1 if not in use */
  char string[1];      /* this byte only contains the size (like with PASCAL strings): */
} *define_prec=NULL, /* tail of the linked list */
  *current_define=NULL;     /* the one that's currently read from */


/*
 *
 *  This part can be very CPU-dependent.
 *  you'll have to update the fields and the
 *  #defines. See at the bottom of the file  
 *  for the code that uses them.
 *
 */


/* linked list of UNDEFINED ADDRESSES */
struct undefined_node
{
  struct undefined_node *next; /* none if NULL */
  int reloc_type;      /* belongs to one of the following relocation type: */
#define RELOC_imm8  1
#define RELOC_imm16 2
#define RELOC_byte  3
#define RELOC_word  4
#define RELOC_dword 5
  int where;           /* index */
  int space;           /* in which memory space */
};

/*
 *   ERROR AND WARNING MESSAGES
 */

void error(char *msg, struct label_struct * ptr )
{
  printf("error (%s:%d): %s", include_prec->filename, Line_Number, msg);
  if (ptr != NULL)
    printf(" ('%s'@%s:%d)", ptr->string+1, (ptr->include_file)->filename, ptr->line);
  putc('\n',stdout);
  exit(7);
}

/* for BISON */
void yyerror(char *s)
{
  error(s,NULL);
}

void warning(char *msg, struct label_struct * ptr )
{
  printf("warning (%s:%d): %s", include_prec->filename, Line_Number, msg);
  if (ptr != NULL)
    printf(" ('%s'@%s:%d)", ptr->string+1, (ptr->include_file)->filename, ptr->line);
  putc('\n',stdout);
  warning_count++;
  return;
}

/*
 *
 *  Number passing between FLEX and BISON (instead of yylvalue)
 *
 */

int yy_number;

int hex2int(char c)
{
  if (c>='A')
  {
    if (c>='a')
      return c-'a'+10;
    else
      return c-'A'+10;
  }
  else
    return c-'0';
}

/*
 *   MEMORY MANAGEMENT   (without desallocation)
 */

/* size of one buffer: (let's begin with little) */
#define MEM_TRIGGER 8192 /* one day, i'll un-#define it
                             so it can change dynamically */

/* memory management global variables: */
int mem_used=MEM_TRIGGER; /* forces mem_increase() to allocate a new buffer at the first call */
char * mem_free;   /* pointer to the current buffer */

/* hand-made malloc */
void * mem_increase(int char_count)
{
  char * p;
  char_count=~7 & (char_count+8);  /* granularity of 64 bits systems */
  if ((char_count+mem_used) >= MEM_TRIGGER)
  {
     if ((mem_free=malloc(MEM_TRIGGER))==NULL)
       error("memory exhausted",NULL);
     mem_used=0;
  }
  p=&mem_free[mem_used];
  mem_used+=char_count;
  return p;
}

/*
 *
 *  LABEL MANAGEMENT
 *
 */

/* does a label exist ? */
struct label_struct *label_exists(char *label_name, int name_len )
{
  register struct label_struct *_p;

/* scans the linked list */
  _p=label_prec;   /* tail */
  while (_p != NULL) /* head ? */
  {
/* length comparison */
    if (_p->string[0] == name_len)
/* then string comparison */
      if (memcmp(_p->string+1,label_name,name_len)==0)
        return _p;
    _p = _p->next;
  }
  return NULL;     /* no match in the box */
}

/* This function is called everytime a label is declared or defined: */
struct label_struct * create_label(char *label_text, register int _c, int parameters)
{
  register struct label_struct *_p;

  if ((_p = label_exists(label_text,_c))!=NULL)
  {
/* some messy error checkings */

    if (parameters & LABEL_DECLARED)
    {

/* the label is already declared */
       if (_p->properties & LABEL_DECLARED)
         error("label already defined",_p);
        else
/* declares this label */
       {
          _p->properties |= parameters;
          _p->line = Line_Number;
          return _p;
       }
    }
    else
    {
/* updates a label */
      _p->properties |= parameters;
      return _p;
    }
  }

/* registers the properties of the label (address, pm/dm...) */
  _p=mem_increase(sizeof(*_p)+_c+1);
  memcpy(_p->string+1,label_text,_c);
  _p->string[0]=_c;         /* string size */
  _p->string[_c+1]=0;       /* trailing zero */
  _p->undefined_list=NULL;
  _p->value=-1;
  _p->space=-1;
  _p->line = Line_Number;
  _p->include_file = include_prec;
  _p->properties = parameters;
  _p->next=label_prec;
  label_prec = _p;
  return _p;
}


/*
 *
 *  Take care of this part, 
 *  don't play too much with it.
 *  This can be a place where bugs hide easily.
 *
 *
 */




/* THE KEY FEATURE FOR A ONE-PASS ASSEMBLER... */
void resolve (struct label_struct *q)
{
  int val=q->value;
  struct undefined_node *p=q->undefined_list;

  while (p) /* scan till the end of the linked list */
  {
    switch(p->reloc_type)
    {
      default: error("unknown resolution type",NULL);
      case RELOC_imm8:
        if ((val>127)|(val<-128))
          error ("imm8 overflow",NULL);
        instr_tab[p->space][p->where+1]|=(val<<4);
        instr_tab[p->space][p->where+2]|=(val>>4) & 15;
        fprintf(list_output,"Resolved the forward <imm8>=0X%02X at instruction 0X%04X(%d)\n",
          val,p->where,p->space);
        break;
      case RELOC_imm16:
        if ((val>32767)|(val<-32768))
          error ("imm16 overflow",NULL);
        instr_tab[p->space][p->where+1]|=(unsigned char)(val<<2);
        instr_tab[p->space][p->where+2]|=(unsigned char)(val>>6)& 255;
        instr_tab[p->space][p->where+3]|=(unsigned char)(val>>14)& 3;
        fprintf(list_output,"Resolved the forward <imm16>=0X%04X at instruction 0X%04X(%d)\n",
          val,p->where,p->space);
        break;
      case RELOC_byte:
        if ((val>127)|(val<-128))
          error ("byte overflow",NULL);
        instr_tab[p->space][p->where]=(unsigned char)val;
        fprintf(list_output,"Resolved the forward <byte>=0X%02X in data @ 0X%04X(%d)\n", val,p->where,p->space);
        break;
      case RELOC_word:
        if ((val>32767)|(val<-32768))
          error ("word overflow",NULL);
        instr_tab[p->space][p->where]=(unsigned char)val;
        instr_tab[p->space][p->where+1]=(unsigned char)(val >> 8);
        fprintf(list_output,"Resolved the forward <word>=0X%04X in data @ 0X%04X(%d)\n", val,p->where,p->space);
        break;
      case RELOC_dword:
        instr_tab[p->space][p->where]=(unsigned char)val;
        instr_tab[p->space][p->where+1]=(unsigned char)(val >> 8);
        instr_tab[p->space][p->where+2]=(unsigned char)(val >> 16);
        instr_tab[p->space][p->where+3]=(unsigned char)(val >> 24);
        fprintf(list_output,"Resolved the forward <dword>=0X%08X in data @ 0X%04X(%d)\n", val,p->where,p->space);
        break;
    }
    p=p->next;
  }
  q->undefined_list=NULL; /* done with it ! */
}


/*
 *  DEFINEs
 */

/* does a DEFINE exist ? */
struct define_node * define_exists(char * name, int length)
{
  register struct define_node * _p = define_prec;
  while (_p)
  {
/* length comparison */
    if (_p->string[0] == length)
/* then string comparison */
      if (memcmp(_p->string+1,name,length)==0)
        return _p;
    _p=_p->next;
  }
  return NULL;
}

/* Does a DEFINE */
struct define_node * create_define(char * name, int name_length, char * definition, int def_length)
{
  register struct define_node * _p;
/* registers the parameters */
  _p=mem_increase(sizeof(*_p)+name_length+1);
  _p->string[0]=name_length;
  _p->line=Line_Number;
  _p->last_scan=NULL;
  _p->buffer_size=def_length;
  _p->char_counter=-1;
  _p->buffer=definition; /* 3/4/99: i forgot this one ! (or was there a transmission error ?) :-/ */
  memcpy(_p->string+1,name,name_length+1);
  _p->next=define_prec;
  define_prec=_p;
  return _p;
}

/*
 *  FILES
 */

/* this re-defines the input method for FLEX */
#define YY_INPUT(buf,result,max_size)  buf[0]=get_new_char(&result);

int scan_from_define=0; /* from a file at the beginning */

/* no buffered reads because FLEX does some weird stuffs */
char get_new_char(int *size)
{
  int character;

  if (scan_from_define)

/* get from a DEFINE buffer */
  {
     character=current_define->buffer[current_define->char_counter++];
     if (current_define->char_counter > current_define->buffer_size)
     {
/* pops the define */
       current_define->char_counter = -1;
       if (current_define->last_scan)
         current_define=current_define->last_scan; /* pop */
       else
         scan_from_define=0; /* nothing to pop */
     }
  }
  else

/* get from a FILE */
  {

get_again:
    character=fgetc(include_prec->input_file);

/* End of file ? */
    if (character==EOF)
    {
      if (fclose(include_prec->input_file))
        error("can't close the file",NULL);
      if (include_prec->next == include_prec)
      {
        *size=0;
        return (0); /* we have reached the end of the main file */
      }
/* pops a file from the stack */
      include_prec=include_prec->next;
      Line_Number = include_prec->node_line_number ;
      goto get_again; /* so EOF is tested... */
    }
   /* counts the lines */
    if (character=='\n')
      Line_Number ++;
  }
  *size=1;
  return (character);
}


/* DISASSEMBLED OUTPUT: */
char instruction_description[200]; /* holds the disassembled string */
int desc_size=0;

void add(char * string)
{
  int i=strlen(string);
  if ((desc_size+i+1) >200)
     error("string too long",NULL);
  memcpy(instruction_description+desc_size,string,i+1);
  desc_size+=i;
}

void SPRINTF(char * str1, int a) /* yet another naughty hack */
{
  char string[30];
  int i;
  sprintf(string,str1,a); /* beware of this function */
  i=strlen(string);
  if (i>=30) error ("line too long",NULL);
  memcpy(instruction_description+desc_size,string,i+1);
  desc_size+=i;
}  
